package cn.qg.qaplatform.encrypt;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.BASE64Decoder;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.security.Key;

public class LexinEncrypt {
    private static final Logger LOGGER = LoggerFactory.getLogger(LexinEncrypt.class);
    /**
     * KEY算法
     */
    private static final String KEY_ALGORITHM = "AES";
    /**
     * 加密算法
     * "/算法/模式/补码方式"
     */
    public static final String CIPHER_ALGORITHM = "AES/ECB/PKCS5Padding";
    /**
     * KEY长度
     */
    private static final int KEY_SIZE = 128;

    /**
     * 编码
     */
    private static final String CHARSET = "UTF-8";

    public static void main(String[] args) {
        try {
            String key = getKeyStr();
        } catch (Exception e) {
            LOGGER.error("e={}", ExceptionUtils.getStackTrace(e));
        }
    }

    /**
     * 获取密钥 - Base64
     *
     * @return
     * @throws Exception
     */
    public static String getKeyStr() throws Exception {
        return Base64.encodeBase64String(getKey().getEncoded());
    }

    /**
     * 获取密钥
     *
     * @return
     * @throws Exception
     */
    public static Key getKey() throws Exception {
        // 实例化
        KeyGenerator kg = KeyGenerator.getInstance(KEY_ALGORITHM);
        // AES 要求密钥长度为128位、192位或256位
        kg.init(KEY_SIZE);
        // 生成密钥
        SecretKey secretKey = kg.generateKey();
        return secretKey;
    }

    /**
     * 数据加密
     *
     * @param data 待加密的数据
     * @param key  加密使用的KEY
     * @return 加密之后的数据
     */
    public static String encryptAndBase64Encode(String data, String key) {
        try {
            return encryptAndBase64Encode(data.getBytes(CHARSET), key);
        } catch (Exception e) {
            LOGGER.error("AES加密报错，error={}", e);
            return null;
        }
    }

    /**
     * 加密后base64编码，指定algorithm
     * @param data
     * @param key
     * @param algorithm
     * @return
     */
    public static String encryptAndBase64Encode(String data, String key, String algorithm, AESKeyType aesKeyType) {
        try {
            return encryptAndBase64Encode(data.getBytes(CHARSET), key, algorithm, aesKeyType);
        } catch (Exception e) {
            LOGGER.error("AES加密报错，error={}", e);
            return null;
        }
    }

    /**
     * 对字节数组加密
     * @param data
     * @param key
     * @return
     */
    public static String encryptAndBase64Encode(byte[] data, String key) {
        try {
            if (StringUtils.isBlank(key)) {
                LOGGER.error("AES Key为空");
                return null;
            }

            return doEncryptAndBase64Encode(data, key, CIPHER_ALGORITHM, AESKeyType.plain);
        } catch (Exception ex) {
            LOGGER.error("AES加密报错，error={}", ex);
            return null;
        }
    }

    /**
     * 对字节数组加密，指定algorithm
     * @param data
     * @param key
     * @return
     */
    public static String encryptAndBase64Encode(byte[] data, String key, String algorithm, AESKeyType aesKeyType) {
        try {
            if (StringUtils.isBlank(key)) {
                LOGGER.error("AES Key为空");
                return null;
            }
            if (StringUtils.isBlank(algorithm)) {
                LOGGER.error("AES 算法为空");
                return null;
            }
            return doEncryptAndBase64Encode(data, key, algorithm, aesKeyType);
        } catch (Exception ex) {
            LOGGER.error("AES加密报错，error={}", ex);
            return null;
        }
    }

    /**
     * 执行加密和base64编码
     * @param data
     * @param key
     * @param algorithm
     * @param aesKeyType
     * @return
     * @throws Exception
     */
    private static String doEncryptAndBase64Encode(byte[] data, String key, String algorithm, AESKeyType aesKeyType) throws Exception {
        algorithm = StringUtils.isBlank(algorithm)?CIPHER_ALGORITHM:algorithm;
        byte[] raw = toKey(key, aesKeyType);
        SecretKeySpec skeySpec = new SecretKeySpec(raw, KEY_ALGORITHM);
        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(Cipher.ENCRYPT_MODE, skeySpec);
        byte[] encrypted = cipher.doFinal(data);
        return new Base64().encodeToString(encrypted);
    }

    /**
     * 数据解密，返回字符串
     * @param data 待解密的数据
     * @param key  解密使用的KEY
     * @return 解密之后的数据
     */
    public static String decryptAfterBase64Decode(String data, String key) {
        try {
            byte[] bytes = decryptAfterBase64DecodeToByte(data, key);
            if (bytes != null && bytes.length > 0){
                return new String(bytes, CHARSET);
            }
        } catch (Exception e) {
            LOGGER.error("AES解密报错，error={}", e);
        }
        return null;
    }

    /**
     * base64解码后解密，指定algorithm
     * @param data
     * @param key
     * @param algorithm
     * @return
     */
    public static String decryptAfterBase64Decode(String data, String key, String algorithm, AESKeyType aesKeyType) {
        try {
            byte[] bytes = decryptAfterBase64DecodeToByte(data, key, algorithm, aesKeyType);
            if (bytes != null && bytes.length > 0){
                return new String(bytes, CHARSET);
            }
        } catch (Exception e) {
            LOGGER.error("AES解密报错，error={}", e);
        }
        return null;
    }

    /**
     * 数据解密，返回字节数据
     * @param data
     * @param key
     * @return
     */
    public static byte[] decryptAfterBase64DecodeToByte(String data, String key) {
        try {
            if (StringUtils.isBlank(key)) {
                LOGGER.error("AES Key为空");
                return null;
            }
            return doDecryptAfterBase64DecodeToBtype(data, key, CIPHER_ALGORITHM, AESKeyType.plain);
        } catch (Exception ex) {
            LOGGER.error("AES解密报错:" + ex.toString());
            return null;
        }
    }

    /**
     * base64解码后解密，返回字节数组，指定算法
     * @param data
     * @param key
     * @param algorithm
     * @return
     */
    public static byte[] decryptAfterBase64DecodeToByte(String data, String key, String algorithm, AESKeyType aesKeyType) {
        try {
            if (StringUtils.isBlank(key)) {
                LOGGER.error("AES Key为空");
                return null;
            }
            if (StringUtils.isBlank(algorithm)) {
                LOGGER.error("AES 算法为空");
                return null;
            }
            return doDecryptAfterBase64DecodeToBtype(data, key, algorithm, aesKeyType);
        } catch (Exception ex) {
            LOGGER.error("AES解密报错:" + ex.toString());
            return null;
        }
    }

    /**
     * 执行base64解码后解密
     * @param data
     * @param key
     * @param algorithm
     * @param aesKeyType
     * @return
     * @throws Exception
     */
    private static byte[] doDecryptAfterBase64DecodeToBtype(String data, String key, String algorithm, AESKeyType aesKeyType) throws Exception {
        algorithm = StringUtils.isBlank(algorithm)?CIPHER_ALGORITHM:algorithm;
        byte[] raw = toKey(key, aesKeyType);
        SecretKeySpec skeySpec = new SecretKeySpec(raw, KEY_ALGORITHM);
        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(Cipher.DECRYPT_MODE, skeySpec);
        byte[] encrypted1 = new Base64().decode(data);
        byte[] original = cipher.doFinal(encrypted1);
        return original;
    }

    /**
     * 转换密钥
     * @param key
     * @param aesKeyType
     * @return
     * @throws Exception
     */
    private static byte[] toKey(String key, AESKeyType aesKeyType) throws Exception {
        if (aesKeyType == AESKeyType.plain){
            return key.getBytes(CHARSET);
        }else if (aesKeyType == AESKeyType.base64){
            return new BASE64Decoder().decodeBuffer(key);
        }
        return null;
    }

    public enum AESKeyType {

        plain("普通字符串"),
        base64("base64编码字符串");

        public String text;

        AESKeyType(String text){
            this.text = text;
        }
    }
}
