const path = require('path')
const _ = require('lodash')
const moment = require('moment')

const appConfig = require(global.configPath)

const Client = require('kubernetes-client').Client
const Config = require('kubernetes-client').config

const etcdService = require('./etcdService')
const dict = require('../service/dictService')

const client = new Client({
  config: Config.fromKubeconfig(
    path.resolve(process.cwd(), appConfig.k8s.configPath),
  ),
  version: appConfig.k8s.version,
})

// const client = new Client({
//   config: Config.fromKubeconfig(
//     path.resolve(process.cwd(), appConfig.k8sTKE.configPath)
//   ),
//   version: appConfig.k8sTKE.version
// })

const formatServiceInfo = (obj) => {
  const portObj = {}
  obj.spec.ports.forEach((i) => {
    portObj[`${i.name}_port`] = i.nodePort
  })

  return _.assign(portObj, {
    cluster_ip: obj.spec.clusterIP,
    ports: obj.spec.ports,
  })
}

const formatPodInfo = (podInfo) => {
  const podStatus = podInfo.status.phase === 'Running' && podInfo.status.conditions.every(i => i.status === 'True')
    ? 'Running' : 'Pending'

  const containerImage = _.get(podInfo.status.containerStatuses, '[0].image', '')

  const ret = {
    name: (podInfo.metadata.labels && podInfo.metadata.labels.app) || podInfo.metadata.name,
    type: podInfo.metadata.labels.tier,
    pod_name: podInfo.metadata.name,
    pod_status: podStatus,
    pod_ip: podInfo.status.podIP,
    host_ip: podInfo.status.hostIP,
    start_time: podInfo.status.startTime,
    live: moment(new Date(podInfo.status.startTime)).startOf('minute').fromNow(),
    image: containerImage,
  }

  if (containerImage !== '') {
    ret.branch = containerImage.split(':')[1]
    ret.branch_no_time = containerImage.split(':')[1].split('-')[0]
    ret.branch_time = containerImage.split(':')[1].split('-')[1]
  }

  return ret
}


exports.listEnv = async function () {
  let res = await client.api.v1.namespaces.get()
  res = _.get(res, 'body.items', [])
  res = _.filter(res, item => !appConfig.k8s.systemNamespace.includes(item.metadata && item.metadata.name))

  const handler = async function (item) {
    const progress = await etcdService.getProgressOfNamespace(item.metadata.name)
    return {
      name: item.metadata.name,
      creation_timestamp: item.metadata.creationTimestamp,
      status: item.status.phase,
      init_status: progress,
    }
  }

  const task = []
  res.forEach((item) => {
    task.push(handler(item))
  })
  return Promise.all(task)
}

exports.listHang = async function () {
  const list = await etcdService.getHangUpEnv()
  const ret = []
  list.node.nodes.forEach((item) => {
    item.nodes.forEach((item) => {
      if (item.key && item.key.indexOf('info') !== -1) ret.push(JSON.parse(item.value))
    })
  })
  return ret.sort()
}

exports.listPreset = async function () {
  const list = await etcdService.listPreset()
  return list.sort((x, y) => x.preset_name > y.preset_name)
}

exports.listEnvVars = async function (namespace) {
  const services = {}

  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get({ qs: { labelSelector: 'tier!=job' } }),
    client.api.v1.namespaces(namespace).services.get(),
  ])

  res[0].body.items.forEach(async (item) => {
    // let serviceName = item.metadata.labels && item.metadata.labels.app || item.metadata.name
    const nameArray = item.metadata.name.split('-')
    const serviceName = nameArray.length > 2 ? nameArray.slice(0, -2).join('-') : nameArray.slice(0, -1).join('-')
    services[serviceName] = formatPodInfo(item)
  })

  res[1].body.items.forEach(async (item) => {
    if (services[item.metadata.name]) {
      services[item.metadata.name] = _.assign(services[item.metadata.name], formatServiceInfo(item))
    }
  })

  const ret = {}

  for (const index in services) {
    if (Object.prototype.hasOwnProperty.call(services, index)) {
      const upperCaseName = index.toUpperCase().replace(/-/g, '_')
      const hostKey = `${upperCaseName}_SERVICE_HOST`
      const portKey = `${upperCaseName}_SERVICE_PORT`
      ret[hostKey] = services[index].host_ip

      if (services[index].ports) {
        ret[portKey] = services[index].ports[0].nodePort

        services[index].ports.forEach((item) => {
          ret[`${portKey}_${item.port}`] = item.nodePort || item.port
        })
      }
    }
  }

  return ret
}

const formatIngressInfo = obj => ({ host: obj.spec && obj.spec.rules && obj.spec.rules[0].host })

exports.getMicroServiceOfNamespace = async function (namespace) {
  const ret = []
  const service = {}

  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get({ qs: { labelSelector: 'tier!=job' } }),
    client.api.v1.namespaces(namespace).services.get(),
    client.apis.extensions.v1beta1.namespaces(namespace).ingresses.get(),
  ])

  res[0].body.items.forEach(async (item) => {
    const serviceName = (item.metadata.labels && item.metadata.labels.app) || item.metadata.name
    service[serviceName] = formatPodInfo(item)
  })

  res[1].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatServiceInfo(item))
    }
  })

  res[2].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatIngressInfo(item))
    }
  })

  for (const index in service) {
    if (Object.prototype.hasOwnProperty.call(service, index)) {
      service[index].status = await etcdService.getStatus(namespace, index)
      ret.push(service[index])
    }
  }

  return ret
}

exports.getMicroServiceOfPod = async function (namespace, name) {
  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get({ qs: { labelSelector: `app=${name},tier!=job` } }),
    client.api.v1.namespaces(namespace).services(name).get(),
    etcdService.getStatus(namespace, name),
  ])

  res[0] = formatPodInfo(res[0].body.items[0])
  res[1] = formatServiceInfo(res[1].body)

  if (!dict.commonService.includes(name)) {
    res[3] = await client.apis.extensions.v1beta1.namespaces(namespace).ingresses(name).get()
    res[3] = formatIngressInfo(res[3].body)
  }

  return _.assign({}, res[0], res[3], { status: res[2] }, { debug_adr: `${res[0].host_ip}:${res[1].debug_port}` }, res[1])
}
