const Router = require('koa-router')
const bodyParser = require('koa-bodyparser')
const logger = require('koa-log4').getLogger('sonar')
const awaitRequest = require('../utils/awaitRequest')
const sleep = require('../utils/sleep')
const sendMail = require('../utils/sendMail')
const { SonarJob } = require('../service/mongoService')

const config = require(global.configPath)

const title = 'sonar 扫描结果'

const sendFailMail = async (data, item, executionTimeMs) => {
  const address = data.gitUser ? `${data.gitUser}@quantgroup.cn` : data.gitUserMail
  const duration = data.duration / 1000
  const executionTime = executionTimeMs / 1000
  const text = `<font color=#FF7F50> 扫描结果 : ${item.status.qualityGateStatus}<br/>`
    + `<font color=#000000> 项目名称 : ${data.projectName}<br/>`
    + `分支名称 : ${data.branchName}<br/>`
    + `扫描时间 : ${duration}s<br/>`
    + `分析时间 : ${executionTime}s<br/>`
    + `扫描日期 : ${item.analysisDate}<br/>`
    + `查看详情 : ${config.qahomeHost}/pipeline/sonarJob?gitUser=${data.gitUser}`
    // + `查看详情 : ${config.sonarHost}/project/activity?id=${data.projectName}&&selected_date=${item.analysisDate}`
  sendMail(address, `${data.projectName} ${title}`, text)
}

const sendExceptionMail = async (data) => {
  const address = data.gitUser ? `${data.gitUser}@quantgroup.cn` : data.gitUserMail
  const text = '<font color=#FF0000> 扫描结果 : 扫描任务执行失败 <br/>'
    + `<font color=#000000> 结果信息 : ${data.error} <br/>`
    + `项目名称 : ${data.projectName}<br/>`
    + `分支名称 : ${data.branchName}<br/>`
    + `查看详情 : ${config.qahomeHost}/pipeline/sonarJob?gitUser=${data.gitUser}`
    // + `Sonar详情 : ${config.sonarHost}/dashboard?id=${data.projectName} <br/>`
    // + `Jenskins详情 : ${data.absoluteUrl}console <br/>`
  sendMail(address, `${data.projectName} ${title}`, text)
}

const sendExistMail = async (data, gitUser) => {
  const address = data.gitUser ? `${data.gitUser}@quantgroup.cn` : `${gitUser}@quantgroup.cn`
  const text = '<font color=#FF7F50> 本次执行的GITHASH曾经扫描过<br/>'
    + `扫描结果 : ${data.sonarResult}<br/>`
    + `<font color=#000000> 项目名称 : ${data.projectName}<br/>`
    + `分支名称 : ${data.branchName}<br/>`
    + `GITHASH : ${data.branchHash}<br/>`
    + `扫描分析日期 : ${data.analysisDate}<br/>`
    + `查看详情 : ${config.qahomeHost}/pipeline/sonarJob?gitUser=${data.gitUser}`
  sendMail(address, `${data.projectName} ${title}`, text)
}

const saveData = async (data) => {
  const sonarJob = await SonarJob.getOneSonarJob({ branchHash: data.branchHash })
  if (sonarJob) {
    await SonarJob.updateSonarJob(data)
  } else {
    await SonarJob.saveSonarJob(data)
  }
}

const callback = async (ctx) => {
  const data = ctx.request.body
  logger.info('sonar callback info', data)

  if (data.sonarJob) {
    logger.info(data.sonarJob.projectName, data.sonarJob.branchName, data.sonarJob.branchHash,
      'GITHASH 曾经 扫描过，扫描结果:', data.sonarJob)
    if (data.sonarJob.sonarResult !== 'OK' && data.mail !== 'noMail') {
      sendExistMail(data.sonarJob, data.gitUser)
    }
    ctx.body = ctx.ok()
    return
  }

  if (data.error) {
    data.buildResult = 'error'
    if (data.mail !== 'noMail') {
      sendExceptionMail(data)
    }
    await saveData(data)
    ctx.body = ctx.ok()
    return
  }

  let flag = true
  let executionTimeMs
  while (flag) {
    const ceTaskResult = await awaitRequest({
      url: `${data.content.ceTaskUrl}`,
      method: 'GET',
    })
    logger.info(data.projectName, data.branchName, data.branchHash, '的分析任务执行信息:', ceTaskResult.task.status)

    if (ceTaskResult.task.status === 'SUCCESS') {
      flag = false
      executionTimeMs = ceTaskResult.task.executionTimeMs
      const res = await awaitRequest({
        url: `${config.sonarHost}/api/project_branches/list?project=${data.projectName}`,
        method: 'GET',
      })
      logger.info('analysisResult', data.projectName, JSON.stringify(res))
      for (const i of res.branches) {
        if (i.name === 'master' && i.status.qualityGateStatus !== 'OK' && data.mail !== 'noMail') {
          logger.info(data.projectName, '触发质量阀，发送邮件通知')
          sendFailMail(data, i, executionTimeMs)
        }
        data.sonarResult = i.status.qualityGateStatus
        data.analysisDate = i.analysisDate
      }
    }
    if (ceTaskResult.task.status === 'FAILED') {
      flag = false
      data.sonarResult = ceTaskResult.task.status
      Object.assign(data, { error: ceTaskResult.task.errorMessage })
      sendExceptionMail(data)
    }
    await sleep(2 * 1000)
  }

  data.buildResult = 'success'
  await saveData(data)
  ctx.body = ctx.ok(data)
}

const getOneSonarJob = async (ctx) => {
  const sonarJob = await SonarJob.getOneSonarJob(ctx.request.query)
  ctx.body = ctx.ok(sonarJob)
}

const queryHandle = function (query) {
  if (query.projectName && query.projectName.trim()) {
    query.projectName = new RegExp(query.projectName)
  } else {
    delete query.projectName
  }

  if (query.branchName && query.branchName.trim()) {
    query.branchName = new RegExp(query.branchName)
  } else {
    delete query.branchName
  }

  if (query.gitUser && query.gitUser.trim()) {
    query.gitUser = new RegExp(query.gitUser)
  } else {
    delete query.gitUser
  }

  if (query.startDate) {
    Object.assign(query, { createdAt: { $gte: query.startDate, $lte: query.endDate } })
    delete query.startDate
    delete query.endDate
  }
}

const getSonarJob = async (ctx) => {
  const query = ctx.request.body.query
  const page = ctx.request.body.page
  queryHandle(query)
  const data = await SonarJob.getSonarJob(query, { _id: 0 }, { createdAt: -1 }, page)
  const count = await SonarJob.getSonarJobCount(query)
  ctx.body = ctx.ok({ data, count })
}

const fetchQuality = async (ctx) => {
  const query = ctx.request.body
  queryHandle(query)

  // const data = await SonarJob.getSonarJobGroup(query);

  let data = await SonarJob.getSonarJob(query, {
    _id: 0, projectName: 1, buildResult: 1, sonarResult: 1,
  }, {}, {});

  data = data.reduce((prev, next) => {
    if (!prev[next.projectName]) {
      prev[next.projectName] = {
        exception: 0,
        error: 0,
        warn: 0,
        success: 0,
        sum: 0,
      }
    }
    if (next.buildResult === 'error') {
      prev[next.projectName].exception += 1
      prev[next.projectName].sum += 1
    }
    if (next.buildResult === 'success') {
      switch (next.sonarResult) {
        case 'OK':
          prev[next.projectName].success += 1
          break
        case 'FAILED':
          prev[next.projectName].exception += 1
          break
        case 'WARN':
          prev[next.projectName].warn += 1
          break
        case 'ERROR':
          prev[next.projectName].error += 1
          break
        default:
      }
      prev[next.projectName].sum += 1
    }
    return prev
  }, {})

  const res = []
  for (const i in data) {
    if (Object.prototype.hasOwnProperty.call(data, i)) {
      res.push({
        projectName: i,
        success: data[i].success,
        warn: data[i].warn,
        error: data[i].error,
        exception: data[i].exception,
        run: data[i].sum,
      })
    }
  }

  ctx.body = ctx.ok(res)
}

const router = new Router()
router
  .use(bodyParser())
  .post('/callback', callback)
  .get('/getSonarJob', getOneSonarJob)
  .post('/getSonarJobs', getSonarJob)
  .post('/fetchQuality', fetchQuality)

module.exports = router;
