const Router = require('koa-router')
const moment = require('moment')
const bodyParser = require('koa-bodyparser')
const logger = require('koa-log4').getLogger('pipeline')
const { PipeLine, PipeLinePush, SonarJob } = require('../service/mongoService')
const jenkinsService = require('../service/jenkinsService')
const dingTalk = require('../utils/dingTalk')
const sleep = require('../utils/sleep')
const { ProConfig } = require('../service/mongoService')
const awaitRequest = require('../utils/awaitRequest')

const config = require(global.configPath)

const pipelineJenkinsBuild = async function (data) {
  const {
    projectName, branchName, type, gitUser, commitMes,
  } = data
  const jobName = `tke-${type}`
  const buildData = {
    GIT_REPO: projectName,
    BRANCH_NAME: branchName,
    DEPLOY: 'false',
    NAMESPACE: 'qahome',
  }
  const res = await jenkinsService.build_with_params(jobName, buildData)
  logger.info('执行 pipeline jenkins job', jobName, commitMes, buildData)
  let flag = true
  while (flag) {
    const apiJson = await jenkinsService.queue_item(res.queueId);
    if (apiJson.executable) {
      flag = false
      const store = {
        buildId: apiJson.executable.number,
        buildJob: apiJson.task.name,
        projectName,
        branchName,
        commitMes,
        gitUser,
        gitEventTime: moment().format('YYYY-MM-DD HH:mm:ss'),
      }
      await PipeLinePush.savePipesPush(store)
    }
    await sleep(1 * 1000)
  }
}

const sonarJenkinsBuild = async (data) => {
  const {
    projectName, branchName, branchHash, gitUser, commitMes,
  } = data
  const jobName = 'tke-sonar'
  const buildData = {
    GIT_REPO: projectName,
    BRANCH_NAME: branchName,
    FORCE_SCAN: 'false',
    GIT_TRIGGER_USER: gitUser,
    COMMIT_MESSAGE: commitMes,
  }
  jenkinsService.build_with_params(jobName, buildData)
  logger.info('执行 sonarCheck jenkins job', jobName, buildData)

  const sonarJob = await SonarJob.getOneSonarJob({ branchHash })
  if (!sonarJob) {
    const saveData = {
      projectName,
      branchName,
      branchHash,
      gitUser,
      commitMes,
    }
    SonarJob.saveSonarJob(saveData)
  }
}

const webhooks = async function (ctx) {
  const req = ctx.request.body
  logger.info('webhooks', req)

  const projectName = req.repository.name
  const branchName = req.ref.replace('refs/heads/', '')
  logger.info(`收到webhooks->>>项目名:${projectName} 分支名:${branchName}`)

  // user_add_to_team project_create之类的操作
  if (req.object_kind !== 'push') {
    logger.info('object_kind不是push，不执行jenkins')
    ctx.body = ctx.ok('object_kind不是push')
    return
  }

  // 增删分支之类的push
  if (req.commits.length === 0) {
    logger.info(`${req.repository.name} ${req.ref} 没有commits，不执行jenkins`)
    ctx.body = ctx.ok(`${req.repository.name}  ${req.ref} 没有commits`)
    return
  }

  const gitUser = req.user_username || req.user_email.split('@')[0]
  const commitMes = req.commits[req.commits.length - 1].message

  let projects = await ProConfig.findByQuery({ is_active: true, sonar: true, type: 'java' }, { project_name: 1 })
  projects = projects.map(i => i.project_name)
  if (projects.includes(projectName)) {
    logger.info('webhooks 触发 sonar jenkins build', projectName, branchName, req.checkout_sha)
    sonarJenkinsBuild({
      projectName, branchName, branchHash: req.checkout_sha, gitUser, commitMes,
    })
  } else {
    logger.info(`${projectName} 不是激活且部署到容器的java服务，不执行sonar jenkins`)
    ctx.body = ctx.ok(`${projectName} 不是激活且部署到容器的java服务`)
  }

  const data = await PipeLine.getMatchPipe(projectName)
  // 获取项目的分类
  let gitObject;
  for (let i = 0; i < data.length; i += 1) {
    const res = data[i].repos
    for (let j = 0; j < res.length; j += 1) {
      if (res[j].repository === projectName && res[j].ref.name === branchName) {
        gitObject = res[j]
      }
    }
  }
  if (gitObject) {
    logger.info('webhooks 触发 pipeline jenkins build', projectName, branchName, req.checkout_sha)
    pipelineJenkinsBuild({
      projectName, branchName, type: gitObject.type, gitUser, commitMes,
    })
  } else {
    logger.info(`未查询到订阅信息->>>项目名:${projectName} 分支名:${branchName}，不执行pipeline jenkins`)
  }

  ctx.body = ctx.ok('success')
}

const getBuildInfoByHooks = async (ctx) => {
  const pipePush = await PipeLinePush.getOnePipePush(ctx.request.query)
  ctx.body = ctx.ok(pipePush)
}

const gitBranch = async (svcName) => {
  const proConfig = await ProConfig.getOneProject({ project_name: svcName }, { git_lab: 1 })
  const version = proConfig.git_lab === 'gitabc.xyqb.com' ? 'v3' : 'v4'
  const token = config.gitlab.token
  const data = await awaitRequest({
    url: `http://${proConfig.git_lab}/api/${version}/projects?per_page=200&search=${svcName}&private_token=${token}&simple=true`,
    method: 'GET',
  })
  if (data.length === 0) {
    return data
  }

  let projectId
  data.forEach((item) => {
    // 再次筛选下数据 因为gitlab返回的数据是模糊查询
    if (item.name === svcName || item.path === svcName) {
      projectId = item.id
    }
  })
  const branchInfo = await awaitRequest({
    url: `http://${proConfig.git_lab}/api/${version}/projects/${projectId}/repository/branches?private_token=${token}&per_page=100`,
    method: 'GET',
  })
  return branchInfo
}

const getGitBranch = async function (ctx) {
  const { name } = ctx.request.query
  const branchInfo = await gitBranch(name)
  if (branchInfo.length === 0) {
    ctx.body = ctx.fail('无拉去此项目的权限')
  } else {
    ctx.body = ctx.ok(branchInfo)
  }
}

const findPipes = async function (ctx) {
  const data = await PipeLine.getPipes(ctx.request.body)
  ctx.body = ctx.ok(data)
}

const getImageData = async (svcName, cluster) => {
  const params = { reponame: `qa-test/${svcName}` }
  const imageData = await awaitRequest({
    url: `${config.api.tke_api}/tag`,
    method: 'POST',
    headers: {
      'Content-Type': 'application/json;charset=utf-8',
      cluster: cluster || 'qa',
    },
    body: JSON.stringify(params),
  })
  return imageData && imageData.data
}

const save = async function (ctx) {
  const data = ctx.request.body
  await PipeLine.savePipes(data)
  logger.info('保存订阅 数据', data)
  for (const item of data.repos) {
    let branchInfo = await gitBranch(item.repository)
    branchInfo = branchInfo.filter(i => i.name === item.ref.name)
    if (branchInfo.length > 0) {
      let imageData = await getImageData(item.repository, data.cluster)
      imageData = imageData && imageData.tagInfo.filter(i => i.tagName.indexOf(item.ref.name.split('-')[0]) !== -1)
      // 如果还没有当前分支的镜像，或者提交时间比镜像制作时间晚时，触发jenkins
      if (!imageData || imageData.length === 0 || branchInfo[0].commit.committed_date.replace('T', ' ') > imageData[0].updateTime) {
        const buildData = {
          projectName: item.repository,
          branchName: item.ref.name,
          type: item.type,
          gitUser: data.update_user || data.new_user,
          commitMes: 'save pipeline application',
        }
        pipelineJenkinsBuild(buildData)
        logger.info('保存订阅 触发 pipeline jenkins build', data.application_name, buildData)
        await sleep(1 * 1000)
      } else {
        logger.info('保存订阅时 已经有更新的镜像 不触发 pipeline jenkins build', data.application_name, item.repository, item.ref.name)
      }
    }
  }
  ctx.body = ctx.ok('ok')
}

const deletePipes = async function (ctx) {
  const id = ctx.request.body.id
  const res = await PipeLine.delete(id)
  ctx.body = ctx.ok(res)
}

const title = '流水线信息如下'

const sendSuccessDing = async (data, pipeLineData, pushData) => {
  const text = `> 执行结果 : ${data.flag}\n\n`
    + `> 项目名称 : ${pushData.projectName}\n\n`
    + `> 分支名称 : ${pushData.branchName}\n\n`
    + `> 镜像名称 : ${data.image}\n\n`
    + `> 部署空间 : ${pipeLineData.namespace}\n\n`
    + `> 触发同学 : ${pushData.gitUser}\n\n`
    + `> 触发信息 : ${pushData.commitMes}\n\n`
    + `> 触发时间 : ${pushData.gitEventTime}\n\n`
    + `[查看详情](${data.absoluteUrl}console)`
  const address = pipeLineData.noticeAddress || data.dingRobotAddr
  await dingTalk(`${pushData.projectName}${title}`, text, address)
}

const sendSonarFailDing = async (data, pipeLineData) => {
  const sonarJob = data.sonarJob
  const text = `> <font color=#FF7F50> 执行结果 : ${data.msg}\n\n`
      + `> 扫描结果 : ${sonarJob.sonarResult}\n\n`
      + `> <font color=#000000> 项目名称 : ${sonarJob.projectName}\n\n`
      + `> 分支名称 : ${sonarJob.branchName}\n\n`
      + `> GITHASH:${sonarJob.branchHash}\n\n`
      + `> 扫描分析日期 : ${sonarJob.analysisDate}\n\n`
      + `[Jenkins详情](${data.absoluteUrl}console)\n\n`
      + `[Sonar详情](${config.sonarHost}/project/activity?id=${sonarJob.projectName}&&selected_date=${sonarJob.analysisDate})`
  const address = pipeLineData.noticeAddress || data.dingRobotAddr
  await dingTalk(`${sonarJob.projectName}${title}`, text, address)
}

const sendExceptionDing = async (data, pipeLineData) => {
  const text = `> <font color=#FF7F50> ${data.msg}\n\n`
    + `> <font color=#000000> 项目名称 : ${data.projectName}\n\n`
    + `> 分支名称 : ${data.branchName}\n\n`
    + `> 触发时间 : ${moment().format('YYYY-MM-DD HH:mm:ss')}\n\n`
    + `[Jenkins详情](${data.absoluteUrl}console)\n\n`
  const address = pipeLineData.noticeAddress || data.dingRobotAddr
  await dingTalk(`${data.projectName}${title}`, text, address)
}

const callback = async (ctx) => {
  const data = ctx.request.body
  logger.info('pipeline callback info', data)
  const {
    buildId, buildJob, image, domain, tier,
  } = data
  const pipePush = await PipeLinePush.getOnePipePush({ buildId, buildJob })
  if (!pipePush) {
    logger.info(`${buildJob} ${buildId} 不是pipeline触发，不发送dingtalk`)
    ctx.body = ctx.ok(`${buildJob} ${buildId} 不是pipeline触发，不发送dingtalk`)
    return
  }
  const pipeLineData = await PipeLine.getMatchPipe(pipePush.projectName)
  for (let i = 0; i < pipeLineData.length; i += 1) {
    const res = pipeLineData[i].repos
    for (let j = 0; j < res.length; j += 1) {
      if (res[j].ref.name === pipePush.branchName && res[j].repository === pipePush.projectName) {
        const dingTalkData = pipeLineData[i];
        if (data.flag === 'notOK') {
          sendSonarFailDing(data, dingTalkData)
        }
        if (data.flag === 'timeout' || data.flag === 'runError') {
          sendExceptionDing(data, dingTalkData)
        }
        if (data.flag === 'success') {
          // 部署到namespace
          if (dingTalkData.isDeploy) {
            const params = {
              namespace: dingTalkData.namespace,
              serviceName: res[j].repository,
              image,
              label: tier,
              type: tier,
              domain,
              debug: 0,
            }
            awaitRequest({
              url: `${config.api.tke_api}/service/modifyImage`,
              method: 'post',
              headers: {
                'Content-Type': 'application/json;charset=utf-8',
                cluster: dingTalkData.cluster || 'qa',
              },
              body: JSON.stringify(params),
            })
            logger.info(`部署服务 ${JSON.stringify(params)}`)
          }
          sendSuccessDing(data, dingTalkData, pipePush)
        }
      }
    }
  }
  ctx.body = ctx.ok()
}

const router = new Router()
router
  .use(bodyParser())
  .post('/save', save)
  .get('/getGitBranch', getGitBranch)
  .post('/find', findPipes)
  .post('/webhooks', webhooks)
  .post('/delete', deletePipes)
  .post('/callback', callback)
  .get('/getBuildInfoByHooks', getBuildInfoByHooks)

module.exports = router
