const Router = require('koa-router')
const _ = require('lodash')
const df = require('date-fns')
const bodyParser = require('koa-bodyparser')
const jsondiffpatch = require('jsondiffpatch');

const config = require(global.configPath)
const { api } = require('../../src')

const dict = require('../service/dictService')
const eosService = require('../service/eosService')
const redisService = require('../service/redisService')
const k8sService = require('../service/k8sService')
const mqService = require('../service/mqService')
const harborService = require('../service/harborServcice')
const ProConfig = require('../service/mongoService').ProConfig
const NamespaceGroups = require('../service/mongoService').NamespaceGroups

let newOsKey = ''
function generateUUID() {
  let d = new Date().getTime()
  const uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
    const r = (d + Math.random() * 16) % 16 | 0
    d = Math.floor(d / 16)
    return (c === 'x' ? r : ((r & 0x3) | 0x8)).toString(16)
  })
  return uuid
}

const getCode = async function (ctx) {
  newOsKey = generateUUID();
  ctx.body = ctx.ok(newOsKey)
};

const getGroups = function (ctx) {
  ctx.body = ctx.ok(dict.groups)
};

// 运行中环境
const getActiveNameSpace = async function (ctx) {
  const env = await k8sService.listEnv();
  let groups = await NamespaceGroups.getGroups() || [];
  groups = _.groupBy(groups, 'name_space');
  env.forEach((item) => {
    item.status_name = dict.status[item.status] || '异常';
    item.disabled = item.status !== 'Active';
    item.creation_timestamp = item.creation_timestamp && df.format(item.creation_timestamp, 'YYYY-MM-DD HH:mm:ss');
    item.job_status_name = item.init_status === 'init' ? '处理中' : '';
    item.job_is_working = item.init_status === 'init';
    if (groups[item.name]) {
      item.group_name = (groups[item.name][0] && groups[item.name][0].group) || ''
    }
  });

  ctx.body = ctx.ok(env)
};

// 挂起环境
// const getHangNamespace = async function (ctx) {
//   const res = await k8sService.listHang();
//   ctx.body = ctx.ok(res)
// };

// const recoveryHang = async (ctx) => {
//   const res = await eosService.recoveryHang(ctx.request.body);
//   ctx.body = ctx.ok(res)
// };

// const deleteHang = async (ctx) => {
//   const res = await eosService.destroyHangUpEnv(ctx.request.body);
//   ctx.body = ctx.ok(res)
// };

const getPodInfo = async (ctx) => {
  const res = await k8sService.getMicroServiceOfPod(
    ctx.request.body.namespace, ctx.request.body.system_name,
  );
  ctx.body = ctx.ok(res)
};

// 获取ns下的服务
const getServiceOfNs = async function (ctx) {
  const list = await k8sService.getMicroServiceOfNamespace(ctx.query.namespace);
  const res = list.reduce((prev, next) => {
    if (!prev[next.type]) prev[next.type] = [];
    prev[next.type].push(next);
    return prev
  }, {});

  ctx.body = ctx.ok(res)
};

const getNsTemplate = async function (ctx) {
  let res = await k8sService.listPreset()
  res = res.map((item) => {
    const update = {
      label: item.preset_name,
      enable_edit: item.enable_edit,
      preset_key: item.preset_key,
      template: {},
    };
    for (const prop in item.details) {
      if (Object.prototype.hasOwnProperty.call(item.details, prop)) {
        const value = item.details[prop].sort().reduce((prev, next) => {
          prev[next] = true;
          return prev
        }, {});
        update.template[prop] = value
      }
    }
    return update
  });

  res.unshift({
    label: '空白',
    enable_edit: false,
    preset_key: '',
    template: {
      common: {},
      frontend: {},
      backend: {},
    },
  });

  ctx.body = ctx.ok(res)
};

// 按common、frontend、backend获取项目
const getSystemsOfType = async function (ctx) {
  const res = await ProConfig.getActiveProjects();

  const obj = {
    backend: [],
    common: [],
    frontend: [],
  };

  res.forEach((item) => {
    if (item.type === 'ui') {
      if (item.project_name === 'paycenter-ui') {
        obj.frontend.push({ label: 'paycenter-ui--new-paycenter-ui', select: false })
      } else {
        obj.frontend.push({ label: item.project_name, select: false })
      }
    } else {
      obj.backend.push({ label: item.project_name, select: false })
    }
  });

  dict.commonService.sort().forEach((item) => {
    obj.common.push({ label: item, select: false })
  });

  ctx.body = ctx.ok(obj)
};

const savePreset = async function (ctx) {
  const { body } = ctx.request;
  for (const prop of Object.keys(body.details)) {
    body.details[prop] = body.details[prop].filter(item => item.select).map(item => item.label)
  }

  await eosService.savePreset(body);
  ctx.body = ctx.ok(body)
};

const deletePreset = async function (ctx) {
  const params = {
    preset_key: ctx.request.body.preset_key,
    name: ctx.request.body.preset_name,
  };
  await eosService.deletePreset(params);
  ctx.body = ctx.ok()
};

const createNamespace = async function (ctx) {
  ctx.qalog.setActive('create_namespace');
  const { body } = ctx.request;
  if (newOsKey !== body.code) {
    ctx.body = ctx.fail('创建确认码不正确');
    return
  }

  let env = await k8sService.listEnv();
  env = _.get(env, 'details.instances', []);
  const exist = _.find(env, o => o.name === ctx.request.body.namespace);
  if (exist) {
    ctx.body = ctx.fail('namespace已存在');
    return
  }

  const ns = {};
  for (const prop in body.systems) {
    if (Object.prototype.hasOwnProperty.call(body.systems, prop)) {
      ns[prop] = body.systems[prop].filter(item => item.select).map(item => item.label)
    }
  }

  await eosService.initEnv({
    namespace: body.namespace,
    microservices: ns,
  });

  ctx.body = ctx.ok()
};

const hang = async (ctx) => {
  const res = await eosService.hang(ctx.request.body);
  ctx.body = ctx.ok(res)
};
//
// // todo
// const refreshNameSpaceProgress = async (ctx) => {
//
// }

const deleteNamespace = async function (ctx) {
  ctx.qalog.setActive('delete_namespace');
  const { body } = ctx.request;
  if (!ctx.validate(body, { namespace: 'required' })) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT);
    return
  }
  await eosService.destroyEnv({
    namespace: body.namespace,
  });
  ctx.body = ctx.ok()
};

const setGroup = async (ctx) => {
  const res = await NamespaceGroups.save(ctx.request.body);
  ctx.body = ctx.ok(res)
};

const getImages = async (ctx) => {
  const repo = `library/${ctx.request.query.name}`.toLowerCase()
  const cache = await redisService.lrange(`${config.harborImageCache.name}_${repo}`, 0, config.harborImageCache.size - 1)

  if (cache && cache.length) {
    ctx.body = ctx.ok(cache);
    return
  }

  let res = await harborService.getTagsOfRepository(repo);
  res.sort((a, b) => new Date(b.created).getTime() - new Date(a.created).getTime());
  res = res.map(item => item.name);
  await redisService.rpush(`${config.harborImageCache.name}_${repo}`, res);
  ctx.body = ctx.ok(res)
};

const restartSystem = async (ctx) => {
  ctx.qalog.setActive('restart_microservice');
  const res = await eosService.restartMicroService({
    microservice_name: ctx.request.body.system_name,
    namespace: ctx.request.body.namespace,
  });
  ctx.body = ctx.ok(res)
};

const createSystem = async (ctx) => {
  ctx.qalog.setActive('create_microservice');
  const temp = {
    namespace: ctx.request.body.namespace,
    image: `${ctx.request.body.system_name}:${ctx.request.body.image}`,
    name: ctx.request.body.system_name,
    tier: ctx.request.body.tier,
  };
  const res = await eosService.createMicroService(temp);
  ctx.body = ctx.ok(res)
};

const restartAllSystem = async (ctx) => {
  ctx.qalog.setActive('restart_all_microservice');
  const { body } = ctx.request;
  body.system_names.forEach(async (item) => {
    await eosService.restartMicroService({
      microservice_name: item,
      namespace: body.namespace,
    })
  });
  ctx.body = ctx.ok('正在重置，请稍候~')
};

const updateSystem = async (ctx) => {
  ctx.qalog.setActive('update_microservice');
  const temp = {
    namespace: ctx.request.body.namespace,
    name: ctx.request.body.system_name,
    image: `${ctx.request.body.system_name}:${ctx.request.body.image}`,
    tier: ctx.request.body.tier,
  };
  const res = await eosService.updateMicroService(temp);
  ctx.body = ctx.ok(res)
};

const updateHost = async (ctx) => {
  ctx.qalog.setActive('update_host');
  const temp = {
    namespace: ctx.request.body.namespace,
    name: ctx.request.body.system_name,
    host: ctx.request.body.host,
  };
  const res = await eosService.updateHost(temp);
  ctx.body = ctx.ok(res)
};

const deleteSystem = async (ctx) => {
  ctx.qalog.setActive('delete_microservice');
  const data = {
    namespace: ctx.request.body.namespace,
    name: ctx.request.body.system_name,
  };
  const res = await eosService.deleteMicroService(data);
  ctx.body = ctx.ok(res)
};

const reloadMQ = async (ctx) => {
  const rule = {
    host: 'required',
  };
  const { body } = ctx.request;
  if (!ctx.validate(body, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT);
    return
  }
  const res = await mqService.getDefinitions();
  await mqService.setDefinitions(body.host, res);
  ctx.body = ctx.ok()
};

const flushRedis = async function (ctx) {
  ctx.qalog.setActive('flush_redis');
  const { namespace } = ctx.request.query;
  const envVars = await k8sService.listEnvVars(namespace);

  const configList = [];
  for (const prop in envVars) {
    if (prop.includes('REDIS_SERVICE_PORT')) {
      configList.push({
        host: envVars.REDIS_SERVICE_HOST,
        port: envVars[prop],
      })
    }
  }
  await redisService.flushMultiRedis(configList);
  ctx.body = ctx.ok({ msg: '清理缓存成功' })
};

const flushRedisTke = async function (ctx) {
  const { namespace } = ctx.request.query;
  const res = await ctx.curl({
    uri: `${api.tke_api}/service/details`,
    method: 'post',
    body: {
      namespace,
      serviceName: 'redis',
      type: 'base',
    },
    headers: {
      cluster: ctx.request.headers.cluster,
    },
  })
  const configList = [];
  const mapping = _.get(res.body, 'data.portMappings')
  const lanIp = _.get(res.body, 'data.lanIp')
  mapping.forEach((item) => {
    configList.push({
      host: lanIp,
      port: item.nodePort,
    })
  })
  await redisService.flushMultiRedis(configList);
  ctx.body = ctx.ok({ msg: '清理缓存成功' })
}

const listEnvVars = async function (ctx) {
  const envVars = await k8sService.listEnvVars(ctx.request.query.namespace);
  ctx.body = ctx.ok(envVars)
}

const mqDiff = async function (ctx) {
  const rule = {
    host: 'required',
  };
  if (!ctx.validate(ctx.request, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT);
    return
  }
  const result = {
    lost: {
      show: false,
      vhost: [],
      queue: [],
    },
    more: {
      show: false,
      vhost: [],
      queue: [],
    },
  }
  const host = ctx.request.body.host
  const current = await mqService.getDefinitionsOfHost(host);
  const online = await mqService.getDefinitions();

  const vhostPatcher = jsondiffpatch.create({
    objectHash(obj) {
      return obj.name
    },
  })
  const _vhostDiff = vhostPatcher.diff(online.vhosts, current.vhosts)

  for (const prop in _vhostDiff) {
    if (/^_[\d]+$/g.test(prop) && _vhostDiff[prop][0] !== '') {
      // _开头，表示当前环境缺少的数据；数组第一个元素为''表示该元素是位置变化了
      result.lost.vhost.push(_vhostDiff[prop][0].name)
    } else if (/^[\d]+$/.test(prop)) {
      // 纯数字表示当前环境新增的
      result.more.vhost.push(_vhostDiff[prop][0].name)
    }
  }

  const queuePatcher = jsondiffpatch.create({
    objectHash(obj) {
      return obj.vhost + obj.name
    },
  })
  const _queueDiff = queuePatcher.diff(online.queues, current.queues)
  for (const prop in _queueDiff) {
    // _开头，表示当前环境缺少的数据；数组第一个元素为''表示该元素是位置变化了
    if (/^_[\d]+$/g.test(prop) && _queueDiff[prop][0] !== '') {
      result.lost.queue.push(`${_queueDiff[prop][0].vhost}--${_queueDiff[prop][0].name}`)
    } else if (/^[\d]+$/.test(prop) && !_queueDiff[prop].arguments) {
      // 纯数字且没有arguments属性的表示当前环境新增的，有arguments属性的是queue特性变化的
      result.more.queue.push(`${_queueDiff[prop][0].vhost}--${_queueDiff[prop][0].name}`)
    }
  }

  if (result.lost.vhost.length || result.lost.queue.length) {
    result.lost.show = true
  }

  if (result.more.vhost.length || result.more.queue.length) {
    result.more.show = true
  }

  ctx.body = ctx.ok(result)
}

const router = new Router();
router
  .use(bodyParser())
  .get('/get_code', getCode)

  // 环境列表
  .get('/get_groups', getGroups)
  .get('/get_name_space', getActiveNameSpace)
  .post('/set_group', setGroup)
  .post('/hang', hang)
  // .post('/refresh_name_space_progress', refreshNameSpaceProgress)
  .post('/delete_namespace', deleteNamespace)

  // 详情
  .get('/get_device', getServiceOfNs)
  .post('/reload_mq', reloadMQ)
  .get('/flush_redis', flushRedis)
  .get('/flush_redis_tke', flushRedisTke)
  .post('/get_pod_info', getPodInfo)
  .get('/get_images', getImages)
  .post('/update_system', updateSystem)
  .post('/update_host', updateHost)
  .post('/delete_system', deleteSystem)
  .post('/create_system', createSystem)
  .get('/list_env_vars', listEnvVars)
  .post('/restart_all_system', restartAllSystem)
  .post('/mqDiff', mqDiff)

// 挂起环境
// .get('/query_hang', getHangNamespace)
// .post('/recovery_hang', recoveryHang)
// .post('/delete_hang', deleteHang)

  // 环境模板
  .post('/save_preset', savePreset)
  .post('/delete_preset', deletePreset)
  .post('/restart_system', restartSystem)

  // 创建新环境
  .get('/get_namespace_template', getNsTemplate)
  .get('/get_systems_by_type', getSystemsOfType)
  .post('/create_custom_namespace', createNamespace)

module.exports = router;
