const logger = require('koa-log4').getLogger('jobs:cleanImage')

const config = require(global.configPath)
const schedule = require('node-schedule')
const harbor = require('../service/harborServcice')

const tagsKeep = config.harborImageCache.size

async function run() {
  const projects = await harbor.getAllProjects()
  const targets = ['library']
  logger.info('准备清理的项目', targets)
  const targetIds = projects.filter(
    item => targets.includes(item.name),
  ).map(item => item.project_id)
  logger.info('准备清理的项目ids', targetIds)

  async function clearRepository(repoName) {
    const tags = await harbor.getTagsOfRepository(repoName)
    tags.sort((a, b) => new Date(a.created).getTime() - new Date(b.created).getTime())
    if (tags.length < tagsKeep) {
      return
    }

    const toRemoveTags = tags.slice(0, tags.length - tagsKeep)
    logger.info(`仓库 ${repoName} 需要清理 ${toRemoveTags.length} 个tag`)

    for (const tag of toRemoveTags) {
      logger.info(`清理tag ${tag.name}`)
      await harbor.removeTag(repoName, tag.name)
    }

    logger.info(`仓库 ${repoName} 清理完成, 共删除 ${toRemoveTags.length} 个tag`)
  }
  async function clearProject(projectId) {
    logger.info('开始清理项目', projectId)
    // 查询项目下的所有仓库
    let repositories = await harbor.getRepositoriesOfProject(projectId)
    // 清理tag数大于指定值的仓库
    repositories = repositories.filter(item => item.tags_count > tagsKeep)
    for (const repo of repositories) {
      await clearRepository(repo.name)
    }
  }

  // 串行执行
  for (const id of targetIds) {
    await clearProject(id)
  }
  logger.info('清理结束')
}

const job = schedule.scheduleJob('0 1 * * *', async () => {
  try {
    await harbor.login()
    await run()
  } catch (e) {
    logger.info('clearImage', e)
  }
})

module.exports = job
