/**
 * 根据不同维度来判断环境是否需要回收
 *
 * * */
const schedule = require('node-schedule')
const logger = require('koa-log4').getLogger('jobs:calculation cpu')
const request = require('request')
const moment = require('moment');
const config = require('../../src');

const elasticsearch = function (namespace) {
  const now = new Date();
  const aWeekAgo = moment().subtract(7, 'days').format()
  return {
    query: {
      bool: {
        must: [
          {
            range: {
              logtime: {
                format: 'strict_date_optional_time',
                gte: aWeekAgo,
                lte: now,
              },
            },
          },
        ],
        filter: [
          {
            bool: {
              filter: [
                {
                  bool: {
                    should: [
                      {
                        match: {
                          type: 'tke-eos',
                        },
                      },
                    ],
                    minimum_should_match: 1,
                  },
                },
                {
                  bool: {
                    should: [
                      {
                        match_phrase: {
                          message: `/service/listEnvVars?namespace=${namespace}`,
                        },
                      },
                    ],
                    minimum_should_match: 1,
                  },
                },
              ],
            },
          },
        ],
        should: [],
        must_not: [],
      },
    },
  }
}

const awaitRequest = function (options) {
  return new Promise((resolve, reject) => {
    request(options, (error, res, body) => {
      if (error) {
        reject(error)
      } else {
        resolve(JSON.parse(body))
      }
    })
  })
}
/**
 * 发送钉钉通知
 * @param msg
 */
const dingtalk = async function (namespace, msg) {
  const dingdingMessage = {
    msgtype: 'markdown',
    markdown: {
      title: '服务环境清理通知',
      text: '> 描述信息 : 以下环境长时间无人使用建议清理\n\n'
        + `> namespace : ${namespace}\n\n`
        + `> 建议原因 : ${msg}\n\n`
        + `> 时间 : ${moment()
          .format('YYYY-MM-DD HH:mm:ss')}\n\n`,
    },
  }
  await awaitRequest({
    url: 'https://oapi.dingtalk.com/robot/send?access_token=fda5fe4616b73ab85c087cdea55dec94d4655705ebaf68e0d0ab330f1e1de00b',
    method: 'POST',
    headers: {
      'Content-Type': 'application/json;charset=utf-8',
    },
    body: JSON.stringify(dingdingMessage),
  })
}
/**
 * 获取所有的服务
 * 一会去掉monitor
 * @returns {Promise<any[]>}
 */
const allServiceList = async function () {
  const namespaceList = await awaitRequest({
    url: `${config.api.tke_api}/namespace`,
    method: 'GET',
  })
  const task = []
  namespaceList.data.namespaces.forEach((item) => {
    const serviceList = awaitRequest({
      url: `${config.api.tke_api}/service?namespace=${item.name}`,
      method: 'GET',
    })
    task.push(serviceList)
  })
  const result = await Promise.all(task)
  return result
}

/**
 * 判断某个命名空间下所有的服务是否都创建于7天前
 * @param servicesList
 */
function filterServicesCreateTime(servicesList) {
  // 所有的服务都是7天前创建的 就返回true
  let flag = false
  const now = moment(new Date()).valueOf()
  for (let i = 0; i < servicesList.length; i += 1) {
    const startTime = moment(servicesList.startTime).valueOf()
    const dayNumber = Math.floor((now - startTime) / (1000 * 60 * 60 * 24))
    if (dayNumber < 7) {
      flag = false
      break
    } else {
      flag = true
    }
  }
  return flag
}

const job = async function () {
  const result = await allServiceList()
  result.forEach(async (item) => {
    const namespace = item.data.namespace
    if (item.data.services.length === 0) {
      dingtalk(namespace, '此命名空间下没有创建的服务')
    } else {
      const all7 = filterServicesCreateTime(item)
      if (all7) {
        // 去获取日志 看看日志是否有调用量
        const res = await awaitRequest({
          url: 'http://172.21.1.13:9200/logstash-indexer-tke-eos-2019.07/_search',
          method: 'POST',
          headers: {
            'Content-Type': 'application/json;charset=utf-8',
          },
          body: JSON.stringify(elasticsearch(namespace)),
        })
        // 如果不存在日志     缺少trarfik 条件
        if (res.hits.total.value === 0) {
          dingtalk(namespace, '此环境在一周时间内无服务使用')
        }
      }
    }
  })
}

module.exports = schedule.scheduleJob('0 0 10 * * ?', async () => {
  try {
    job()
  } catch (e) {
    logger.info('calculation cpu', e)
  }
})
