const Router = require('koa-router')
const bodyParser = require('koa-bodyparser')
const _ = require('lodash')
const mysql = require('mysql2/promise')
const logger = require('koa-log4').getLogger('dbsync')
const jenkinsService = require('../service/jenkinsService')
const k8sService = require('../service/k8sService')
const { DBConfig } = require('../service/mongoService')
const { api } = require('../../src')

const domain = 'liangkebang.com'
let domainTKE = 'liangkebang.net'

const getDBs = async function (ctx) {
  let dbs = await DBConfig.getDBName()
  dbs.unshift({ dbname: 'all' })
  dbs.unshift({ dbname: 'latest' })
  dbs = dbs.filter(i => i.dbname !== 'workflow')
  ctx.body = ctx.ok(dbs)
}

const getNamespace = async function (ctx) {
  const res = await k8sService.listEnv()
  ctx.body = ctx.ok(res)
}

const dbSync = async function (ctx) {
  const req = ctx.request.body

  const rule = {
    dbname: 'required',
    namespace: 'required',
  }

  if (!ctx.validate(req, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT)
    return
  }

  const env = await k8sService.listEnvVars(req.namespace)
  const res = await jenkinsService.build_with_params('sync_database_schema', {
    depth: 1,
    sync_db_name: req.dbname,
    target_db_host: env.DB_SERVICE_HOST,
    target_db_port: env.DB_SERVICE_PORT,
    re_replace_ip: req.namespace,
    not_delete_business_data: req.not_delete_business_data,
    use_cache: req.use_cache,
    domain,
  })
  ctx.body = ctx.ok({ location: res.location })
}

const dbSyncTke = async function (ctx) {
  const req = ctx.request.body
  // todo: 移到配置文件
  let publicIp
  if (ctx.request.headers.cluster === 'qke') {
    publicIp = '172.21.10.85'
    domainTKE = 'liangkebang.com'
  } else {
    publicIp = '172.17.5.17'
    domainTKE = 'liangkebang.net'
  }

  const rule = {
    dbname: 'required',
    namespace: 'required',
  }

  // todo: 改为新的validate方式
  if (!ctx.validate(req, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT)
    return
  }

  let res = await ctx.curl({
    uri: `${api.tke_api}/service/details`,
    headers: {
      cluster: ctx.request.headers.cluster,
    },
    body: {
      namespace: req.namespace,
      serviceName: req.mysqlName,
      type: 'base',
    },
  })

  if (res.body.code !== '0000') {
    ctx.body = ctx.fail(`${res.body.msg}：mysql`)
    return
  }

  const mapping = _.get(res.body, 'data.portMappings').filter(item => item.port === 3306)
  const port = _.get(mapping, '[0].nodePort', '')
  if (!port) {
    ctx.body = ctx.fail('获取mysql服务端口号失败')
    return
  }

  res = await jenkinsService.build_with_params('sync_database_schema', {
    depth: 1,
    sync_db_name: req.dbname,
    target_db_host: publicIp,
    target_db_port: port,
    re_replace_ip: req.namespace,
    not_delete_business_data: req.not_delete_business_data,
    use_cache: req.use_cache,
    domain: domainTKE,
  })
  ctx.body = ctx.ok({ location: res.location })
}

const dbSyncQuery = async function (ctx) {
  const req = ctx.request.body
  const number = req.location.match(/item\/([\d]+)/)[1]
  const apiJson = await jenkinsService.queue_item(number)

  if (apiJson.executable) {
    const buildInfo = await jenkinsService.build_info('sync_database_schema', apiJson.executable.number)
    const logInfo = await jenkinsService.console_output('sync_database_schema', apiJson.executable.number)
    if (buildInfo.building) {
      ctx.body = ctx.ok({ status: 'building', log: logInfo })
      return
    }
    ctx.body = ctx.ok({ status: 'complete', msg: buildInfo, log: logInfo })
    return
  }
  ctx.body = ctx.ok({ status: 'queue', log: { body: '' } })
}

const getDropSQL = async (ctx) => {
  const { dbName } = ctx.request.query
  const select = `select \`sql\`,base from core_sqlrecord where to_days(date) = to_days(now()) 
    and state like '%Execute Successfully%' and \`sql\` like '%drop %'`
  const connection = await mysql.createConnection({
    host: '172.30.4.8',
    user: 'sync_del',
    password: 'JhHgXdbopEPO6r6K',
    database: 'yearning',
  })
  const [rows] = await connection.execute(select)
  const res = rows.reduce((a, b) => {
    if (b.base === dbName) {
      a += `${b.sql};`
    }
    return a
  }, '')
  logger.info(dbName, 'dropSQL: ', res)
  ctx.body = res
}

const router = new Router()
router
  .get('/get_dbs', getDBs)
  .get('/get_name_space', getNamespace)
  .get('/getDropSQL', getDropSQL)
  .post('/db_sync', bodyParser(), dbSync)
  .post('/tke', bodyParser(), dbSyncTke)
  .post('/db_sync_query', bodyParser(), dbSyncQuery)

module.exports = router
