
const qs = require('querystring');
const logger = require('koa-log4').getLogger('service:harbor');
const config = require('../../src')
const http = require('./httpService')


class Harbor {
  constructor(config) {
    this.sessionId = ''
    this.config = config
  }

  request(option) {
    const req = Object.assign({
      timeout: 120000,
      headers: {},
    }, option)
    if (req.target) {
      req.url = `${this.config.api}/${req.target}`
    }
    if (this.sessionId) {
      req.headers.cookie = `beegosessionID=${this.sessionId}; rem-username=${this.config.user}`
    }
    if (req.method === 'post') {
      req.data = qs.stringify(req.data)
    }
    return http(req).catch((e) => {
      logger.error(e)
    })
  }

  login() {
    return this.request({
      method: 'post',
      target: 'login',
      rawResponse: true,
      data: {
        principal: this.config.user,
        password: this.config.password,
      },
    }).then((res) => {
      // 获取cookie
      if (res.headers['set-cookie'] && res.headers['set-cookie'][0]) {
        const match = /beegosessionID=([^;]+);/.exec(res.headers['set-cookie'][0])
        this.sessionId = (match && match[1]) || ''
        logger.info('harbor更新session:', this.sessionId)
      }
      return res.headers['set-cookie']
    }).catch()
  }

  statistics() {
    return this.request({
      method: 'get',
      target: '/api/statistics',
    })
  }

  async getAllProjects() {
    // const total = (await this.statistics()).public_project_count
    // 现在只有几个项目，先不查其他分页的数据
    return this.request({
      method: '',
      target: 'api/projects?page=1&page_size=100',
    })
  }

  // 获取项目下的最后更新仓库，按更新时间正序排
  async getRepositoriesOfProject(projectId) {
    let result = []
    const self = this
    const pageSize = config.harborImageCache.size
    function getPage(page = 1) {
      return self.request({
        method: 'get',
        target: `api/repositories?page=${page}&pageSize=${pageSize}&project_id=${projectId}`,
        rawResponse: true,
      }).then((res) => {
        result = result.concat(res.data)
        return res
      })
    }
    const res = await getPage(1)
    const total = res.headers['x-total-count'] || 0
    if (total <= pageSize) {
      return result
    }


    const totalPages = Math.ceil(total / pageSize)
    const task = []
    for (let i = 2; i <= totalPages; i += 1) {
      task.push(getPage(i))
    }
    await Promise.all(task)
    return result
  }

  getTagsOfRepository(repoName) {
    return this.request({
      method: 'get',
      target: `api/repositories/${repoName}/tags?detail=0`,
    })
  }

  removeTag(repoName, tagName) {
    return this.request({
      method: 'delete',
      target: `api/repositories/${repoName}/tags/${tagName}`,
    })
  }
}


const ins = new Harbor(config.harbor)

module.exports = ins
