const Router = require('koa-router')
const bodyParser = require('koa-bodyparser')
const _ = require('lodash')
const { ProConfig } = require('../service/mongoService')

const config = require(global.configPath)

const getProjectByType = async function (ctx) {
  let item = await ProConfig.findByQuery({ des: { $ne: '海外' } }, {}, { project_name: 1 })
  item = _.groupBy(item, 'type')

  const arr = []
  for (const key in item) {
    if (Object.prototype.hasOwnProperty.call(item, key)) {
      arr.push({
        name: key,
        data: item[key],
      })
    }
  }
  ctx.body = ctx.ok(arr)
}

const getProject = async function (ctx) {
  let sort = {}
  const data = ctx.request.query
  if (data.sort) {
    sort = data.sort.indexOf('+') ? { [data.sort.substr(1)]: -1 } : { [data.sort.substr(1)]: 1 }
    delete data.sort
  }
  const item = await ProConfig.findByQuery(data, {}, sort)
  ctx.body = ctx.ok(item)
}

const getProjectForJenkins = async function (ctx) {
  const data = ctx.request.query
  const query = { is_active: true, deploy_to_testenv: true }
  if (data.type) {
    Object.assign(query, { type: data.type })
  }
  let item = await ProConfig.findByQuery(query, { project_name: 1 }, { project_name: 1 })
  item = item.map(i => i.project_name)
  ctx.body = item.join('\n')
}

const save = async function (ctx) {
  const rule = {
    git_path: 'required',
    type: 'required',
  }
  const data = ctx.request.body
  if (!ctx.validate(data, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT)
    return
  }
  // 支持中文域名
  const match = data.git_path.match(/(?<=@)([A-Z0-9a-z\u4e00-\u9fa5-\.]+):([\w-\.]+)\/(.*).git$/)
  if (!match || match.length !== 4) {
    ctx.body = ctx.fail(ctx.Code.ERROR_GITPATH)
    return
  }

  if (data.host_name && ~data.host_name.indexOf('://')) {
    ctx.body = ctx.fail('线上域名不要有 http://')
    return
  }

  data.git_lab = match[1]
  data.git_path_group = match[2]
  data.project_name = match[3]
  for (const prop in data) {
    if (typeof prop === 'string' && data[prop] && data[prop].length) {
      data[prop] = data[prop].trim()
    }
  }
  if (data.database) {
    data.database = data.database.split(/ +/).join(' ')
  }
  await ProConfig.saveProject(data)
  ctx.body = ctx.ok()
}

const getNewPort = async function (ctx) {
  const port = await ProConfig.getNewPort()
  ctx.body = ctx.ok(port + 1)
}

const remove = async function (ctx) {
  const rule = {
    _id: 'required',
  }
  const data = ctx.request.query
  if (!ctx.validate(data, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT)
    return
  }
  const item = await ProConfig.delete(ctx.request.query._id)
  ctx.body = ctx.ok(item)
}

const getInfo = async function (ctx) {
  const rule = {
    system_name: 'required',
  }
  const data = ctx.request.query
  if (!ctx.validate(data, rule)) {
    ctx.body = ctx.fail(ctx.Code.ERROR_INPUT)
    return
  }

  const item = await ProConfig.getOneProject({ project_name: data.system_name })
  if (data.key && item[data.key]) {
    ctx.body = item[data.key]
    return
  }
  ctx.body = ctx.ok(item)
}

const getType = async function (ctx) {
  const item = await ProConfig.getType()
  ctx.body = ctx.ok(item)
}

const getDomain = async function (ctx) {
  const item = await ProConfig.getOneProject({ project_name: ctx.request.query.project_name })
  if (item && item.project_name === 'recruiting-management') {
    item.host_name = 'recruitui.liangkebang.net'
  }
  if (item && item.project_name === 'smart-recruitment') {
    item.host_name = 'recruitapi.liangkebang.net'
  }
  ctx.body = ctx.ok(item)
}

const getK8sConf = async function (ctx) {
  const item = await ProConfig.getOneProject({ project_name: ctx.request.query.project_name }, {
    cpuRequest: 1, cpuLimit: 1, memRequest: 1, memLimit: 1, _id: 0,
  })
  ctx.body = ctx.ok(item)
}

const subSet = function (arr1, arr2) {
  const set1 = new Set(arr1);
  const set2 = new Set(arr2);
  const subset = [];
  for (const item of set1) {
    if (!set2.has(item)) {
      subset.push(item);
    }
  }
  return subset;
};

const getLatestSvcDB = async (ctx) => {
  const ns = ctx.request.query.namespace
  const res = await ctx.curl({
    uri: `${config.api.tke_api}/service?namespace=${ns}`,
    method: 'get',
    headers: {
      cluster: ctx.request.headers.cluster || 'qa',
    },
  })
  const allDB = []
  const notLatestDB = []
  const svcs = _.get(res, 'body.data.services', [])

  for (const i of svcs) {
    if (i.labels.type === 'java') {
      const proconfig = await ProConfig.getOneProject({ project_name: i.serviceName }, { database: 1, _id: 0 })
      if (proconfig.database) {
        allDB.push(...proconfig.database.split(' '))
        if (i.image.indexOf('latest') === -1 && i.image.indexOf(':master-') === -1) {
          notLatestDB.push(...proconfig.database.split(' '))
        }
      }
    }
  }
  const latestDB = subSet(allDB, notLatestDB)
  if (ctx.request.query.type === 'json') {
    ctx.body = ctx.ok(latestDB)
  } else {
    ctx.body = latestDB.join(' ')
  }
}

const router = new Router()
router
  .use(bodyParser())
  .get('/get_project_by_type', getProjectByType)
  .get('/get_project', getProject)
  .get('/get_project_for_jenkins', getProjectForJenkins)
  .get('/get_info', getInfo)
  .get('/get_new_port', getNewPort)
  .get('/delete', remove)
  .post('/save', save)
  .get('/get_type', getType)
  .get('/getHostName', getDomain)
  .get('/getK8sConf', getK8sConf)
  .get('/getLatestSvcDB', getLatestSvcDB)

module.exports = router
