package cn.qg.holmes.utils;


import com.google.common.collect.Lists;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component
public class ApplicationContextUtils implements ApplicationContextAware {

    private static ApplicationContext applicationContext;

    public static ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        ApplicationContextUtils.applicationContext = applicationContext;
    }

    public static boolean containsBean(String name) {
        return applicationContext.containsBean(name);
    }

    /**
     * 根据名称获取一个对象.
     *
     * @param name bean名称
     * @return Object 指定的bean
     * @throws BeansException 如果找不到bean
     */
    public static <T> T getBean(String name) throws BeansException {
        return (T) applicationContext.getBean(name);
    }

    /**
     * 获取类型为requiredType的对象.
     *
     * @param <T>          需求的bean类型
     * @param requiredType 需求的bean类型
     * @return 指定类型的bean
     * @throws BeansException 如果找不到匹配的类型
     */
    public static <T> T getBean(Class<T> requiredType) throws BeansException {
        return applicationContext.getBean(requiredType);
    }

    /**
     * 获取名称为name的bean，自动转为所需类型.
     *
     * @param <T>          需求的bean类型
     * @param name         bean名称
     * @param requiredType 需求的bean类型
     * @return 指定类型的bean
     * @throws BeansException 如果找不到匹配的类型，或是类型不能被转换，或是bean实例化失败
     */
    public static <T> T getBean(String name, Class<T> requiredType) throws BeansException {
        return applicationContext.getBean(name, requiredType);
    }

    /**
     * 根据类型获取实现类列表
     * @param requiredType
     * @param <T>
     * @return
     * @throws BeansException
     */
    public static <T> List<T> getBeansByType(Class<T> requiredType) throws BeansException {
        Map<String, T> beansOfType = applicationContext.getBeansOfType(requiredType);
        if(MapUtils.isEmpty(beansOfType)){
            return Lists.newArrayList();
        }
        return  beansOfType.values()
                .stream().collect(Collectors.toList());
    }
}
