package cn.qg.holmes.task.jira;

import cn.qg.holmes.entity.jira.DingRobot;
import cn.qg.holmes.entity.jira.JiraBugPool;
import cn.qg.holmes.service.jira.DingRobotService;
import cn.qg.holmes.service.jira.JiraBugPoolService;
import cn.qg.holmes.service.jira.JiraIssueService;
import cn.qg.holmes.utils.DingdingUtils;
import com.atlassian.jira.rest.client.api.domain.Issue;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * Jira BUG通知到钉钉定时任务类
 */
@Component
@Slf4j
public class JiraToDingding {

    @Autowired
    JiraIssueService jiraIssueService;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    @Autowired
    DingRobotService dingRobotService;

    @Value("${sjgd.ding.url}")
    private String sjgdDingUrl;

    @Value("${task.debug}")
    private String taskDebug;

    /**
     * 数据工单bug统计
     * @throws Exception
     */
    @Scheduled(cron = "0 0 19 * * ?")
    public void SJGDJiraIssueStatisticsTimedJob() throws Exception {
        if (taskDebug.equals("true")) {
            log.info("开始执行数据工单BUG统计定时任务！");
            Iterable<Issue> issues = jiraIssueService.getUnsolvedIssueListByProject("SJGD");
            String markdownMsg = DingdingUtils.buildMarkdownMsgForUnsolvedIssueList(issues);
            DingdingUtils.sendToDingding(markdownMsg, sjgdDingUrl);
        } else {
            log.info("当前部署的是mock，不执行定时任务！");
        }
    }

    /**
     * bug池里BUG循环
     * @throws Exception
     */
    @Scheduled(cron = "0 0 10-20 ? * MON-FRI")
    public void cycleJiraBugPool() throws Exception {
        if (taskDebug.equals("true")) {
            log.info("开始执行jira bug pool定时任务！");
            QueryWrapper<JiraBugPool> jiraBugPoolQueryWrapper = new QueryWrapper<>();
            jiraBugPoolQueryWrapper.eq("enable", 1);
            List<JiraBugPool> jiraBugPoolList = jiraBugPoolService.list(jiraBugPoolQueryWrapper);
            for (JiraBugPool jiraBugPool: jiraBugPoolList) {
                Issue issue = jiraIssueService.getJiraIssueByKey(jiraBugPool.getKey());
                if (issue != null) {
                    String resolveResult = issue.getStatus().getName();
                    // 如果已关闭，仅修改状态
                    if (resolveResult.equals("关闭")) {
                        jiraBugPool.setEnable(0);
                        // 更新状态
                        jiraBugPoolService.saveOrUpdate(jiraBugPool);
                    } else if (resolveResult.equals("已解决")) {
                        // 暂时只打印日志，不做逻辑处理
                        log.info("BUG:{}已解决", issue.getKey());
                    } else {
                        DateTime startDate = issue.getCreationDate();
                        DateTime curDate = new DateTime();
                        int duration = jiraIssueService.calculateBugFixTime(startDate, curDate);
                        log.info("{} BUG持续时间：{}小时", issue.getKey(), duration);
                        // 如果已超过4个小时，则发送钉钉通知
                        if (duration >= 4) {
                            DingRobot dingRobot = dingRobotService.getById(jiraBugPool.getDingRobotId());
                            // 项目进行中才发送通知
                            if (dingRobot.getStatus() == 1) {
                                DingdingUtils.sendToDingding(DingdingUtils.buildBugFixRemindMsg(issue, duration), jiraBugPool.getDingUrl());
                            }
                        }
                    }
                } else {
                    log.info("JIRA上未找到该issue，将BUG: {} 置为无效", jiraBugPool.getKey());
                    jiraBugPool.setEnable(0);
                }
            }
        }
    }
}
