package cn.qg.holmes.service.auto.impl;

import cn.qg.holmes.service.auto.AutoUtilsService;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * @author libo
 * created at 2021-03-30
 */
@Service
@Slf4j
public class AutoUtilsServiceImpl implements AutoUtilsService {

    @Autowired
    RedisUtils redisUtils;

    /**
     * 将请求headers中的变量替换成对应的值
     * @param headers 请求头json
     * @param variables 参数json
     * @return
     */
    public Map<String, String> replaceHeaders(String headers, String variables, UUID uuid) {
        log.info("开始执行请求头替换！");
        Map<String, String> headersMap = JSON.parseObject(headers, Map.class);
        Map<String, String> varMap = JSON.parseObject(variables, Map.class);
        log.info("替换之前的headers：{}", headersMap);
        log.info("参数列表：{}", varMap);
        for (String key: varMap.keySet()) {
            String value = varMap.get(key);
            headersMap.put(key, redisUtils.get(uuid + "_" + value.substring(1)).toString());
        }
        log.info("替换之后的headers：{}", headersMap);
        return headersMap;
    }

    /**
     * 将请求体中的变量替换成对应的值
     * @param parameters 请求参数json
     * @param variables 变量json
     * @return 替换后的参数Map
     */
    public Map<String, String> replaceParameters(String parameters, String variables, UUID uuid) {
        log.info("开始执行请求参数替换！");
        Map<String, String> parameterMap = JSON.parseObject(parameters, Map.class);
        Map<String, String> varMap = JSON.parseObject(variables, Map.class);
        log.info("替换之前的参数：{}", parameterMap);
        log.info("参数列表：{}", varMap);
        for (String key: varMap.keySet()) {
            String value = varMap.get(key);
            parameterMap.put(key, redisUtils.get(uuid + "_" + value.substring(1)).toString());
        }
        log.info("替换之后的参数：{}", parameterMap);
        return parameterMap;
    }

    /**
     * 参数替换-新
     * @param parameters headers或者parameters
     * @param uuid 唯一标识
     */
    @Override
    public Map<String, String> replaceVariables(String parameters, UUID uuid) {
        Map<String, String> parameterMap = JSON.parseObject(parameters, Map.class);
        log.info("替换之前的参数：{}", parameterMap);
        for (String key: parameterMap.keySet()) {
            String value = parameterMap.get(key);
            if (value.startsWith("$")) {
                parameterMap.put(key, redisUtils.get(uuid + "_" + value.substring(1)).toString());
            }
        }
        log.info("替换之后的参数：{}", parameterMap);
        return parameterMap;
    }

    /**
     * 将响应中的值解析出来并存储到redis值
     * @param response 响应json
     * @param extract 解析的字段及保存到redis的key
     * @return true-解析成功，false-解析失败
     */
    public boolean extractResponse(String response, String extract, UUID uuid) {
        Map<String, String> extractMap = JSON.parseObject(extract, Map.class);
        try {
            for (String key: extractMap.keySet()) {
                String value = extractMap.get(key);
                String redisKey = uuid + "_" + key;
                redisUtils.set(redisKey, JsonPath.read(response, value));
                redisUtils.expire(redisKey, 120);
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    /**
     * 响应断言
     * @return
     */
    public boolean assertResponse(String response, List<Map> validateList) {
        for (Map<String, String> map: validateList) {
            String comparator = map.get("comparator");
            String checkpoint = JsonPath.read(response, map.get("check")).toString();
            String expectValue = map.get("expect");
            boolean result = singleAssertResponse(comparator, checkpoint, expectValue);
            if (!result) {
                return false;
            }
        }
        return true;
    }

    /**
     *
     * @param comparator
     * @param checkpoint
     * @param expectValue
     * @return
     */
    public boolean singleAssertResponse(String comparator, String checkpoint, String expectValue) {
        switch (comparator) {
            case "eq":
                return checkpoint.equals(expectValue);
            case "gt":
                return Integer.parseInt(checkpoint) > Integer.parseInt(expectValue);
            case "lt":
                return Integer.parseInt(checkpoint) < Integer.parseInt(expectValue);
            case "neq":
                return !checkpoint.equals(expectValue);
            default:
                log.info("暂不支持该比较符号：{}", comparator);
                break;
        }
        return false;
    }
}
