package cn.qg.holmes.controller.k8s;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import cn.qg.holmes.service.k8s.DockerProjectService;
import cn.qg.holmes.service.k8s.ImageService;
import cn.qg.holmes.service.k8s.MqService;
import cn.qg.holmes.service.k8s.TkeService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * k8及容器操作相关接口
 * @author libo
 */
@CrossOrigin
@RestController
@RequestMapping("/k8s")
public class K8sController {

    @Autowired
    TkeService tkeService;

    @Autowired
    ImageService imageService;

    @Autowired
    DockerProjectService dockerProjectService;

    @Autowired
    MqService mqService;

    /**
     * 获取namespace列表
     * @return
     */
    @GetMapping("/namespace")
    public JsonResult getNamespaceList() {
        return JsonResult.buildSuccessResult(tkeService.getNamespaceList());
    }

    /**
     * 获取pod列表
     * @param namespace 环境
     * @return
     */
    @GetMapping("/service/list")
    public JsonResult getServiceList(@RequestParam String namespace) {
        List<Map<String, Object>> podList = tkeService.getPodList(namespace);
        return JsonResult.buildSuccessResult(podList);
    }

    /**
     * 根据服务名称获取镜像列表
     * @param repoName 服务名称
     * @return
     */
    @GetMapping("/image/list")
    public JsonResult getImageListByServiceName(@RequestParam String repoName) {
        return JsonResult.buildSuccessResult(imageService.getImageListByService(repoName));
    }

    /**
     * 新增deployment, 如果service和pvc没有，也会同时创建
     * @param serviceCreateVo 创建实体类
     * @return
     */
    @PostMapping("/service/create")
    public JsonResult createPod(@RequestBody ServiceCreateVo serviceCreateVo) {
        try {
            String serviceName = serviceCreateVo.getServiceName();
            String type = serviceCreateVo.getType();
            String label = serviceCreateVo.getLabel();
            String namespace = serviceCreateVo.getNamespace();
            String image = serviceCreateVo.getImage();
            QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("project_name", serviceName);
            if (label.equals("base")) {
                // Service
                if (!tkeService.queryIfServiceExistByName(namespace, serviceName)) {
                    if (serviceName.equals("mysql")) {
                        tkeService.createMysqlService(namespace);
                    }
                    if (serviceName.equals("redis")) {
                        tkeService.createRedisService(namespace);
                    }
                    if (serviceName.equals("mongodb")) {
                        tkeService.createMongodbService(namespace);
                    }
                    if (serviceName.equals("zookeeper")) {
                        tkeService.createZookeeperService(namespace);
                    }
                    if (serviceName.equals("rabbitmq")) {
                        tkeService.createRabbitmqService(namespace);
                    }
                }
                // Pvc
                if (!tkeService.queryIfPvcExistByName(namespace, serviceName)) {
                    if (serviceName.equals("mysql")) {
                        tkeService.createMysqlPvc(namespace);
                    }
                    // redis不需要pvc
//                    if (serviceName.equals("redis")) {
//                        tkeService.createRedisPvc(namespace);
//                    }
                    if (serviceName.equals("mongodb")) {
                        tkeService.createMongodbPvc(namespace);
                    }
                    if (serviceName.equals("zookeeper")) {
                        tkeService.createZookeeperPvc(namespace);
                    }
                    if (serviceName.equals("rabbitmq")) {
                        tkeService.createRabbitmqPvc(namespace);
                    }
                }
                // deployment
                if (!tkeService.queryIfDeploymentExistByName(namespace, serviceName)) {
                    if (serviceName.equals("mysql")) {
                        tkeService.createMysqlDeployment(namespace, image);
                    }
                    if (serviceName.equals("redis")) {
                        tkeService.createRedisDeployment(namespace, image);
                    }
                    if (serviceName.equals("mongodb")) {
                        tkeService.createMongoDbDeployment(namespace, image);
                    }
                    if (serviceName.equals("zookeeper")) {
                        tkeService.createZookeeperDeployment(namespace, image);
                    }
                    if (serviceName.equals("rabbitmq")) {
                        tkeService.createRabbitmqDeployment(namespace, image);
                    }
                }
            } else if (label.equals("java")) {
                DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
                if (!tkeService.queryIfServiceExistByName(namespace, serviceName)) {
                    tkeService.createJavaService(namespace, serviceName);
                }
                if (!tkeService.queryIfDeploymentExistByName(namespace, serviceName)) {
                    tkeService.createJavaDeployment(serviceCreateVo, dockerProject);
                }
                if (!tkeService.queryIfIngressExistByName(namespace, serviceName)) {
                    tkeService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());
                }
            } else if (label.equals("ui") || label.equals("node")) {
                DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
                if (!tkeService.queryIfServiceExistByName(namespace, serviceName)) {
                    tkeService.createUIAndNodeService(namespace, serviceName, type, label);
                }
                if (!tkeService.queryIfDeploymentExistByName(namespace, serviceName)) {
                    tkeService.createUIAndNodeDeployment(serviceCreateVo, dockerProject);
                }
                if (!tkeService.queryIfIngressExistByName(namespace, serviceName)) {
                    tkeService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());
                }
            } else {
                return JsonResult.buildErrorStateResult("暂不支持！", false);
            }
            return JsonResult.buildSuccessResult("pod创建成功！");
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("pod创建失败！", false);
        }
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName podName
     * @return
     */
    @PostMapping("/service/redeploy")
    public JsonResult resetPodByName(String namespace, String podName) {
        return JsonResult.buildSuccessResult(tkeService.resetPod(namespace, podName));
    }

    /**
     * 实际上是更新pod
     * @param namespace 环境
     * @param serviceName 服务名
     * @param image 镜像名
     * @return
     */
    @PostMapping("/service/modify")
    public JsonResult modifyPod(String namespace, String serviceName, String image) {
        return JsonResult.buildSuccessResult(tkeService.updateDeployment(namespace, serviceName, image));
    }

    /**
     * 删除pod
     * @param namespace 环境
     * @param serviceName 服务名称
     * @return
     */
    @PostMapping("/service/delete")
    public JsonResult deletePodByName(String namespace, String serviceName) {
        return JsonResult.buildSuccessResult(tkeService.deleteDeployment(namespace, serviceName));
    }

    /**
     * 根据类型获取所有的镜像名
     * @param namespace qa-base，qa-ui，qa-java等
     * @return
     */
    @GetMapping("/getSingleTypeApp")
    public JsonResult getSingleTypeApp(@RequestParam String namespace) {
        return JsonResult.buildSuccessResult(imageService.getSingleTypeApp(namespace));
    }

    @PostMapping("/sync/mq")
    public JsonResult syncOnlineMqToTest(String host) {
        return JsonResult.buildSuccessResult(mqService.setDefinitions(host));
    }
}
