package cn.qg.holmes.service.effect.impl;

import cn.qg.holmes.service.effect.DatabaseSyncService;
import cn.qg.holmes.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Slf4j
@Service
public class DatabaseSyncServiceImpl implements DatabaseSyncService {

    @Autowired
    RedisUtils redisUtils;

    @Value("${cash_loan_flow.sql}")
    private String cashLoanFlowSql;

    @Value("${contract.sql}")
    private String contractSql;

    @Value("${payment_center.sql}")
    private String paymentCenterSql;

    private final String dbSyncPrefix = "dbsync:";

    /**
     * 实现StringBuilder的replaceAll
     *
     * @param stb
     * @param oldStr 被替换的字符串
     * @param newStr 替换oldStr
     * @return
     */
    public StringBuilder replaceAll(StringBuilder stb, String oldStr, String newStr) {
        if (stb == null || oldStr == null || newStr == null || stb.length() == 0 || oldStr.length() == 0)
            return stb;
        int index = stb.indexOf(oldStr);
        if (index > -1 && !oldStr.equals(newStr)) {
            int lastIndex = 0;
            while (index > -1) {
                stb.replace(index, index + oldStr.length(), newStr);
                lastIndex = index + newStr.length();
                index = stb.indexOf(oldStr, lastIndex);
            }
        }
        return stb;
    }

    /**
     * 从中间库获取数据库相关信息
     * @param ip 中间库 ip
     * @param port 中间库 port
     * @param username  中间库用户名
     * @param password 中间库密码
     * @param dbName 数据库名
     * @return
     */
    @Override
    public boolean getDbInfoFromSource(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;
        String dbRedisKey = dbSyncPrefix + dbName;
        String tableListKey = dbSyncPrefix + dbName + ":tables";
        Connection connection = null;
        PreparedStatement preparedStatement = null;

        // 创建数据库的redis值
        String dbRedisValue = "";
        // 表数据redis值
        StringBuilder insertRedisValue = new StringBuilder();
        // 建表语句redis值
        String createTableRedisValue = "";
        // 表名列表redis值
        StringBuilder tableListStr = new StringBuilder();

        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);

            if (!redisUtils.hasKey(dbRedisKey)) {
                String dbSql = String.format("SHOW CREATE DATABASE %s", dbName);
                preparedStatement = connection.prepareStatement(dbSql);
                ResultSet dbResultSet = preparedStatement.executeQuery();
                while (dbResultSet.next()) {
                    dbRedisValue = dbResultSet.getString(2);
                }
                dbResultSet.close();
                // 设置建库语句的redis值
                redisUtils.set(dbRedisKey, dbRedisValue, 86400);
            }

            DatabaseMetaData databaseMetaData = connection.getMetaData();
            // 获取所有表名
            ResultSet tableResultSet = databaseMetaData.getTables(null, null, null, new String[]{"TABLE"});
            // 遍历表，一个表一个表处理
            while (tableResultSet.next()) {
                String tableName = tableResultSet.getString("TABLE_NAME");
                tableListStr.append(tableName);
                tableListStr.append("\n");

                long startTime = System.currentTimeMillis();
                log.info("开始获取表{}的数据", tableName);
                String createTableKey = dbSyncPrefix + dbName + ":" + tableName + ":create" ;
                String insertTableKey = dbSyncPrefix + dbName + ":" + tableName + ":insert" ;

                // 表结构
                if (!redisUtils.hasKey(createTableKey)) {
                    // 获取所有建表语句
                    String sql = String.format("SHOW CREATE TABLE %s", tableName);
                    preparedStatement = connection.prepareStatement(sql);
                    ResultSet pResultSet = preparedStatement.executeQuery();
                    while (pResultSet.next()) {
                        createTableRedisValue = pResultSet.getString(2);
                    }
                    pResultSet.close();

                    redisUtils.set(createTableKey, createTableRedisValue, 86400);
                    createTableRedisValue = "";
                }

                // 表数据
                if (!redisUtils.hasKey(insertTableKey)) {
                    ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                    while (columnResultSet.next()) {
                        // 字段名称
                        String columnName = columnResultSet.getString("COLUMN_NAME");
                        // 数据类型
                        String columnType = columnResultSet.getString("TYPE_NAME");

                        columnNameList.add(columnName);
                        columnTypeList.add(columnType);
                    }
                    columnResultSet.close();

                    StringBuilder columnArrayStr = new StringBuilder();
                    for (String column : columnNameList) {
                        if (columnArrayStr.length() == 0) {
                            columnArrayStr.append("`").append(column).append("`");
                        } else {
                            columnArrayStr.append(",`").append(column).append("`");
                        }
                    }

                    preparedStatement = connection.prepareStatement("select " + columnArrayStr + " from " + tableName);
                    ResultSet selectResultSet = preparedStatement.executeQuery();
                    StringBuilder insertSql = new StringBuilder();
                    while (selectResultSet.next()) {
                        String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
                        insertSql
                                .append("insert into ")
                                .append(tableName)
                                .append(" (")
                                .append(columnArrayStr)
                                .append(") ")
                                .append("values (")
                                .append(rowValues)
                                .append(");");
                        insertSql = replaceAll(insertSql, "\n", "<br/>");
                        insertSql.append("\n");
                        insertRedisValue.append(insertSql);
                        // 清空insertSql
                        insertSql.delete(0, insertSql.length());
                    }
                    selectResultSet.close();
                    columnNameList.clear();
                    columnTypeList.clear();

                    redisUtils.set(insertTableKey, insertRedisValue.toString(), 86400);
                    long endTime = System.currentTimeMillis();
                    log.info("{}表数据获取完成，共花费{}秒", tableName, (endTime - startTime) / 1000);
                    // 清空insertRedisValue
                    insertRedisValue.delete(0, insertRedisValue.length());
                }
            }
            tableResultSet.close();
            redisUtils.set(tableListKey, tableListStr.toString(), 86400);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 同步数据库信息到目标地址
     * @param ip 目标库ip
     * @param port 目标库端口
     * @param username 目标库用户名
     * @param password 目标库密码
     * @param dbName 目标库名称
     * @return
     */
    @Override
    public boolean syncDbToDest(String ip, String port, String username, String password, String dbName, String namespace, boolean businessData) {
        try {
            String tableListKey = dbSyncPrefix + dbName + ":tables";
            List<String> tableList = Arrays.asList(redisUtils.get(tableListKey).toString().split("\n"));
            if (tableList.size() > 0) {
                for (String tableName: tableList) {
                    syncSingleTableToDest(ip, port, username, password, dbName, tableName, businessData);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    /**
     * 获取单个表
     * @param ip 同步库ip
     * @param port 同步库端口
     * @param username 同步库用户名
     * @param password 同步库密码
     * @param dbName 数据库名
     * @param tableName 表名
     * @return
     */
    @Override
    public boolean getSingleTableFromSource(String ip, String port, String username, String password, String dbName, String tableName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        String dbRedisKey = dbSyncPrefix + dbName;
        String tableCreateKey = dbSyncPrefix + dbName + ":" + tableName + ":create";
        String tableInsertKey = dbSyncPrefix + dbName + ":" + tableName + ":insert";
        String dbRedisValue = null;
        StringBuilder insertRedisValue = new StringBuilder();

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(true);
            DatabaseMetaData databaseMetaData = connection.getMetaData();

            // 判断是否有创建数据库的redis值
            if (!redisUtils.hasKey(dbRedisKey)) {
                String dbSql = String.format("SHOW CREATE DATABASE %s", dbName);
                preparedStatement = connection.prepareStatement(dbSql);
                ResultSet dbResultSet = preparedStatement.executeQuery();
                while (dbResultSet.next()) {
                    dbRedisValue = dbResultSet.getString(2);
                }
                dbResultSet.close();
                // 设置建库语句的redis值
                redisUtils.set(dbRedisKey, dbRedisValue, 86400);
            }

            if (!redisUtils.hasKey(tableCreateKey)) {
                String sql = String.format("SHOW CREATE TABLE %s", tableName);
                preparedStatement = connection.prepareStatement(sql);
                ResultSet resultSet = preparedStatement.executeQuery();
                while (resultSet.next()) {
                    redisUtils.set(tableCreateKey, resultSet.getString(2), 600);
                }
                resultSet.close();
            }

            if (!redisUtils.hasKey(tableInsertKey)) {
                log.info("redis中没有{}表的数据，开始获取表数据。", tableName);
                ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                while (columnResultSet.next()) {
                    // 字段名称
                    String columnName = columnResultSet.getString("COLUMN_NAME");
                    // 数据类型
                    String columnType = columnResultSet.getString("TYPE_NAME");

                    columnNameList.add(columnName);
                    columnTypeList.add(columnType);;
                }
                columnResultSet.close();

                StringBuilder columnArrayStr = new StringBuilder();
                for (String column : columnNameList) {
                    if (columnArrayStr.length() == 0) {
                        columnArrayStr.append("`").append(column).append("`");
                    } else {
                        columnArrayStr.append(",`").append(column).append("`");
                    }
                }

                String selectSQL = String.format("select %s from %s", columnArrayStr, tableName);
                preparedStatement = connection.prepareStatement(selectSQL);
                ResultSet selectResultSet = preparedStatement.executeQuery();
                StringBuilder insertSql = new StringBuilder();
                while (selectResultSet.next()) {
                    String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
//                    String insertSql = String.format("insert into %s (%s) values(%s);", tableName, columnArrayStr, rowValues);
//                    insertSql = insertSql.replaceAll("\n", "<br/>");
//                    insertSql = insertSql + "\n";
                    insertSql
                            .append("insert into ")
                            .append(tableName)
                            .append(" (")
                            .append(columnArrayStr)
                            .append(") ")
                            .append("values (")
                            .append(rowValues)
                            .append(");");
                    insertSql = replaceAll(insertSql, "\n", "<br/>");
                    insertSql.append("\n");
                    insertRedisValue.append(insertSql);
                    insertSql.delete(0, insertSql.length());
                }
                selectResultSet.close();

                redisUtils.set(tableInsertKey, insertRedisValue.toString(), 600);
                insertRedisValue.delete(0, insertRedisValue.length());
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 同步单张表
     * @param ip
     * @param port
     * @param username
     * @param password
     * @param dbName
     * @param tableName
     * @param businessData 是否同步业务数据
     * @return
     */
    @Override
    public boolean syncSingleTableToDest(String ip, String port, String username, String password, String dbName, String tableName, boolean businessData) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;
        String urlWithoutDb = "jdbc:mysql://" + ip + ":" + port;

        String createDbKey = dbSyncPrefix + dbName;
        String tableCreateKey = dbSyncPrefix + dbName + ":" + tableName + ":create";
        String tableInsertKey = dbSyncPrefix + dbName + ":" + tableName + ":insert";

        Connection connection = null;
        Connection connectionWithoutDb = null;
        PreparedStatement preparedStatement = null;
        Statement statement = null;

        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();

        try {
            Class.forName(driver);

            String createDbSql = redisUtils.get(createDbKey).toString();
            String createTableSql = redisUtils.get(tableCreateKey).toString();
            String insertTableSql = redisUtils.get(tableInsertKey).toString();
            String dropTableSql = String.format("DROP TABLE if exists %s", tableName);

            // 暂存原有数据
            StringBuilder backupDataSql = new StringBuilder();
            List<String> dbList = getDatabaseList(ip, port, username, password);

            if (dbList.contains(dbName)) {
                // 环境中数据库已存在
                // 创建数据库连接
                connection = DriverManager.getConnection(url, username, password);
                connection.setAutoCommit(false);
                DatabaseMetaData databaseMetaData = connection.getMetaData();
                // 获取表列表
                List<String> tableList = getTableListByDb(ip, port, username, password, dbName);
                if (tableList.contains(tableName)) {
                    // 如果表存在
                    if (businessData) {
                        // 需要保留业务数据
                        log.info("因需要保留业务数据，因此先备份环境中{}.{}表的数据", dbName, tableName);
                        ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                        while (columnResultSet.next()) {
                            // 字段名称
                            String columnName = columnResultSet.getString("COLUMN_NAME");
                            // 数据类型
                            String columnType = columnResultSet.getString("TYPE_NAME");

                            columnNameList.add(columnName);
                            columnTypeList.add(columnType);;
                        }
                        columnResultSet.close();

                        StringBuilder columnArrayStr = new StringBuilder();
                        for (String column : columnNameList) {
                            if (columnArrayStr.length() == 0) {
                                columnArrayStr.append("`").append(column).append("`");
                            } else {
                                columnArrayStr.append(",`").append(column).append("`");
                            }
                        }

                        String selectSQL = String.format("select %s from %s", columnArrayStr, tableName);
                        preparedStatement = connection.prepareStatement(selectSQL);
                        ResultSet selectResultSet = preparedStatement.executeQuery();
                        StringBuilder insertSql = new StringBuilder();
                        while (selectResultSet.next()) {
                            String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
                            insertSql
                                    .append("insert into ")
                                    .append(tableName)
                                    .append(" (")
                                    .append(columnArrayStr)
                                    .append(") ")
                                    .append("values (")
                                    .append(rowValues)
                                    .append(");");
                            insertSql = replaceAll(insertSql, "\n", "<br/>");
                            insertSql.append("\n");
                            backupDataSql.append(insertSql);
                            insertSql.delete(0, insertSql.length());
                        }
                        selectResultSet.close();

                        // 如果表存在，则首先删除表
                        log.info("表数据备份完成，删除环境中的{}.{}表", dbName, tableName);
                        preparedStatement = connection.prepareStatement(dropTableSql);
                        preparedStatement.execute();

                        // 重新创建表
                        log.info("使用同步库中的表结构重新创建表: {}.{}", dbName, tableName);
                        preparedStatement = connection.prepareStatement(createTableSql);
                        preparedStatement.execute();

                        // 环境备份数据不为空，插入环境备份的数据
                        if (backupDataSql.length() > 0) {
                            log.info("开始插入环境中备份的数据！");
                            statement = connection.createStatement();
                            for (String sql: backupDataSql.toString().split("\n")) {
                                statement.addBatch(sql);
                            }
                        }
                    } else {
                        // 不需要保留业务数据
                        // 表存在，则首先删除表
                        preparedStatement = connection.prepareStatement(dropTableSql);
                        preparedStatement.execute();

                        // 重新创建表
                        log.info("使用同步库中的表结构重新创建表: {}.{}", dbName, tableName);
                        preparedStatement = connection.prepareStatement(createTableSql);
                        preparedStatement.execute();

                        // 同步库中数据不为空时，插入同步库中的数据
                        if (!StringUtils.isEmpty(insertTableSql)) {
                            log.info("开始插入同步库中的数据！");
                            statement = connection.createStatement();
                            for (String sql: insertTableSql.split("\n")) {
                                statement.addBatch(sql);
                            }
                        }
                    }
                } else {
                    // 表不存在，创建表
                    log.info("使用同步库中的表结构创建表: {}.{}", dbName, tableName);
                    preparedStatement = connection.prepareStatement(createTableSql);
                    preparedStatement.execute();

                    // 同步库中数据不为空时，插入同步库中的数据
                    if (!StringUtils.isEmpty(insertTableSql)) {
                        log.info("开始插入同步库中的数据！");
                        statement = connection.createStatement();
                        for (String sql: insertTableSql.split("\n")) {
                            statement.addBatch(sql);
                        }
                    }
                }
            } else {
                // 环境中数据库不存在, 此时无论是否保留业务数据
                // 创建数据库
                log.info("环境中不存在数据库，首先创建数据库：{}", dbName);
                connectionWithoutDb = DriverManager.getConnection(urlWithoutDb, username, password);
                connectionWithoutDb.setAutoCommit(false);
                preparedStatement = connectionWithoutDb.prepareStatement(createDbSql);
                preparedStatement.execute();

                connection = DriverManager.getConnection(url, username, password);
                connection.setAutoCommit(false);

                // 创建表
                log.info("使用同步库中的表结构创建表: {}.{}", dbName, tableName);
                preparedStatement = connection.prepareStatement(createTableSql);
                preparedStatement.execute();

                // 同步库中数据不为空时，插入同步库中的数据
                if (!StringUtils.isEmpty(insertTableSql)) {
                    log.info("开始插入同步库中的数据！");
                    statement = connection.createStatement();
                    for (String sql: insertTableSql.split("\n")) {
                        statement.addBatch(sql);
                    }
                }
            }

            if (statement != null) {
                statement.executeBatch();
                statement.clearBatch();
            }
            connection.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.info("表{}.{}同步失败.", dbName, tableName);
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (connectionWithoutDb != null) {
                    connectionWithoutDb.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
                if (statement != null) {
                    statement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        log.info("表{}.{}同步完成.", dbName, tableName);
        return true;
    }

    /**
     * 获取数据库列表
     * @param ip 同步库ip
     * @param port 同步库端口号
     * @param username 同步库用户名
     * @param password 同步库密码
     * @return
     */
    @Override
    public List<String> getDatabaseList(String ip, String port, String username, String password) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port;

        List<String> databaseList = new ArrayList<>();
        Connection connection = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(true);

            String showDatabasesSql = "SHOW DATABASES";
            PreparedStatement preparedStatement = connection.prepareStatement(showDatabasesSql);
            ResultSet dbListResultSet = preparedStatement.executeQuery();
            while (dbListResultSet.next()) {
                databaseList.add(dbListResultSet.getString(1));
            }
            preparedStatement.close();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (Exception e) {
                connection = null;
            }
        }
        return databaseList;
    }

    /**
     * 获取某个数据库下表列表
     * @param ip mysql ip
     * @param port mysql port
     * @param username mysql 用户名
     * @param password mysql 密码
     * @param dbName 数据库名
     * @return
     */
    @Override
    public List<String> getTableListByDb(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        List<String> tableList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(false);

            String tablesSql = "SHOW TABLES";
            preparedStatement = connection.prepareStatement(tablesSql);
            ResultSet tableListResultSet = preparedStatement.executeQuery();
            while (tableListResultSet.next()) {
                tableList.add(tableListResultSet.getString(1));
            }
            return tableList;

        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            try {
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
                if (connection != null) {
                    connection.close();
                }
            } catch (Exception e) {
                connection = null;
            }
        }
    }

    /**
     * 从同步库获取表结构
     * @param ip
     * @param port
     * @param username
     * @param password
     * @param dbName
     * @param tableName
     * @return
     */
    @Override
    public boolean getSourceDbStructure(String ip, String port, String username, String password, String dbName, String tableName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        Connection connection = null;
        PreparedStatement preparedStatement = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(true);

            if ("all".equals(tableName)) {
                String tableListKey = dbSyncPrefix + dbName + ":tableList";
                List<String> tableList = getTableListByDb(ip, port, username, password, dbName);
                redisUtils.set(tableListKey, StringUtils.join(tableList, " "), 600);
                for (String table: tableList) {
                    String tableRedisKey = dbSyncPrefix + dbName + ":" + table + ":create";
                    if (!redisUtils.hasKey(tableRedisKey)) {
                        String sql = String.format("SHOW CREATE TABLE %s", table);
                        preparedStatement = connection.prepareStatement(sql);

                        ResultSet resultSet = preparedStatement.executeQuery();
                        while (resultSet.next()) {
                            redisUtils.set(tableRedisKey, resultSet.getString(2), 600);
                        }
                        resultSet.close();
                        preparedStatement.clearBatch();
                    }
                }
            } else {
                String tableRedisKey =  dbSyncPrefix + dbName + ":" + tableName + ":create";
                if (!redisUtils.hasKey(tableRedisKey)) {
                    String sql = String.format("SHOW CREATE TABLE %s", tableName);
                    preparedStatement = connection.prepareStatement(sql);
                    ResultSet resultSet = preparedStatement.executeQuery();

                    while (resultSet.next()) {
                        redisUtils.set(tableRedisKey, resultSet.getString(2), 600);
                    }
                    resultSet.close();
                }
                preparedStatement.clearBatch();
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 将同步库的表结构同步到环境中
     * @param ip
     * @param port
     * @param username
     * @param password
     * @param dbName
     * @param tableName
     * @return
     */
    @Override
    public boolean syncDbStructureToDest(String ip, String port, String username, String password, String dbName, String tableName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        Connection connection = null;
        Statement statement = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(true);
            statement = connection.createStatement();

            if ("all".equals(tableName)) {
                String tableListKey = dbSyncPrefix + dbName + ":tableList";
                List<String> tableList = Arrays.asList(redisUtils.get(tableListKey).toString().split(" "));
                // 删除key
                redisUtils.del(tableListKey);
                for (String table: tableList) {
                    // 首先删除表
                    String dropTableSql = "DROP TABLE IF EXISTS " + table;
                    statement.execute(dropTableSql);
                    statement.clearBatch();

                    // 然后同步表结构
                    String tableRedisKey = dbSyncPrefix + dbName + ":" + table + ":create";
                    String createTableSql = redisUtils.get(tableRedisKey).toString();
                    redisUtils.del(tableRedisKey);
                    statement.execute(createTableSql);
                    statement.clearBatch();
                }
            } else {
                // 首先删除表
                String dropTableSql = "DROP TABLE IF EXISTS " + tableName;
                statement.execute(dropTableSql);

                // 然后同步表结构
                String tableRedisKey = dbSyncPrefix + dbName + ":" + tableName + ":create";
                String createTableSql = redisUtils.get(tableRedisKey).toString();
                redisUtils.del(tableRedisKey);
                statement.execute(createTableSql);
                statement.clearBatch();
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (statement != null) {
                    statement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }


    /**
     * 获取表数据一行的所有值
     * @param rs
     * @param size
     * @param columnTypeList
     * @return
     */
    private String getRowValues(ResultSet rs, int size, List<String> columnTypeList) {
        try {
            StringBuilder rowValues = new StringBuilder();
            for (int i = 1; i <= size; i++) {
                String columnValue = null;

                // 获取字段值
                columnValue = getValue(rs, i, columnTypeList.get(i - 1));
                // 如果是空值不添加单引号
                if (null != columnValue) {
//                    if (columnValue.contains("'")) {
//                        columnValue = "\"" + columnValue + "\"";
//                    } else {
//                        columnValue = "'" + columnValue + "'";
//                    }
                    columnValue = columnValue.replace("'", "\\'");
                    columnValue = columnValue.replace("\"", "\\\"");
                    columnValue = "'" + columnValue + "'";
                }
                // 拼接字段值
                if (rowValues.length() == 0) {
                    rowValues.append(columnValue);
                } else {
                    rowValues.append(",").append(columnValue);
                }
            }

            return rowValues.toString();
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("获取表数据一行的所有值异常");
            return null;
        }
    }

    /**
     * 根据类型获取字段值
     * @return
     * @author  2018年9月6日 上午11:16:00
     */
    private String getValue(ResultSet resultSet, Integer index, String columnType) {
        try {
            if ("int".equals(columnType) || "INT".equals(columnType)) {
                // 整数
                Object intValue = resultSet.getObject(index);
                if (null == intValue) {
                    return null;
                }
                return intValue + "";
            } else if ("bigint".equals(columnType) || "BIGINT".equals(columnType)) {
                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ( "bigint unsigned".equals(columnType) || "BIGINT UNSIGNED".equals(columnType)) {
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("smallint".equals(columnType) || "SMALLINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("tinyint".equals(columnType) || "TINYINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("mediumint".equals(columnType) || "MEDIUMINT".equals(columnType)) {
                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("integer".equals(columnType) || "INTEGER".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("float".equals(columnType) || "FLOAT".equals(columnType)) {

                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("double".equals(columnType) || "DOUBLE".equals(columnType)) {
                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("decimal".equals(columnType) || "DECIMAL".equals(columnType)) {
                // 浮点数-金额类型
                BigDecimal value = resultSet.getBigDecimal(index);
                if (null == value) {
                    return null;
                }
                return value.toString();
            } else if ("char".equals(columnType) || "CHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("varchar".equals(columnType) || "VARCHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("tinytext".equals(columnType) || "TINYTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("text".equals(columnType) || "TEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("mediumtext".equals(columnType) || "MEDIUMTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("longtext".equals(columnType) || "LONGTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("year".equals(columnType) || "YEAR".equals(columnType)) {
                // 时间类型：范围 1901/2155 格式 YYYY
                String year = resultSet.getString(index);
                if (null == year) {
                    return null;
                }
                // 只需要年的字符即可，
                return year.substring(0, 4);
            } else if ("date".equals(columnType) || "DATE".equals(columnType)) {
                // 时间类型：范围 '1000-01-01'--'9999-12-31' 格式 YYYY-MM-DD
                return resultSet.getString(index);
            } else if ("time".equals(columnType) || "TIME".equals(columnType)) {
                // 时间类型：范围 '-838:59:59'到'838:59:59' 格式 HH:MM:SS
                return resultSet.getString(index);
            } else if ("datetime".equals(columnType) || "DATETIME".equals(columnType)) {
                // 时间类型：范围 '1000-01-01 00:00:00'--'9999-12-31 23:59:59' 格式 YYYY-MM-DD HH:MM:SS
                return resultSet.getString(index);
            } else if ("timestamp".equals(columnType) || "TIMESTAMP".equals(columnType)) {
                // 时间类型：范围 1970-01-01 00:00:00/2037 年某时 格式 YYYYMMDD HHMMSS 混合日期和时间值，时间戳
                return resultSet.getString(index);
            } else if ("bit".equals(columnType) || "BIT".equals(columnType)) {
                String value = resultSet.getString(index);
                if (value.equals("false")) {
                    return "0";
                } else if (value.equals("true")) {
                    return "1";
                } else {
                    return value;
                }
            } else {
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.err.println("获取数据库类型值异常");
            return "0000-00-00 00:00:00";
        }
    }

    private String replaceDomain(String sql, String namespace) {
        if (sql.contains("xyqb.com")) {
            return sql.replace(".xyqb.com", "-" + namespace + ".liangkebang.net");
        } else if (sql.contains(".q-gp.com")) {
            return sql.replace(".q-gp.com", "-" + namespace + ".liangkebang.net");
        } else {
            return sql;
        }
    }
}
