package cn.qg.holmes.task;

import cn.qg.holmes.entity.quality.DingRobot;
import cn.qg.holmes.entity.quality.JiraBugPool;
import cn.qg.holmes.service.quality.DingRobotService;
import cn.qg.holmes.service.quality.JiraBugPoolService;
import cn.qg.holmes.service.quality.JiraService;
import cn.qg.holmes.utils.DingdingUtils;
import com.atlassian.jira.rest.client.api.domain.Issue;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * JIRA BUG相关定时任务
 */
@Component
@Slf4j
public class BugNotifyTask {

    @Autowired
    JiraService jiraService;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    @Autowired
    DingRobotService dingRobotService;

    @Value("${sjgd.ding.url}")
    private String sjgdDingUrl;

    @Value("${task.debug}")
    private String taskDebug;

    /**
     * 数据工单bug统计
     * @throws Exception
     */
//    @Scheduled(cron = "0 0 19 * * ?")
    public void SJGDJiraIssueStatisticsTimedJob() throws Exception {
        if (taskDebug.equals("true")) {
            log.info("开始执行数据工单BUG统计定时任务！");
            Iterable<Issue> issues = jiraService.getUnsolvedIssueListByProject("SJGD");
            String markdownMsg = DingdingUtils.buildMarkdownMsgForUnsolvedIssueList(issues);
            DingdingUtils.sendToDingding(markdownMsg, sjgdDingUrl);
        } else {
            log.info("当前部署的是mock，不执行定时任务！");
        }
    }

    /**
     * bug池里BUG循环
     * @throws Exception
     */
//    @Scheduled(cron = "0 0 10-20 ? * MON-FRI")
    public void cycleJiraBugPool() throws Exception {
        if (taskDebug.equals("true")) {
            log.info("开始执行jira bug pool定时任务！");
            QueryWrapper<JiraBugPool> jiraBugPoolQueryWrapper = new QueryWrapper<>();
            jiraBugPoolQueryWrapper.eq("enable", 1);
            List<JiraBugPool> jiraBugPoolList = jiraBugPoolService.list(jiraBugPoolQueryWrapper);
            for (JiraBugPool jiraBugPool: jiraBugPoolList) {
                Issue issue = jiraService.getJiraIssueByKey(jiraBugPool.getKey());
                if (issue != null) {
                    String resolveResult = issue.getStatus().getName();
                    // 如果已关闭，仅修改状态
                    if (resolveResult.equals("关闭")) {
                        jiraBugPool.setEnable(0);
                        // 更新状态
                        jiraBugPoolService.saveOrUpdate(jiraBugPool);
                    } else if (resolveResult.equals("已解决")) {
                        // 暂时只打印日志，不做逻辑处理
                        log.info("BUG:{}已解决", issue.getKey());
                    } else {
                        DateTime startDate = issue.getCreationDate();
                        DateTime curDate = new DateTime();
                        int duration = jiraService.calculateBugFixTime(startDate, curDate);
                        log.info("{} BUG持续时间：{}小时", issue.getKey(), duration);
                        // 如果已超过4个小时，则发送钉钉通知
                        if (duration >= 4) {
                            DingRobot dingRobot = dingRobotService.getById(jiraBugPool.getDingRobotId());
                            // 项目存在，且项目进行中才发送通知
                            if (dingRobot != null) {
                                if (dingRobot.getStatus() == 1) {
                                    DingdingUtils.sendToDingding(DingdingUtils.buildBugFixRemindMsg(issue, duration), jiraBugPool.getDingUrl());
                                }
                            } else {
                                log.info("提测项目不存在，暂不发送钉钉通知！");
                            }
                        }
                    }
                } else {
                    log.info("JIRA上未找到该issue，将BUG: {} 置为无效", jiraBugPool.getKey());
                    jiraBugPool.setEnable(0);
                }
            }
        }
    }

    /**
     * 每晚6点循环处理jira bug pool里未解决的BUG
     */
//    @Scheduled(cron = "0 */1 * * * ?")
    @Scheduled(cron = "0 0 18 ? * MON-FRI")
    public void DailyCycleHandleJiraBugPool() {
        if (taskDebug.equals("true")) {
            // 首先遍历ding_robot, 获取状态是进行中的项目
            QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper<>();
            dingRobotQueryWrapper.eq("status", 1);
            List<DingRobot> dingRobotList = dingRobotService.list(dingRobotQueryWrapper);
            if (dingRobotList.size() > 0) {
                // 遍历进行中的项目，从jira_bug_pool里获取对应的状态是未解决的BUG
                for (DingRobot dingRobot: dingRobotList) {
                    QueryWrapper<JiraBugPool> jiraBugPoolQueryWrapper = new QueryWrapper<>();
                    jiraBugPoolQueryWrapper
                            .eq("ding_robot_id", dingRobot.getId())
                            .eq("enable", 1);
                    List<JiraBugPool> jiraBugPools = jiraBugPoolService.list(jiraBugPoolQueryWrapper);
                    List<Issue> issueList = new ArrayList<>();
                    for (JiraBugPool jiraBugPool: jiraBugPools) {
                        Issue issue = jiraService.getJiraIssueByKey(jiraBugPool.getKey());
                        if (issue != null) {
                            String resolveResult = issue.getStatus().getName();
                            // 如果已关闭，仅修改状态
                            if (resolveResult.equals("关闭")) {
                                jiraBugPool.setEnable(0);
                                // 更新状态
                                jiraBugPoolService.saveOrUpdate(jiraBugPool);
                            } else if (resolveResult.equals("已解决")) {
                                // 暂时只打印日志，不做逻辑处理
                                log.info("BUG:{}已解决", issue.getKey());
                            } else {
                                // 其他状态的直接加到通知列表
                                log.info("BUG:{}未解决，加到通知列表.", jiraBugPool.getKey());
                                issueList.add(issue);
                            }
                        } else {
                            log.info("JIRA上未找到该issue，将BUG: {} 置为无效", jiraBugPool.getKey());
                            jiraBugPool.setEnable(0);
                            jiraBugPoolService.saveOrUpdate(jiraBugPool);
                        }
                    }
                    // 判断列表是否为空，如果不为空，则忘群里发送通知
                    if (issueList.size() > 0) {
                        DingdingUtils.sendToDingding(DingdingUtils.buildDailyCycleJiraBugPoolMsg(issueList), dingRobot.getDingUrl());
                    }
                }
            }
        }
    }
}
