package cn.qg.holmes.controller.mock.keystone.jdbuy;

import cn.qg.holmes.bean.KeystoneService;
import cn.qg.holmes.entity.mock.keystone.ProductItem;
import cn.qg.holmes.entity.mock.keystone.jdbuy.*;
import cn.qg.holmes.utils.DateUtils;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.*;

@Slf4j
@RestController
public class OrderController {

    /**
     * 商品税率
     */
    private final BigDecimal TAX = new BigDecimal("0.13");
    /**
     * SKU分配，随便MOCK一个
     */
    private final Integer SKU_CATEGORY = 15924;
    @Autowired
    KeystoneService keystoneService;
    @Autowired
    RedisUtils redisUtils;
    private String QYG_ORDER_PREFIX = "qyg_order_";

    /**
     * 7.3 企业购提交订单
     *
     * @param submitOrderRequest 请求参数
     * @return
     */
    @PostMapping("/api/order/submitOrder")
    public JdSubmitOrderResponse qygSubmitOrder(SubmitOrderRequest submitOrderRequest) {
        log.info("收到企业购提交订单请求：{}", JSON.toJSONString(submitOrderRequest));
        JdSubmitOrderResponse response = new JdSubmitOrderResponse();
        JSONArray orderPriceSnapArray = JSON.parseArray(submitOrderRequest.getOrderPriceSnap());
        JSONArray skuInfoArray = JSON.parseArray(submitOrderRequest.getSku());
        List<SubmitOrderRequest.JdSkuPriceInfo> orderPriceSnaps = JSON.parseObject(orderPriceSnapArray.toJSONString(), new TypeReference<List<SubmitOrderRequest.JdSkuPriceInfo>>() {
        });
        List<SubmitOrderRequest.JdSkuInfo> skuInfos = JSON.parseObject(skuInfoArray.toJSONString(), new TypeReference<List<SubmitOrderRequest.JdSkuInfo>>() {
        });
        Map<String, BigDecimal> skuPriceMap = convertSkuPriceMap(orderPriceSnaps);
        // 计算订单总金额
        BigDecimal orderPrice = new BigDecimal(0);
        for (SubmitOrderRequest.JdSkuInfo jdSkuInfo : skuInfos) {
            BigDecimal skuNum = new BigDecimal(jdSkuInfo.getNum());
            String skuId = jdSkuInfo.getSkuId();
            BigDecimal price = skuPriceMap.get(skuId);
            orderPrice = orderPrice.add(price.multiply(skuNum));
        }
        response.setSuccess(true);
        response.setResultCode("0001");
        response.setResultMessage("下单成功");

        JdSubmitOrderResponse.JdSumbitOrderInfo jdSumbitOrderInfo = new JdSubmitOrderResponse.JdSumbitOrderInfo();
        String jdOrderId = RandomStringUtils.randomNumeric(12);
        jdSumbitOrderInfo.setJdOrderId(Long.valueOf(jdOrderId));
        jdSumbitOrderInfo.setFreight(calculateFreight(orderPrice));
        jdSumbitOrderInfo.setOrderPrice(orderPrice);
        BigDecimal orderNakedPrice = orderPrice.divide(TAX.add(new BigDecimal(1)), BigDecimal.ROUND_HALF_UP);
        jdSumbitOrderInfo.setOrderNakedPrice(orderNakedPrice);
        jdSumbitOrderInfo.setOrderTaxPrice(orderNakedPrice.multiply(TAX).setScale(2, BigDecimal.ROUND_HALF_UP));

        List<JdSubmitOrderResponse.JdBizSku> bizSkuList = new ArrayList<>();
        for (SubmitOrderRequest.JdSkuInfo sku : skuInfos) {
            JdSubmitOrderResponse.JdBizSku bizSku = new JdSubmitOrderResponse.JdBizSku();
            String skuId = sku.getSkuId();
            ProductItem productItem = keystoneService.getProductItemById(skuId, "jdbuy");
            bizSku.setSkuId(Long.valueOf(skuId));
            bizSku.setNum(sku.getNum());
            bizSku.setCategory(SKU_CATEGORY);
            BigDecimal price = skuPriceMap.get(skuId);
            bizSku.setPrice(price);
            bizSku.setName(productItem.getName());
            bizSku.setTax(TAX.multiply(new BigDecimal(102)));
            BigDecimal nakedPrice = price.divide(TAX.add(new BigDecimal(1)), BigDecimal.ROUND_HALF_UP);
            bizSku.setNakedPrice(nakedPrice);
            bizSku.setTaxPrice(price.subtract(nakedPrice));
            bizSku.setType(0);
            bizSku.setOid(0L);

            bizSkuList.add(bizSku);
        }
        jdSumbitOrderInfo.setSku(bizSkuList);

        response.setResult(jdSumbitOrderInfo);
        // 设置缓存，方便后面查询京东订单详情接口使用
        redisUtils.set(QYG_ORDER_PREFIX + jdOrderId, JSON.toJSONString(response), 604800);
        return response;
    }

    /**
     * 7.5 确认预占库存订单
     *
     * @return
     */
    @PostMapping("/api/order/confirmOrder")
    public JSONObject qygConfirmOrder(JdConfirmOrderRequest request) {
        JSONObject response = new JSONObject();
        response.put("success", true);
        response.put("resultMessage", "确认下单成功");
        response.put("resultCode", "0003");
        response.put("result", true);
        return response;
    }

    /**
     * 7.6 取消未确认订单
     */
    @PostMapping("/api/order/cancel")
    public JSONObject qygOrderCancel(JdCancelOrderRequest request) {
        JSONObject resposne = new JSONObject();
        resposne.put("success", true);
        resposne.put("resultMessage", "取消订单成功");
        resposne.put("resultCode", "0002");
        resposne.put("result", true);
        return resposne;
    }

    /**
     * 7.7 查询订单详情
     */
    @PostMapping("/api/order/selectJdOrder")
    public JdOrderDetailResponse qygOrderDetail(JdOrderDetailRequest request) {
        String jdOrderId = request.getJdOrderId();
        String queryExts = request.getQueryExts();
        String jdOrderInfoRedisValue = (String) redisUtils.get(QYG_ORDER_PREFIX + jdOrderId);
        JdSubmitOrderResponse orderInfo = null;
        if (!StringUtils.isEmpty(jdOrderInfoRedisValue)) {
            orderInfo = JSONObject.parseObject(jdOrderInfoRedisValue, JdSubmitOrderResponse.class);
        }
        JdOrderDetailResponse response = new JdOrderDetailResponse();
        response.setSuccess(true);
        response.setResultMessage("");
        response.setResultCode("0000");

        JSONObject result = new JSONObject();
        result.put("pOrder", 0);
        result.put("orderState", 1);
        result.put("jdOrderId", Long.valueOf(jdOrderId));
        result.put("state", 1);
        result.put("submitState", 1);
        result.put("type", 2); // 1-父订单，2-子订单
        if (orderInfo != null) {
            result.put("freight", orderInfo.getResult().getFreight());
            result.put("orderPrice", orderInfo.getResult().getOrderPrice());
            result.put("orderNakedPrice", orderInfo.getResult().getOrderNakedPrice());
            result.put("orderTaxPrice", orderInfo.getResult().getOrderTaxPrice());
            result.put("sku", orderInfo.getResult().getSku());
        } else {
            // 如果缓存里取不到值，就随便给个值
            result.put("freight", 0);
            result.put("orderPrice", 10);
            result.put("orderNakedPrice", 8);
            result.put("orderTaxPrice", 2);
            result.put("sku", new JSONObject());
        }
        if (queryExts != null && queryExts.contains("orderType")) {
            result.put("orderType", 1);
        }
        if (queryExts != null && queryExts.contains("createOrderTime")) {
            result.put("createOrderTime", DateUtils.convertDate(new Date(), "yyyy-MM-dd hh:mm:ss"));
        }
        if (queryExts != null && queryExts.contains("finishTime")) {
            result.put("finishTime", DateUtils.convertDate(new Date(), "yyyy-MM-dd hh:mm:ss"));
        }
        if (queryExts != null && queryExts.contains("jdOrderState")) {
            // 16-等待确认收货
            result.put("jdOrderState", 16);
        }
        result.put("paymentType", 4);

        response.setResult(result);
        return response;
    }

    /**
     * 7.8 查询配送信息
     */
    @PostMapping("/api/order/orderTrack")
    public JdOrderTrackResponse getOrderTrack(JdOrderTrackRequest request) throws ParseException {
        String jdOrderId = request.getJdOrderId();
        JdOrderTrackResponse response = new JdOrderTrackResponse();
        response.setSuccess(true);
        response.setResultMessage("");
        response.setResultCode("0000");

        JdOrderTrackResponse.OrderTrackInfo orderTrackInfo = new JdOrderTrackResponse.OrderTrackInfo();
        List<JdOrderTrackResponse.OrderTrack> orderTrackList = new ArrayList<>();
        List<JdOrderTrackResponse.WaybillCode> waybillCodeList = new ArrayList<>();

        JdOrderTrackResponse.OrderTrack orderTrack1 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack2 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack3 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack4 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack5 = new JdOrderTrackResponse.OrderTrack();

        JdOrderTrackResponse.WaybillCode waybillCode = new JdOrderTrackResponse.WaybillCode();

        orderTrack1.setContent("您提交了订单，请等待系统确认");
        orderTrack1.setMsgTime(DateUtils.convertStr("2022-03-01 23:27:13", "yyyy-MM-dd hh:mm:ss"));
        orderTrack1.setOperator("客户");

        orderTrack2.setContent("温馨提示：您的订单将由京东指定厂家为您发货，请您耐心等待");
        orderTrack2.setMsgTime(DateUtils.convertStr("2022-03-01 23:27:45", "yyyy-MM-dd hh:mm:ss"));
        orderTrack2.setOperator("系统");

        orderTrack3.setContent("您的订单正在出库，请耐心等待");
        orderTrack3.setMsgTime(DateUtils.convertStr("2022-03-02 15:22:34", "yyyy-MM-dd hh:mm:ss"));
        orderTrack3.setOperator("系统");

        orderTrack4.setContent("您的订单已交付京东快递, 运单号为JDVC12992534651");
        orderTrack4.setMsgTime(DateUtils.convertStr("2022-03-02 15:25:01", "yyyy-MM-dd hh:mm:ss"));
        orderTrack4.setOperator("系统");

        orderTrack5.setContent("您的订单开始配送");
        orderTrack5.setMsgTime(DateUtils.convertStr("2022-03-03 10:22:34", "yyyy-MM-dd hh:mm:ss"));
        orderTrack5.setOperator("京东快递");

        orderTrackList.add(orderTrack1);
        orderTrackList.add(orderTrack2);
        orderTrackList.add(orderTrack3);
        orderTrackList.add(orderTrack4);
        orderTrackList.add(orderTrack5);

        waybillCode.setOrderId(Long.valueOf(jdOrderId));
        waybillCode.setCarrier("京东快递");
        waybillCode.setDeliveryOrderId("JDVD02626228685");
        waybillCode.setParentId(0L);
        waybillCodeList.add(waybillCode);

        orderTrackInfo.setJdOrderId(jdOrderId);
        orderTrackInfo.setOrderTrack(orderTrackList);
        orderTrackInfo.setWaybillCode(waybillCodeList);

        response.setResult(orderTrackInfo);
        return response;
    }

    /**
     * 11.1 查询推送消息
     *
     * @return
     */
    @PostMapping("/api/message/get")
    public JSONObject qygMessageGet(JdCompanyMsgRequest request) {
        String type = request.getType();
        JSONObject response = new JSONObject();
        response.put("success", true);
        response.put("resultMessage", "");
        response.put("resultCode", "0000");
        JSONArray result = new JSONArray();
        switch (type) {
            case "10":
                // TODO
                break;
            case "31":
                // TODO
                break;
            default:
                break;
        }
        response.put("result", result);
        return response;
    }


    /**
     * 列表转Map，key为skuId，value是price
     *
     * @param orderPriceSnaps JdSkuPriceInfo列表
     * @return
     */
    public Map<String, BigDecimal> convertSkuPriceMap(List<SubmitOrderRequest.JdSkuPriceInfo> orderPriceSnaps) {
        Map<String, BigDecimal> map = new HashMap<>();
        for (SubmitOrderRequest.JdSkuPriceInfo jdSkuPriceInfo : orderPriceSnaps) {
            map.put(jdSkuPriceInfo.getSkuId(), jdSkuPriceInfo.getPrice());
        }
        return map;
    }

    /**
     * 计算运费
     *
     * @param totalPrice 总价格
     * @return
     */
    public BigDecimal calculateFreight(BigDecimal totalPrice) {
        if (totalPrice.compareTo(new BigDecimal("99")) >= 0) {
            return new BigDecimal("0");
        } else {
            return new BigDecimal("6");
        }
    }
}
