package cn.qg.holmes.service.k8s;

import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import io.fabric8.kubernetes.api.model.*;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.apps.DeploymentSpec;
import io.fabric8.kubernetes.api.model.apps.DeploymentStatus;
import io.fabric8.kubernetes.api.model.apps.DeploymentStrategy;
import io.fabric8.kubernetes.api.model.extensions.*;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.*;

@Slf4j
@Component
public class K8sService {

    @Autowired
    RedisUtils redisUtils;

    @Value("${no.healthcheck.service}")
    private String noHealthCheckService;

    private KubernetesClient kubernetesClient;

    public K8sService() {
        try {
            String configYAML = String.join("\n", readConfigFile("kube-config.yml"));
            Config config = Config.fromKubeconfig(configYAML);
            kubernetesClient = new DefaultKubernetesClient(config);
            String configCrt = String.join("\n", readConfigFile("tke-cluster-ca.crt"));
            config.setCaCertData(configCrt);
            log.info("k8s连接初始化成功！");
        } catch (Exception e) {
            log.info("k8s service 初始化失败！");
            e.printStackTrace();
        }
    }

    /**
     * 读取k8s配置文件
     * @param file 配置文件名
     * @return
     * @throws IOException
     */
    private static List<String> readConfigFile(String file) throws IOException {
        String str = "";
        ClassPathResource resource = new ClassPathResource(file);
        InputStream in = resource.getInputStream();
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        List<String> result = new ArrayList<>();
        while ((str = br.readLine()) != null) {
            result.add(str);
        }
        return result;
    }

    /**
     * 构造ObjectMeta对象
     * @param namespace   环境
     * @param serviceName 服务名称
     * @param type        类型，java、ui、base等等
     * @return
     */
    public ObjectMeta buildObjectMeta(String namespace, String serviceName, String type) {
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("type", type);
        labels.put("qcloud-app", serviceName);
        objectMeta.setLabels(labels);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        return objectMeta;
    }

    public ResourceRequirements buildResourceRequirements(String cpuRequestAmount, String cpuRequestFormat, String memRequestAmount, String memRequestFormat,
                                                          String cpuLimitAmount, String cpuLimitFormat, String memLimitAmount, String memLimitFormat) {
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount(cpuRequestAmount);
        cpuQuantity.setFormat(cpuRequestFormat);
        memoryQuantity.setAmount(memRequestAmount);
        memoryQuantity.setFormat(memRequestFormat);
        cpuLimit.setAmount(cpuLimitAmount);
        cpuLimit.setFormat(cpuLimitFormat);
        memoryLimit.setAmount(memLimitAmount);
        memoryLimit.setFormat(memLimitFormat);
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        return resourceRequirements;
    }

    /**
     * 获取namespace列表
     * @return
     */
    public List<Map> getNamespaceList(String env) {
        // 首先从缓存获取
        String NS_PREFIX = "k8s:namespace:";
        Object redisValue = redisUtils.get(NS_PREFIX + env);
        if (redisValue != null) {
//            log.info("从缓存获取到{}环境namespace列表：{}", env, redisValue);
            return JSON.parseArray(redisValue.toString(), Map.class);
        } else {
            List<Map> resultList = new ArrayList<>();
            List<Namespace> namespaceList = kubernetesClient.namespaces().list().getItems();
            for (Namespace namespace : namespaceList) {
                if (namespace.getMetadata().getAnnotations() != null && env.equals("all")) {
                    String name = namespace.getMetadata().getName();
                    // 获取Service
                    Service service = kubernetesClient.services().inNamespace(name).withName("mysql").get();
                    // 获取Pod
                    Map<String, String> labels = new HashMap<>();
                    labels.put("qcloud-app", "mysql");
                    labels.put("type", "base");
                    List<Pod> podList = kubernetesClient.pods().inNamespace(name).withLabels(labels).list().getItems();
                    Integer port = null;
                    String host = null;
                    if (service != null) {
                        port = service.getSpec().getPorts().get(0).getNodePort();
                    }
                    if (podList.size() == 1) {
                        host = podList.get(0).getStatus().getHostIP();
                    }
                    if (podList.size() >= 2) {
                        for (Pod pod: podList) {
                            if (pod.getStatus().getHostIP() != null) {
                                host = pod.getStatus().getHostIP();
                                break;
                            }
                        }
                    }
                    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
                    df.setTimeZone(TimeZone.getTimeZone("UTC"));
                    SimpleDateFormat df2 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    Map map = new HashMap<>();
                    map.put("name", name);
                    map.put("description", namespace.getMetadata().getAnnotations().get("description"));
                    map.put("owner", namespace.getMetadata().getAnnotations().get("owner"));
                    map.put("status", namespace.getStatus().getPhase());
                    map.put("mysqlHost", host);
                    map.put("mysqlPort", port);
                    try {
                        map.put("createdAt", df2.format(df.parse(namespace.getMetadata().getCreationTimestamp())));
                    } catch (Exception e) {
                        log.info("时间解析异常！");
                        e.printStackTrace();
                        map.put("createdAt", namespace.getMetadata().getCreationTimestamp());
                    }
                    resultList.add(map);
                } else if (namespace.getMetadata().getAnnotations() != null && namespace.getMetadata().getAnnotations().get("description").equals(env)) {
                    String name = namespace.getMetadata().getName();
                    // 获取Service
                    Service service = kubernetesClient.services().inNamespace(name).withName("mysql").get();
                    // 获取Pod
                    Map<String, String> labels = new HashMap<>();
                    labels.put("qcloud-app", "mysql");
                    labels.put("type", "base");
                    List<Pod> podList = kubernetesClient.pods().inNamespace(name).withLabels(labels).list().getItems();
                    Integer port = null;
                    String host = null;
                    if (service != null) {
                        port = service.getSpec().getPorts().get(0).getNodePort();
                    }
                    if (podList.size() == 1) {
                        host = podList.get(0).getStatus().getHostIP();
                    }
                    if (podList.size() >= 2) {
                        for (Pod pod: podList) {
                            if (pod.getStatus().getHostIP() != null) {
                                host = pod.getStatus().getHostIP();
                                break;
                            }
                        }
                    }
                    SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
                    df.setTimeZone(TimeZone.getTimeZone("UTC"));
                    SimpleDateFormat df2 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                    Map<String, Object> map = new HashMap<>();
                    map.put("name", name);
                    map.put("description", namespace.getMetadata().getAnnotations().get("description"));
                    map.put("owner", namespace.getMetadata().getAnnotations().get("owner"));
                    map.put("status", namespace.getStatus().getPhase());
                    map.put("mysqlHost", host);
                    map.put("mysqlPort", port);
                    try {
                        map.put("createdAt", df2.format(df.parse(namespace.getMetadata().getCreationTimestamp())));
                    } catch (Exception e) {
                        log.info("时间解析异常！");
                        e.printStackTrace();
                        map.put("createdAt", namespace.getMetadata().getCreationTimestamp());
                    }
                    resultList.add(map);
                }
            }
//            log.info("获取{}环境namespace列表：{}", env, JSON.toJSONString(resultList));
            // 设置缓存
            redisUtils.set(NS_PREFIX + env, JSON.toJSONString(resultList), 1800);
            return resultList;
        }
    }

    /**
     * 获取运行中的pod列表
     * @param namespace 环境
     * @return
     */
    public Map<String, String> getMysqlAddressByNamespace(String namespace) {
        // 获取Service
        Service service = kubernetesClient.services().inNamespace(namespace).withName("mysql").get();
        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "mysql");
        labels.put("type", "base");
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).withLabels(labels).list().getItems();
        String port = null;
        String host = null;
        if (service != null) {
            port = String.valueOf(service.getSpec().getPorts().get(0).getNodePort());
        }
        if (podList.size() == 1) {
            host = podList.get(0).getStatus().getHostIP();
        }
        if (podList.size() >= 2) {
            for (Pod pod: podList) {
                if (pod.getStatus().getHostIP() != null) {
                    host = pod.getStatus().getHostIP();
                    break;
                }
            }
        }
        Map<String, String> resultMap = new HashMap<>();
        resultMap.put("host", host);
        resultMap.put("port", port);
        return resultMap;
    }


    public List<Map<String, Object>> getPodListNew(String namespace) {
        // 获取Pod列表
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).list().getItems();
        // 获取Service列表
        List<Service> serviceList = kubernetesClient.services().inNamespace(namespace).list().getItems();
        // 获取ingress列表
        List<Ingress> ingressList = kubernetesClient.extensions().ingresses().inNamespace(namespace).list().getItems();
        Map<String, Map<String, Object>> podServiceMap = new HashMap<>();
        // 遍历Pod列表
        for (Pod pod: podList) {
            if (!pod.getStatus().getPhase().equals("Failed")) {
                String serviceName = pod.getMetadata().getLabels().get("qcloud-app");
                podServiceMap.put(serviceName, formatPodInfo(pod));
            }
        }
        // 遍历Service列表
        for (Service service: serviceList) {
            String serviceName = service.getMetadata().getName();
            podServiceMap = assign(serviceName, podServiceMap, formatServiceInfo(service));
        }
        // 遍历Ingress列表
        for (Ingress ingress: ingressList) {
            String serviceName = ingress.getMetadata().getName();
            podServiceMap = assign(serviceName, podServiceMap, formatIngressInfo(ingress));
        }
        // 对象转列表
        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Map<String, Object> map: podServiceMap.values()) {
            if ((map != null) && (map.get("serviceName") != null))  {
                resultList.add(map);
            }
        }
        // 根据serviceName对列表进行排序
        Collections.sort(resultList, (m1, m2) -> {
           String serviceName1 = m1.get("serviceName").toString();
           String serviceName2 = m2.get("serviceName").toString();
            return serviceName1.compareTo(serviceName2);
        });
        return resultList;
    }

    public Map<String, Map<String, Object>> assign(String serviceName, Map<String, Map<String, Object>> podServiceMap, Map<String, Object> otherMap) {
        Map<String, Object> podService = podServiceMap.get(serviceName);
        for (Map.Entry<String, Object> entry: otherMap.entrySet()) {
            if (podService != null) {
                podService.put(entry.getKey(), entry.getValue());
            }
        }
        podServiceMap.put(serviceName, podService);
        return podServiceMap;
    }

    /**
     * 处理pod相关信息
     * @param pod
     * @return
     */
    public Map<String, Object> formatPodInfo(Pod pod) {
        Map<String, Object> podMap = new HashMap<>();
        podMap.put("serviceName", pod.getMetadata().getLabels().get("qcloud-app"));
        podMap.put("podName", pod.getMetadata().getName());
        podMap.put("status", getPodStatus(pod));
        podMap.put("podIp", pod.getStatus().getPodIP());
        podMap.put("lanIp", pod.getStatus().getHostIP());
//        podMap.put("startTime", pod.getStatus().getStartTime());
        // 特殊处理，格式化创建时间
        try {
            if (pod.getStatus().getStartTime() != null) {
                SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
                df.setTimeZone(TimeZone.getTimeZone("UTC"));
                podMap.put("createdAt", df.parse(pod.getStatus().getStartTime()));
            } else {
                SimpleDateFormat df2 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                podMap.put("createdAt", df2.format(new Date()));
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.info("时间转换异常！");
            podMap.put("createdAt", pod.getStatus().getStartTime());
        }
        if (pod.getStatus().getContainerStatuses().size() > 0) {
            podMap.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
            podMap.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
        } else {
            podMap.put("image", null);
            podMap.put("imageId", null);
        }
        if (pod.getMetadata().getLabels() != null) {
            podMap.put("labels", pod.getMetadata().getLabels());
        }
        return podMap;
    }

    /**
     * 格式化pod状态
     * @param pod
     * @return
     */
    public String getPodStatus(Pod pod) {
        if (pod.getMetadata().getDeletionTimestamp() != null) {
            return  "Terminating";
        } else {
            String status = "";
            List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
            if (containerStatuses.size() == 0) {
                status = "Pending";
            } else {
                if (containerStatuses.get(0).getState().getWaiting() != null) {
                    status = "ContainerCreating";
                }
                if (containerStatuses.get(0).getState().getRunning() != null && !containerStatuses.get(0).getReady()) {
                    status = "Waiting";
                }
                if (containerStatuses.get(0).getState().getRunning() != null && containerStatuses.get(0).getReady()) {
                    status = "Normal";
                }
            }
            return status;
        }
    }

    public Map<String, Object> formatServiceInfo(Service service) {
        Map<String, Object> serviceMap = new HashMap<>();
        serviceMap.put("serviceType", service.getSpec().getType());
        List<Map<String, Object>> portMappingList = new ArrayList<>();
        List<ServicePort> servicePortList = service.getSpec().getPorts();
        if (servicePortList.size() > 0) {
            for (ServicePort servicePort : servicePortList) {
                if (servicePort.getNodePort() != null) {
                    serviceMap.put("port_" + servicePort.getName(), servicePort.getNodePort());
                }
                Map<String, Object> portMap = new HashMap<>();
                portMap.put("name", servicePort.getName());
                portMap.put("nodePort", servicePort.getNodePort());
                portMap.put("port", servicePort.getPort());
                portMap.put("protocol", servicePort.getProtocol());
                portMap.put("targetPort", servicePort.getTargetPort());
                portMappingList.add(portMap);
            }
        }
        if (portMappingList.size() > 0) {
            serviceMap.put("portMappings", portMappingList);
        }
        return serviceMap;
    }

    public Map<String, Object> formatIngressInfo(Ingress ingress) {
        Map<String, Object> ingressMap = new HashMap<>();
        if (ingress.getSpec().getRules() != null && ingress.getSpec().getRules().size() > 0) {
            ingressMap.put("host", ingress.getSpec().getRules().get(0).getHost());
        } else {
            ingressMap.put("host", "");
        }
        return ingressMap;
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName   podName
     * @return
     */
    public boolean resetPod(String namespace, String podName) {
        return kubernetesClient.pods().inNamespace(namespace).withName(podName).delete();
    }

    /**
     * 删除一个pod
     * @param namespace   环境
     * @param serviceName 服务名
     * @return
     */
    public boolean deleteDeployment(String namespace, String serviceName) {
        // 删除Service
//        log.info("删除{}环境{}的Service.", namespace, serviceName);
//        kubernetesClient.services().inNamespace(namespace).withName(serviceName).delete();
//        // 删除Ingress
//        log.info("删除{}环境{}的Ingress.", namespace, serviceName);
//        kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).delete();
        // 删除eployment
        log.info("删除{}环境{}的pod.", namespace, serviceName);
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).delete();
    }

    /**
     * 部署Java服务
     * @return
     */
    public Service createJavaService(String namespace, String serviceName, Integer debug) {
        Service service = new Service();
        ObjectMeta objectMeta = new ObjectMeta();
        ServiceSpec serviceSpec = new ServiceSpec();
        // 设置meta
        Map<String, String> labels = new HashMap<>();
        labels.put("type", "java");
        labels.put("qcloud-app", serviceName);

        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labels);

        // 设置spec
        List<ServicePort> portList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setName("80");
        servicePort1.setPort(80);
        ServicePort servicePort2 = new ServicePort();
        servicePort2.setName("5005");
        servicePort2.setPort(5005);
        ServicePort servicePort3 = new ServicePort();
        servicePort3.setName("20880");
        servicePort3.setPort(20880);
        portList.add(servicePort1);
        portList.add(servicePort2);
        portList.add(servicePort3);
        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", serviceName);

        // 设置serviceSpec
        if (debug == 1) {
            serviceSpec.setType("NodePort");
        } else {
            serviceSpec.setType("ClusterIP");
        }
        serviceSpec.setPorts(portList);
        serviceSpec.setSelector(selector);

        // 设置service
        service.setApiVersion("v1");
        service.setKind("Service");
        service.setMetadata(objectMeta);
        service.setSpec(serviceSpec);
        log.info("开始创建Java Service: {}", service.toString());
        return kubernetesClient.services().inNamespace(namespace).create(service);
    }

    /**
     * 部署Java Deployment
     * @param serviceCreateVo
     * @param dockerProject
     * @return
     */
    public Deployment createJavaDeployment(ServiceCreateVo serviceCreateVo, DockerProject dockerProject) {
        String namespace = serviceCreateVo.getNamespace();
        String serviceName = serviceCreateVo.getServiceName();
        String mock = serviceCreateVo.getMock().toString();
        String debug = serviceCreateVo.getDebug().toString();
        String cluster = serviceCreateVo.getCluster();
        String label = serviceCreateVo.getLabel();
        String image = serviceCreateVo.getImage();
        Deployment deployment = new Deployment();
        ObjectMeta objectMeta = new ObjectMeta();
        DeploymentSpec deploymentSpec = new DeploymentSpec();

        // 设置metadata
        Map<String, String> labels = new HashMap<>();
        labels.put("type", label);
        labels.put("qcloud-app", serviceName);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labels);

        // 设置labelSelector
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", serviceName);
        labelSelector.setMatchLabels(matchLabels);

        // 设置strategy
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // 设置pod Template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        ObjectMeta templateObjectMeta = new ObjectMeta();
        Map<String, String> templateLabels = new HashMap<>();
        templateLabels.put("qcloud-app", serviceName);
        templateLabels.put("type", label);
        templateLabels.put("mock", mock);
        templateObjectMeta.setLabels(templateLabels);

        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setName(serviceName);
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("IfNotPresent");

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("filelog");
        volumeMount.setMountPath("/home/logs");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("NAMESPACE");
        envVar1.setValue(namespace);
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("NAMESPACECLUSTER");
        envVar2.setValue(namespace + "@" + cluster);
        EnvVar envVar3 = new EnvVar();
        envVar3.setName("CLUSTER");
        envVar3.setValue(cluster);
        EnvVar envVar4 = new EnvVar();
        envVar4.setName("SYSTEM_NAME");
        envVar4.setValue(serviceName);
        EnvVar envVar5 = new EnvVar();
        envVar5.setName("HOSTS");
        envVar5.setValue("192.168.4.50=>eos.quantgroups.com&172.17.1.7=>apollo-dev.quantgroups.com&172.20.3.11=>sentry.quantgroups.com&");
        EnvVar envVar6 = new EnvVar();
        envVar6.setName("DEBUG");
        envVar6.setValue(debug);
        EnvVar envVar7 = new EnvVar();
        envVar7.setName("MOCK");
        envVar7.setValue(mock);
        EnvVar envVar8 = new EnvVar();
        EnvVarSource envVarSource = new EnvVarSource();
        ObjectFieldSelector objectFieldSelector = new ObjectFieldSelector();
        objectFieldSelector.setFieldPath("status.hostIP");
        envVarSource.setFieldRef(objectFieldSelector);
        envVar8.setName("DUBBO_IP_TO_REGISTRY");
        envVar8.setValueFrom(envVarSource);
        envVarList.add(envVar1);
        envVarList.add(envVar2);
        envVarList.add(envVar3);
        envVarList.add(envVar4);
        envVarList.add(envVar5);
        envVarList.add(envVar6);
        envVarList.add(envVar7);
        envVarList.add(envVar8);
        container.setEnv(envVarList);

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount(dockerProject.getCpuRequest());
        cpuQuantity.setFormat("m");
        memoryQuantity.setAmount(dockerProject.getMemRequest());
        memoryQuantity.setFormat("Mi");
        cpuLimit.setAmount(dockerProject.getCpuLimit());
        cpuLimit.setFormat("m");
        memoryLimit.setAmount(dockerProject.getMemLimit());
        memoryLimit.setFormat("Mi");
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        limits.put("cpu", cpuLimit);
        limits.put("memory", memoryLimit);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // 需要做健康检查的不创建存活检查和就绪检查
        List<String> noHealthCheckServiceList = Arrays.asList(noHealthCheckService.split(","));
        if (!noHealthCheckServiceList.contains(serviceName)) {
            // livenessProbe
            Probe livenessProbe = new Probe();
            ExecAction execAction = new ExecAction();
            List<String> commandList = new ArrayList<>();
            commandList.add("/home/quant_group/readyCheck.sh");
            execAction.setCommand(commandList);
            livenessProbe.setExec(execAction);
            livenessProbe.setInitialDelaySeconds(200);
            livenessProbe.setSuccessThreshold(1);
            livenessProbe.setFailureThreshold(40);
            container.setLivenessProbe(livenessProbe);

            //readinessProbe
            Probe readinessProbe = new Probe();
            readinessProbe.setExec(execAction);
            readinessProbe.setInitialDelaySeconds(200);
            readinessProbe.setTimeoutSeconds(2);
            readinessProbe.setPeriodSeconds(40);
            container.setReadinessProbe(readinessProbe);
        }

        containerList.add(container);

        // volumes
        List<Volume> volumeList = new ArrayList<>();
        Volume volume = new Volume();
        HostPathVolumeSource hostPathVolumeSource = new HostPathVolumeSource();
        hostPathVolumeSource.setPath("/var/log/containers2/" + namespace + "/" + serviceName);
        volume.setName("filelog");
        volume.setHostPath(hostPathVolumeSource);
        volumeList.add(volume);

        // imagePullSecrets
        List<LocalObjectReference> referenceList = new ArrayList<>();
        LocalObjectReference reference1 = new LocalObjectReference();
        LocalObjectReference reference2 = new LocalObjectReference();
        reference1.setName("qcloudregistrykey");
        reference2.setName("tencenthubkey");
        referenceList.add(reference1);
        referenceList.add(reference2);

        // 设置podSpec
        podSpec.setTerminationGracePeriodSeconds(0L);
        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumeList);
        podSpec.setRestartPolicy("Always");
        podSpec.setDnsPolicy("ClusterFirst");
        podSpec.setImagePullSecrets(referenceList);

        // 设置PodTemplateSpec
        podTemplateSpec.setMetadata(templateObjectMeta);
        podTemplateSpec.setSpec(podSpec);

        // 设置Deployment Spec
        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        // deployment设置
        deployment.setApiVersion("apps/v1");
        deployment.setKind("Deployment");
        deployment.setMetadata(objectMeta);
        deployment.setSpec(deploymentSpec);

        log.info("开始创建Java Deployment：{}", deployment.toString());
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(deployment);
    }

    /**
     * 创建UI或Node 服务
     * @param namespace   环境
     * @param serviceName 服务名
     * @param serviceType 服务类型
     * @param label
     * @return
     */
    public Service createUIAndNodeService(String namespace, String serviceName, String serviceType, String label) {
        Service uiService = new Service();
        ObjectMeta objectMeta = new ObjectMeta();
        ServiceSpec serviceSpec = new ServiceSpec();
        // 设置meta
        Map<String, String> labels = new HashMap<>();
        labels.put("type", label);
        labels.put("qcloud-app", serviceName);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labels);
        // 设置spec
        List<ServicePort> portList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setName("80");
        servicePort1.setPort(80);
        portList.add(servicePort1);
        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", serviceName);

        // 设置serviceSpec
        serviceSpec.setType("ClusterIP");
        serviceSpec.setPorts(portList);
        serviceSpec.setSelector(selector);

        // 设置service
        uiService.setApiVersion("v1");
        uiService.setKind("Service");
        uiService.setMetadata(objectMeta);
        uiService.setSpec(serviceSpec);
        log.info("开始创建UI/Node Service：{}", uiService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(uiService);
    }

    /**
     * 部署UI或Node deployment
     * @param serviceCreateVo
     * @param dockerProject
     * @return
     */
    public Deployment createUIAndNodeDeployment(ServiceCreateVo serviceCreateVo, DockerProject dockerProject) {
        String serviceName = serviceCreateVo.getServiceName();
        String namespace = serviceCreateVo.getNamespace();
        String label = serviceCreateVo.getLabel();
        String image = serviceCreateVo.getImage();
        String cluster = serviceCreateVo.getCluster();
        String debug = serviceCreateVo.getDebug().toString();

        Deployment deployment = new Deployment();

        // 设置meta
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labelMap = new HashMap<>();
        labelMap.put("type", label);
        labelMap.put("qcloud-app", serviceName);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labelMap);

        // 设置spec
        DeploymentSpec deploymentSpec = new DeploymentSpec();

        // 设置labelSelector
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", serviceName);
        labelSelector.setMatchLabels(matchLabels);

        // 设置strategy
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // 设置pod Template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        ObjectMeta templateObjectMeta = new ObjectMeta();
        Map<String, String> templateLabels = new HashMap<>();
        templateLabels.put("qcloud-app", serviceName);
        templateLabels.put("type", label);
        templateObjectMeta.setLabels(templateLabels);

        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setName(serviceName);
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("IfNotPresent");

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("filelog");
        volumeMount.setMountPath("/home/logs");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("NAMESPACE");
        envVar1.setValue(namespace);
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("SYSTEM_NAME");
        envVar2.setValue(serviceName);
        EnvVar envVar3 = new EnvVar();
        envVar3.setName("CLUSTER");
        envVar3.setValue(cluster);
        EnvVar envVar4 = new EnvVar();
        envVar4.setName("NODE_ENV");
        envVar4.setValue("test");
        EnvVar envVar5 = new EnvVar();
        envVar5.setName("HOSTS");
        envVar5.setValue("192.168.4.50=>eos.quantgroups.com&172.17.1.7=>apollo-dev.quantgroups.com&172.20.3.11=>sentry.quantgroups.com&");
        EnvVar envVar6 = new EnvVar();
        envVar6.setName("DEBUG");
        envVar6.setValue(debug);
        envVarList.add(envVar1);
        envVarList.add(envVar2);
        envVarList.add(envVar3);
        envVarList.add(envVar4);
        envVarList.add(envVar5);
        envVarList.add(envVar6);
        container.setEnv(envVarList);

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount(dockerProject.getCpuRequest());
        cpuQuantity.setFormat("m");
        memoryQuantity.setAmount(dockerProject.getMemRequest());
        memoryQuantity.setFormat("Mi");
        cpuLimit.setAmount(dockerProject.getCpuLimit());
        cpuLimit.setFormat("m");
        memoryLimit.setAmount(dockerProject.getMemLimit());
        memoryLimit.setFormat("Mi");
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        limits.put("cpu", cpuLimit);
        limits.put("memory", memoryLimit);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("/home/quant_group/readyCheck.sh");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);
        livenessProbe.setFailureThreshold(5);
        container.setLivenessProbe(livenessProbe);

        //readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(30);
        readinessProbe.setTimeoutSeconds(2);
        readinessProbe.setPeriodSeconds(5);
        container.setReadinessProbe(readinessProbe);

        containerList.add(container);

        // volumes
        List<Volume> volumeList = new ArrayList<>();
        Volume volume = new Volume();
        HostPathVolumeSource hostPathVolumeSource = new HostPathVolumeSource();
        hostPathVolumeSource.setPath("/var/log/containers2/" + namespace + "/" + serviceName);
        volume.setName("filelog");
        volume.setHostPath(hostPathVolumeSource);
        volumeList.add(volume);

        // imagePullSecrets
        List<LocalObjectReference> referenceList = new ArrayList<>();
        LocalObjectReference reference1 = new LocalObjectReference();
        LocalObjectReference reference2 = new LocalObjectReference();
        reference1.setName("qcloudregistrykey");
        reference2.setName("tencenthubkey");
        referenceList.add(reference1);
        referenceList.add(reference2);

        // 设置podSpec
        podSpec.setTerminationGracePeriodSeconds(0L);
        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumeList);
        podSpec.setRestartPolicy("Always");
        podSpec.setDnsPolicy("ClusterFirst");
        podSpec.setImagePullSecrets(referenceList);

        // 设置PodTemplateSpec
        podTemplateSpec.setMetadata(templateObjectMeta);
        podTemplateSpec.setSpec(podSpec);

        // 设置Deployment Spec
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);
        deploymentSpec.setSelector(labelSelector);

        // deployment设置
        deployment.setApiVersion("apps/v1");
        deployment.setKind("Deployment");
        deployment.setMetadata(objectMeta);
        deployment.setSpec(deploymentSpec);

        log.info("开始创建ui/node Deployment：{}", deployment.toString());
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(deployment);
    }

    public PersistentVolumeClaim createRedisPvc(String namespace) {
        PersistentVolumeClaim redisPvc = new PersistentVolumeClaim();

        log.info("开始创建redis pvc：{}", redisPvc.toString());
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(redisPvc);
    }

    public Service createRedisService(String namespace) {
        Service redisService = new Service();

        ObjectMeta objectMeta = buildObjectMeta(namespace, "redis", "base");

        ServiceSpec serviceSpec = new ServiceSpec();

        // 设置spec
        List<ServicePort> portList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        ServicePort servicePort2 = new ServicePort();
        ServicePort servicePort3 = new ServicePort();
        ServicePort servicePort4 = new ServicePort();
        ServicePort servicePort5 = new ServicePort();
        servicePort1.setName("6379");
        servicePort1.setPort(6379);
        servicePort2.setName("6380");
        servicePort2.setPort(6380);
        servicePort3.setName("6381");
        servicePort3.setPort(6381);
        servicePort4.setName("6382");
        servicePort4.setPort(6382);
        servicePort5.setName("6383");
        servicePort5.setPort(6383);
        portList.add(servicePort1);
        portList.add(servicePort2);
        portList.add(servicePort3);
        portList.add(servicePort4);
        portList.add(servicePort5);
        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", "redis");

        // 设置serviceSpec
        serviceSpec.setType("NodePort");
        serviceSpec.setPorts(portList);
        serviceSpec.setSelector(selector);

        // 设置service
        redisService.setApiVersion("v1");
        redisService.setKind("Service");
        redisService.setMetadata(objectMeta);
        redisService.setSpec(serviceSpec);

        log.info("开始创建redis Service: {}", redisService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(redisService);
    }

    /**
     * 创建redis container
     *
     * @param image 镜像名
     * @param port  端口号
     * @return
     */
    public Container createRedisContainer(String image, String port) {
        // TODO: 2021/7/13 cpu和内存控制
        Container container = new Container();
        container.setName("redis-" + port);
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("IfNotPresent");
        List<String> commandList = new ArrayList<>();
        commandList.add("redis-server");
        commandList.add("--port");
        commandList.add(port);
        container.setCommand(commandList);

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount("100");
        cpuQuantity.setFormat("m");
        memoryQuantity.setAmount("1000");
        memoryQuantity.setFormat("Mi");
        cpuLimit.setAmount("100");
        cpuLimit.setFormat("m");
        memoryLimit.setAmount("4000");
        memoryLimit.setFormat("Mi");
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> probeCommandList = new ArrayList<>();
        probeCommandList.add("redis-cli");
        probeCommandList.add("-p");
        probeCommandList.add(port);
        probeCommandList.add("info");
        execAction.setCommand(probeCommandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);
        container.setLivenessProbe(livenessProbe);

        //readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(5);
        readinessProbe.setTimeoutSeconds(1);
        readinessProbe.setPeriodSeconds(5);
        container.setReadinessProbe(readinessProbe);

        return container;
    }

    /**
     * 创建redis deployment
     *
     * @param namespace
     * @param image
     * @return
     */
    public Deployment createRedisDeployment(String namespace, String image) {
        Deployment redisDeployment = new Deployment();

        // 设置meta
        ObjectMeta objectMeta = buildObjectMeta(namespace, "redis", "base");

        // 设置spec
        DeploymentSpec deploymentSpec = new DeploymentSpec();

        // 设置labelSelector
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", "redis");
        labelSelector.setMatchLabels(matchLabels);

        // 设置strategy
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // 设置pod Template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();

        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container1 = createRedisContainer(image, "6379");
        Container container2 = createRedisContainer(image, "6380");
        Container container3 = createRedisContainer(image, "6381");
        Container container4 = createRedisContainer(image, "6382");
        Container container5 = createRedisContainer(image, "6383");

        containerList.add(container1);
        containerList.add(container2);
        containerList.add(container3);
        containerList.add(container4);
        containerList.add(container5);

        // imagePullSecrets
        List<LocalObjectReference> referenceList = new ArrayList<>();
        LocalObjectReference reference1 = new LocalObjectReference();
        LocalObjectReference reference2 = new LocalObjectReference();
        reference1.setName("qcloudregistrykey");
        reference2.setName("tencenthubkey");
        referenceList.add(reference1);
        referenceList.add(reference2);

        podSpec.setContainers(containerList);
        podSpec.setDnsPolicy("ClusterFirst");
        podSpec.setImagePullSecrets(referenceList);
        podSpec.setRestartPolicy("Always");
        podSpec.setTerminationGracePeriodSeconds(30L);

        // 设置PodTemplateSpec
        podTemplateSpec.setMetadata(objectMeta);
        podTemplateSpec.setSpec(podSpec);

        // 设置Deployment Spec
        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);
        deploymentSpec.setSelector(labelSelector);

        DeploymentStatus deploymentStatus = new DeploymentStatus();

        // deployment设置
        redisDeployment.setApiVersion("apps/v1");
        redisDeployment.setKind("Deployment");
        redisDeployment.setMetadata(objectMeta);
        redisDeployment.setSpec(deploymentSpec);
        redisDeployment.setStatus(deploymentStatus);

        log.info("创建redis Deployment：{}", redisDeployment);
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(redisDeployment);
    }

    /**
     * 创建mysql service
     * @param namespace 环境
     * @return
     */
    public Service createMysqlService(String namespace) {
        Service mysqlService = new Service();

        // 创建metadata
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labelMap = new HashMap<>();
        labelMap.put("type", "base");
        labelMap.put("qcloud-app", "mysql");
        objectMeta.setLabels(labelMap);
        objectMeta.setName("mysql");
        objectMeta.setNamespace(namespace);

        // 创建spec
        ServiceSpec serviceSpec = new ServiceSpec();
        List<ServicePort> servicePortList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setPort(3306);
        servicePortList.add(servicePort1);
        Map<String, String> selectorMap = new HashMap<>();
        selectorMap.put("qcloud-app", "mysql");

        serviceSpec.setType("NodePort");
        serviceSpec.setPorts(servicePortList);
        serviceSpec.setSelector(selectorMap);

        // 设置service
        mysqlService.setApiVersion("v1");
        mysqlService.setKind("Service");
        mysqlService.setMetadata(objectMeta);
        mysqlService.setSpec(serviceSpec);

        log.info("开始创建Mysql Service：{}", mysqlService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(mysqlService);
    }

    /**
     * 创建mysql pvc
     *
     * @param namespace 环境
     */
    public PersistentVolumeClaim createMysqlPvc(String namespace) {
        PersistentVolumeClaim mysqlPvc = new PersistentVolumeClaim();

        // 设置metadata
        ObjectMeta objectMeta = new ObjectMeta();
        objectMeta.setName("mysql-" + namespace);
        objectMeta.setNamespace(namespace);

        // 设置spec
        PersistentVolumeClaimSpec pvcSpec = new PersistentVolumeClaimSpec();
        List<String> accessModeList = new ArrayList<>();
        accessModeList.add("ReadWriteOnce");

        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Quantity quantity = new Quantity();
        quantity.setAmount("20");
        quantity.setFormat("Gi");
        requests.put("storage", quantity);
        resourceRequirements.setRequests(requests);

        pvcSpec.setAccessModes(accessModeList);
        pvcSpec.setResources(resourceRequirements);

        mysqlPvc.setApiVersion("v1");
        mysqlPvc.setKind("PersistentVolumeClaim");
        mysqlPvc.setMetadata(objectMeta);
        mysqlPvc.setSpec(pvcSpec);

        log.info("开始创建Mysql pvc：{}", mysqlPvc);
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(mysqlPvc);
    }

    /**
     * 创建Mysql deployment
     */
    public Deployment createMysqlDeployment(String namespace, String image) {
        Deployment mysqlDeployment = new Deployment();

        // 设置metadata
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("type", "base");
        labels.put("qcloud-app", "mysql");
        objectMeta.setLabels(labels);
        objectMeta.setName("mysql");
        objectMeta.setNamespace(namespace);

        // 设置spec
        DeploymentSpec deploymentSpec = new DeploymentSpec();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", "mysql");
        LabelSelector labelSelector = new LabelSelector();
        labelSelector.setMatchLabels(matchLabels);
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // template的metadata
        ObjectMeta specMeta = new ObjectMeta();
        specMeta.setLabels(labels);

        // template的spec
        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("Always");
        container.setName("mysql");

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount("200");
        cpuQuantity.setFormat("m");
        memoryQuantity.setAmount("1");
        memoryQuantity.setFormat("Gi");
        cpuLimit.setAmount("2");
        cpuLimit.setFormat("");
        memoryLimit.setAmount("1");
        memoryLimit.setFormat("Gi");
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        limits.put("cpu", cpuLimit);
        limits.put("memory", memoryLimit);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("MYSQL_DATABASE");
        envVar1.setValue("db");
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("MYSQL_PASSWORD");
        envVar2.setValue("qatest");
        EnvVar envVar3 = new EnvVar();
        envVar3.setName("MYSQL_ROOT_PASSWORD");
        envVar3.setValue("Quantgroup2017");
        EnvVar envVar4 = new EnvVar();
        envVar4.setName("MYSQL_USER");
        envVar4.setValue("qa");
        envVarList.add(envVar1);
        envVarList.add(envVar2);
        envVarList.add(envVar3);
        envVarList.add(envVar4);
        container.setEnv(envVarList);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("mysql");
        commandList.add("-uqa");
        commandList.add("-pqatest");
        commandList.add("-e");
        commandList.add("SELECT 1");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);
        container.setLivenessProbe(livenessProbe);

        // readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(5);
        readinessProbe.setPeriodSeconds(5);
        readinessProbe.setTimeoutSeconds(1);
        container.setReadinessProbe(readinessProbe);

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("mysql");
        volumeMount.setMountPath("/var/lib/mysql");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        // 设置container
        containerList.add(container);
        podSpec.setContainers(containerList);

        // volumes
        List<Volume> volumeList = new ArrayList<>();
        Volume volume = new Volume();
        PersistentVolumeClaimVolumeSource persistentVolumeClaimVolumeSource = new PersistentVolumeClaimVolumeSource();
        persistentVolumeClaimVolumeSource.setClaimName("mysql-" + namespace);
        volume.setName("mysql");
        volume.setPersistentVolumeClaim(persistentVolumeClaimVolumeSource);
        volumeList.add(volume);
        podSpec.setVolumes(volumeList);

        // imagePullSecrets
        List<LocalObjectReference> objectReferences = new ArrayList<>();
        LocalObjectReference objectReference1 = new LocalObjectReference();
        objectReference1.setName("qcloudregistrykey");
        LocalObjectReference objectReference2 = new LocalObjectReference();
        objectReference2.setName("tencenthubkey");
        objectReferences.add(objectReference1);
        objectReferences.add(objectReference2);
        podSpec.setImagePullSecrets(objectReferences);

        // restartPolicy
        podSpec.setRestartPolicy("Always");
        podSpec.setTerminationGracePeriodSeconds(30L);

        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        podTemplateSpec.setMetadata(specMeta);
        podTemplateSpec.setSpec(podSpec);

        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        DeploymentStatus deploymentStatus = new DeploymentStatus();

        mysqlDeployment.setApiVersion("apps/v1");
        mysqlDeployment.setKind("Deployment");
        mysqlDeployment.setMetadata(objectMeta);
        mysqlDeployment.setSpec(deploymentSpec);
        mysqlDeployment.setStatus(deploymentStatus);

        log.info("开始创建Mysql Deployment：{}", mysqlDeployment.toString());
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(mysqlDeployment);
    }

    /**
     * 创建mongodb Service
     *
     * @param namespace 环境
     * @return
     */
    public Service createMongodbService(String namespace) {
        Service mongodbService = new Service();

        ObjectMeta objectMeta = buildObjectMeta(namespace, "mongodb", "base");

        ServiceSpec serviceSpec = new ServiceSpec();
        serviceSpec.setType("NodePort");
        ServicePort servicePort = new ServicePort();
        servicePort.setPort(27017);

        List<ServicePort> servicePortList = new ArrayList<>();
        servicePortList.add(servicePort);
        serviceSpec.setPorts(servicePortList);

        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", "mongodb");
        serviceSpec.setSelector(selector);

        mongodbService.setApiVersion("v1");
        mongodbService.setKind("Service");
        mongodbService.setMetadata(objectMeta);
        mongodbService.setSpec(serviceSpec);

        log.info("开始创建mongodb Service: {}", mongodbService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(mongodbService);
    }

    /**
     * 创建mongodb pvc
     * @param namespace
     * @return
     */
    public PersistentVolumeClaim createMongodbPvc(String namespace) {
        PersistentVolumeClaim mongodbPvc = new PersistentVolumeClaim();

        ObjectMeta objectMeta = new ObjectMeta();
        objectMeta.setName("mongodb-" + namespace);
        objectMeta.setNamespace(namespace);

        PersistentVolumeClaimSpec pvcSpec = new PersistentVolumeClaimSpec();
        List<String> accessModes = new ArrayList<>();
        accessModes.add("ReadWriteOnce");
        pvcSpec.setAccessModes(accessModes);

        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Quantity storage = new Quantity();
        storage.setAmount("10");
        storage.setFormat("Gi");
        requests.put("storage", storage);
        resourceRequirements.setRequests(requests);
        pvcSpec.setResources(resourceRequirements);

        mongodbPvc.setApiVersion("v1");
        mongodbPvc.setKind("PersistentVolumeClaim");
        mongodbPvc.setMetadata(objectMeta);
        mongodbPvc.setSpec(pvcSpec);

        log.info("开始创建mongodb pvc：{}", mongodbPvc.toString());
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(mongodbPvc);
    }

    /**
     * 创建mongodb deployment
     * @param namespace
     * @param image
     * @return
     */
    public Deployment createMongoDbDeployment(String namespace, String image) {
        Deployment mongodbDeployment = new Deployment();

        // 设置meta
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labelMap = new HashMap<>();
        labelMap.put("type", "base");
        labelMap.put("qcloud-app", "mongodb");
        objectMeta.setName("mongodb");
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labelMap);

        DeploymentSpec deploymentSpec = new DeploymentSpec();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", "mongodb");
        LabelSelector labelSelector = new LabelSelector();
        labelSelector.setMatchLabels(matchLabels);
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // template spec
        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("Always");
        container.setName("mongodb");

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuRequest = new Quantity();
        Quantity memoryRequest = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuRequest.setAmount("256");
        cpuRequest.setFormat("m");
        memoryRequest.setAmount("256");
        memoryRequest.setFormat("Mi");
        cpuLimit.setAmount("1024");
        cpuLimit.setFormat("m");
        memoryLimit.setAmount("512");
        memoryLimit.setFormat("Mi");
        requests.put("cpu", cpuRequest);
        requests.put("memory", memoryRequest);
        limits.put("cpu", cpuLimit);
        limits.put("memory", memoryLimit);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("MONGO_INITDB_ROOT_USERNAME");
        envVar1.setValue("qa");
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("MONGO_INITDB_ROOT_PASSWORD");
        envVar2.setValue("qatest");
        envVarList.add(envVar1);
        envVarList.add(envVar2);
        container.setEnv(envVarList);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("mongo");
        commandList.add("-uqa");
        commandList.add("-pqatest");
        commandList.add("admin");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);
        container.setLivenessProbe(livenessProbe);

        // readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(5);
        readinessProbe.setPeriodSeconds(5);
        readinessProbe.setTimeoutSeconds(1);
        container.setReadinessProbe(readinessProbe);

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("mongodb");
        volumeMount.setMountPath("/var/lib/mongo");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        containerList.add(container);

        // volumes
        List<Volume> volumeList = new ArrayList<>();
        PersistentVolumeClaimVolumeSource persistentVolumeClaimVolumeSource = new PersistentVolumeClaimVolumeSource();
        persistentVolumeClaimVolumeSource.setClaimName("mongodb-" + namespace);
        Volume volume = new Volume();
        volume.setName("mongodb");
        volume.setPersistentVolumeClaim(persistentVolumeClaimVolumeSource);
        volumeList.add(volume);

        // imagePullSecrets
        List<LocalObjectReference> imagePullSecrets = new ArrayList<>();
        LocalObjectReference localObjectReference1 = new LocalObjectReference();
        localObjectReference1.setName("qcloudregistrykey");
        LocalObjectReference localObjectReference2 = new LocalObjectReference();
        localObjectReference2.setName("tencenthubkey");
        imagePullSecrets.add(localObjectReference1);
        imagePullSecrets.add(localObjectReference2);

        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumeList);
        podSpec.setImagePullSecrets(imagePullSecrets);
        podSpec.setRestartPolicy("Always");
        podSpec.setTerminationGracePeriodSeconds(30L);

        // template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        podTemplateSpec.setSpec(podSpec);
        podTemplateSpec.setMetadata(objectMeta);

        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        DeploymentStatus deploymentStatus = new DeploymentStatus();

        mongodbDeployment.setApiVersion("apps/v1");
        mongodbDeployment.setKind("Deployment");
        mongodbDeployment.setMetadata(objectMeta);
        mongodbDeployment.setSpec(deploymentSpec);
        mongodbDeployment.setStatus(deploymentStatus);

        log.info("开始创建mongodb Deployment：{}", mongodbDeployment);
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(mongodbDeployment);
    }

    /**
     * 创建zookeeper pvc
     * @param namespace
     * @return
     */
    public PersistentVolumeClaim createZookeeperPvc(String namespace) {
        PersistentVolumeClaim zookeeperPvc = new PersistentVolumeClaim();

        ObjectMeta objectMeta = new ObjectMeta();
        objectMeta.setName("zookeeper-" + namespace);
        objectMeta.setNamespace(namespace);

        PersistentVolumeClaimSpec pvcSpec = new PersistentVolumeClaimSpec();
        List<String> accessModes = new ArrayList<>();
        accessModes.add("ReadWriteOnce");
        ResourceRequirements resourceRequirements = new ResourceRequirements();

        Map<String, Quantity> requests = new HashMap<>();
        Quantity storage = new Quantity();
        storage.setAmount("10");
        storage.setFormat("Gi");
        requests.put("storage", storage);
        resourceRequirements.setRequests(requests);
        pvcSpec.setAccessModes(accessModes);
        pvcSpec.setResources(resourceRequirements);

        zookeeperPvc.setApiVersion("v1");
        zookeeperPvc.setKind("PersistentVolumeClaim");
        zookeeperPvc.setMetadata(objectMeta);
        zookeeperPvc.setSpec(pvcSpec);

        log.info("开始创建Zookeeper pvc: {}", zookeeperPvc.toString());
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(zookeeperPvc);
    }

    /**
     * 创建zookeeper Service
     * @param namespace
     * @return
     */
    public Service createZookeeperService(String namespace) {
        Service zookeeperService = new Service();

        // metadata
        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("type", "base");
        labels.put("qcloud-app", "zookeeper");
        objectMeta.setLabels(labels);
        objectMeta.setName("zookeeper");
        objectMeta.setNamespace(namespace);

        // spec
        ServiceSpec serviceSpec = new ServiceSpec();
        serviceSpec.setType("NodePort");
        List<ServicePort> servicePortList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setName("2181");
        servicePort1.setPort(2181);
        servicePortList.add(servicePort1);
        serviceSpec.setPorts(servicePortList);

        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", "zookeeper");
        serviceSpec.setSelector(selector);

        // 设置Service
        zookeeperService.setApiVersion("v1");
        zookeeperService.setKind("Service");
        zookeeperService.setMetadata(objectMeta);
        zookeeperService.setSpec(serviceSpec);

        log.info("开始创建zookeeper Service: {}", zookeeperService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(zookeeperService);
    }

    /**
     * 创建zookeeper deployment
     * @param namespace
     * @param image
     * @return
     */
    public Deployment createZookeeperDeployment(String namespace, String image) {
        Deployment zookeeperDeployment = new Deployment();

        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("type", "base");
        labels.put("qcloud-app", "zookeeper");
        objectMeta.setLabels(labels);
        objectMeta.setName("zookeeper");
        objectMeta.setNamespace(namespace);

        // selector
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", "zookeeper");
        labelSelector.setMatchLabels(matchLabels);

        // strategy
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // PodSpec
        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("Always");
        container.setName("zookeeper");
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Quantity cpuRequest = new Quantity();
        Quantity menRequest = new Quantity();
        cpuRequest.setAmount("100");
        cpuRequest.setFormat("m");
        menRequest.setAmount("300");
        menRequest.setFormat("Mi");
        requests.put("cpu", cpuRequest);
        requests.put("memory", menRequest);
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuLimit = new Quantity();
        Quantity memLimit = new Quantity();
        cpuLimit.setAmount("200");
        cpuLimit.setFormat("m");
        memLimit.setAmount("300");
        memLimit.setFormat("Mi");
        limits.put("cpu", cpuLimit);
        limits.put("memory", memLimit);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("/zookeeper-3.4.13/readyCheck.sh");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);
        livenessProbe.setFailureThreshold(3);
        livenessProbe.setPeriodSeconds(10);
        livenessProbe.setTimeoutSeconds(1);
        container.setLivenessProbe(livenessProbe);

        // readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(5);
        readinessProbe.setTimeoutSeconds(1);
        readinessProbe.setPeriodSeconds(5);
        readinessProbe.setFailureThreshold(3);
        readinessProbe.setSuccessThreshold(1);
        container.setReadinessProbe(readinessProbe);

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("zookeeper");
        volumeMount.setMountPath("/var/lib/zookeeper");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar = new EnvVar();
        envVar.setName("ZOO_USER");
        envVar.setValue("zookeeper");
        envVarList.add(envVar);
        container.setEnv(envVarList);

        container.setTerminationMessagePath("/dev/termination-log");
        container.setTerminationMessagePolicy("File");

        containerList.add(container);

        // volumes
        List<Volume> volumes = new ArrayList<>();
        Volume volume = new Volume();
        volume.setName("zookeeper");
        PersistentVolumeClaimVolumeSource pvcSource = new PersistentVolumeClaimVolumeSource();
        pvcSource.setClaimName("zookeeper-" + namespace);
        volume.setPersistentVolumeClaim(pvcSource);
        volumes.add(volume);

        List<LocalObjectReference> localObjectReferenceList = new ArrayList<>();
        LocalObjectReference localObjectReference1 = new LocalObjectReference();
        LocalObjectReference localObjectReference2 = new LocalObjectReference();
        localObjectReference1.setName("qcloudregistrykey");
        localObjectReference2.setName("tencenthubkey");
        localObjectReferenceList.add(localObjectReference1);
        localObjectReferenceList.add(localObjectReference2);

        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumes);
        podSpec.setDnsPolicy("ClusterFirst");
        podSpec.setImagePullSecrets(localObjectReferenceList);
        podSpec.setRestartPolicy("Always");
        podSpec.setTerminationGracePeriodSeconds(30L);

        // template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        podTemplateSpec.setMetadata(objectMeta);
        podTemplateSpec.setSpec(podSpec);

        DeploymentSpec deploymentSpec = new DeploymentSpec();
        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setProgressDeadlineSeconds(600);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        DeploymentStatus deploymentStatus = new DeploymentStatus();

        zookeeperDeployment.setApiVersion("apps/v1");
        zookeeperDeployment.setKind("Deployment");
        zookeeperDeployment.setMetadata(objectMeta);
        zookeeperDeployment.setSpec(deploymentSpec);
        zookeeperDeployment.setStatus(deploymentStatus);

        log.info("开始创建zookeeper Deployment：{}", zookeeperDeployment.toString());
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(zookeeperDeployment);
    }

    /**
     * 创建rabbitmq pvc
     * @param namespace
     * @return
     */
    public PersistentVolumeClaim createRabbitmqPvc(String namespace) {
        PersistentVolumeClaim rabbitmqPvc = new PersistentVolumeClaim();

        ObjectMeta objectMeta = new ObjectMeta();
        objectMeta.setName("rabbitmq-" + namespace);
        objectMeta.setNamespace(namespace);

        PersistentVolumeClaimSpec pvcSpec = new PersistentVolumeClaimSpec();

        List<String> accessModes = new ArrayList<>();
        accessModes.add("ReadWriteOnce");
        pvcSpec.setAccessModes(accessModes);

        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Quantity storage = new Quantity();
        storage.setAmount("10");
        storage.setFormat("Gi");
        requests.put("storage", storage);
        resourceRequirements.setRequests(requests);
        pvcSpec.setResources(resourceRequirements);

        rabbitmqPvc.setApiVersion("v1");
        rabbitmqPvc.setKind("PersistentVolumeClaim");
        rabbitmqPvc.setMetadata(objectMeta);
        rabbitmqPvc.setSpec(pvcSpec);

        log.info("开始创建rabbitmq Pvc：{}", rabbitmqPvc.toString());
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(rabbitmqPvc);
    }

    /**
     * 创建rabbitmq service
     * @param namespace
     * @return
     */
    public Service createRabbitmqService(String namespace) {
        Service rabbitmqService = new Service();

        ObjectMeta objectMeta = new ObjectMeta();

        Map<String, String> labels = new HashMap<>();
        labels.put("type", "base");
        labels.put("qcloud-app", "rabbitmq");
        objectMeta.setLabels(labels);
        objectMeta.setName("rabbitmq");
        objectMeta.setNamespace(namespace);

        ServiceSpec serviceSpec = new ServiceSpec();
        serviceSpec.setType("NodePort");
        List<ServicePort> servicePortList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setName("tcp-5672-5672");
        servicePort1.setPort(5672);
        ServicePort servicePort2 = new ServicePort();
        servicePort2.setName("tcp-15672-15672");
        servicePort2.setPort(15672);
        servicePortList.add(servicePort1);
        servicePortList.add(servicePort2);
        serviceSpec.setPorts(servicePortList);

        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", "rabbitmq");
        serviceSpec.setSelector(selector);

        rabbitmqService.setApiVersion("v1");
        rabbitmqService.setKind("Service");
        rabbitmqService.setMetadata(objectMeta);
        rabbitmqService.setSpec(serviceSpec);

        log.info("开始创建rabbitmq Service: {}", rabbitmqService.toString());
        return kubernetesClient.services().inNamespace(namespace).create(rabbitmqService);
    }

    /**
     * 创建rabbitmq deployment
     * @param namespace
     * @param image
     * @return
     */
    public Deployment createRabbitmqDeployment(String namespace, String image) {
        Deployment rabbitmqDeployment = new Deployment();

        ObjectMeta objectMeta = buildObjectMeta(namespace, "rabbitmq", "base");

        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", "rabbitmq");
        labelSelector.setMatchLabels(matchLabels);

        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        ResourceRequirements resourceRequirements = buildResourceRequirements("100", "m",
                "1500", "Mi", "200", "m", "1500", "Mi");

        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("RABBITMQ_DEFAULT_USER");
        envVar1.setValue("qa");
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("RABBITMQ_DEFAULT_PASS");
        envVar2.setValue("qatest");
        envVarList.add(envVar1);
        envVarList.add(envVar2);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("/opt/readyCheck.sh");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(100);
        livenessProbe.setSuccessThreshold(1);

        //readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(30);
        readinessProbe.setTimeoutSeconds(2);
        readinessProbe.setPeriodSeconds(5);
        readinessProbe.setSuccessThreshold(1);

        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setImage("ccr.ccs.tencentyun.com/" + image);
        container.setImagePullPolicy("IfNotPresent");
        container.setName("rabbitmq");
        container.setResources(resourceRequirements);
        container.setEnv(envVarList);
        container.setLivenessProbe(livenessProbe);
        container.setReadinessProbe(readinessProbe);
        containerList.add(container);

        List<Volume> volumeList = new ArrayList<>();

        // imagePullSecrets
        List<LocalObjectReference> localObjectReferenceList = new ArrayList<>();
        LocalObjectReference localObjectReference1 = new LocalObjectReference();
        LocalObjectReference localObjectReference2 = new LocalObjectReference();
        localObjectReference1.setName("qcloudregistrykey");
        localObjectReference2.setName("tencenthubkey");
        localObjectReferenceList.add(localObjectReference1);
        localObjectReferenceList.add(localObjectReference2);

        PodSpec podSpec = new PodSpec();
        podSpec.setHostname("rabbitmq-" + namespace);
        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumeList);
        podSpec.setImagePullSecrets(localObjectReferenceList);
        podSpec.setRestartPolicy("Always");
        podSpec.setTerminationGracePeriodSeconds(30L);

        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        podTemplateSpec.setMetadata(objectMeta);
        podTemplateSpec.setSpec(podSpec);

        DeploymentSpec deploymentSpec = new DeploymentSpec();
        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        DeploymentStatus deploymentStatus = new DeploymentStatus();

        rabbitmqDeployment.setApiVersion("apps/v1");
        rabbitmqDeployment.setKind("Deployment");
        rabbitmqDeployment.setMetadata(objectMeta);
        rabbitmqDeployment.setSpec(deploymentSpec);
        rabbitmqDeployment.setStatus(deploymentStatus);

        log.info("开始创建rabbitmq Deployment：{}", rabbitmqDeployment.toString());
        return kubernetesClient.apps().deployments().inNamespace(namespace).create(rabbitmqDeployment);
    }

    /**
     * 查询PersistentVolumeClaims是否存在
     * @param namespace 环境
     * @param pvcName pvc名称
     * @return
     */
    public boolean queryIfPvcExistByName(String namespace, String pvcName) {
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).withName(pvcName + "-" + namespace).get() != null;
    }

    /**
     * 查询Deployment是否存在
     * @param namespace 环境
     * @param deploymentName Deployment名称
     * @return
     */
    public boolean queryIfDeploymentExistByName(String namespace, String deploymentName) {
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(deploymentName).get() != null;
    }

    /**
     * 查询Service是否存在
     * @param namespace 环境
     * @param serviceName Service名称
     * @return
     */
    public boolean queryIfServiceExistByName(String namespace, String serviceName) {
        return kubernetesClient.services().inNamespace(namespace).withName(serviceName).get() != null;
    }
    /**
     * 更新部署pod
     *
     * @param namespace   环境
     * @param serviceName 服务名
     * @param image       镜像名
     * @return
     */
    public Deployment updateDeployment(String namespace, String serviceName, String image) {
        Deployment deployment = kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).get();
        List<Container> containerList = deployment.getSpec().getTemplate().getSpec().getContainers();
        containerList.get(0).setImage("ccr.ccs.tencentyun.com/" + image);
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).createOrReplace(deployment);
    }

    /**
     * 创建Ingress，主要用来配置域名
     * @param namespace 环境
     * @param serviceName 服务名
     * @param domain 域名
     */
    public Ingress createIngress(String namespace, String serviceName, String domain) {
        Ingress ingress = new Ingress();

        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> annotationMap = new HashMap<>();
        annotationMap.put("kubernetes.io/ingress.class", "traefik");
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setAnnotations(annotationMap);
        objectMeta.setGeneration(1L);

        IngressSpec ingressSpec = new IngressSpec();
        List<IngressRule> ingressRuleList = new ArrayList<>();
        IngressRule ingressRule = new IngressRule();
        HTTPIngressRuleValue httpIngressRuleValue = new HTTPIngressRuleValue();
        List<HTTPIngressPath> paths = new ArrayList<>();
        HTTPIngressPath ingressPath = new HTTPIngressPath();
        ingressPath.setPath("/");
        IngressBackend ingressBackend = new IngressBackend();
        ingressBackend.setServiceName(serviceName);
        IntOrString servicePort = new IntOrString();
        if (serviceName.equals("heimdallr")) {
            servicePort.setIntVal(5555);
        } else {
            servicePort.setIntVal(80);
        }
        ingressBackend.setServicePort(servicePort);
        ingressPath.setBackend(ingressBackend);
        paths.add(ingressPath);
        httpIngressRuleValue.setPaths(paths);
        ingressRule.setHttp(httpIngressRuleValue);

        ingressRule.setHost(domain + "-" + namespace + ".liangkebang.net");
        ingressRule.setHttp(httpIngressRuleValue);
        ingressRuleList.add(ingressRule);
        ingressSpec.setRules(ingressRuleList);

        ingress.setApiVersion("extensions/v1beta1");
        ingress.setKind("Ingress");
        ingress.setMetadata(objectMeta);
        ingress.setSpec(ingressSpec);
        return kubernetesClient.extensions().ingresses().inNamespace(namespace).create(ingress);
    }

    /**
     * 查询Ingress是否存在
     * @param namespace 环境
     * @param serviceName 服务名
     * @return
     */
    public boolean queryIfIngressExistByName(String namespace, String serviceName) {
        return kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).get() != null;
    }

    /**
     * 获取Service详情
     * @param namespace 环境
     * @param serviceName service名称
     * @return
     */
    public Service getServiceDetail(String namespace, String serviceName) {
        return kubernetesClient.services().inNamespace(namespace).withName(serviceName).get();
    }

    public Pod getPodDetail(String namespace, String serviceType, String serviceName) {
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> labelMap = new HashMap<>();
        labelMap.put("type", serviceType);
        labelMap.put("qcloud-app", serviceName);
        labelSelector.setMatchLabels(labelMap);
        return kubernetesClient.pods().inNamespace(namespace).withLabelSelector(labelSelector).list().getItems().get(0);
    }

    /**
     * 获取pod和service的一些信息
     * @param namespace
     * @param serviceType
     * @param serviceName
     * @return
     */
    public Map<String, Object> getPodAndServiceInfo(String namespace, String serviceType, String serviceName) {
        Service service = getServiceDetail(namespace, serviceName);
        Pod pod = getPodDetail(namespace, serviceType, serviceName);
        Map<String, Object> map = new HashMap<>();
        if (pod.getStatus().getPhase().equals("Running")) {
            // 端口映射
            List<Map<String, Object>> portMappingList = new ArrayList<>();
            ObjectMeta podMetadata = pod.getMetadata();
            if (service != null) {
                List<ServicePort> servicePortList = service.getSpec().getPorts();
                if (servicePortList.size() > 0) {
                    for (ServicePort servicePort : servicePortList) {
                        if (servicePort.getNodePort() != null) {
                            map.put("port_" + servicePort.getName(), servicePort.getNodePort());
                        }
                        Map<String, Object> portMap = new HashMap<>();
                        portMap.put("name", servicePort.getName());
                        portMap.put("nodePort", servicePort.getNodePort());
                        portMap.put("port", servicePort.getPort());
                        portMap.put("protocol", servicePort.getProtocol());
                        portMap.put("targetPort", servicePort.getTargetPort());
                        portMappingList.add(portMap);
                    }
                }
                if (portMappingList.size() > 0) {
                    map.put("portMappings", portMappingList);
                }
            }
            map.put("createdAt", podMetadata.getCreationTimestamp());
            map.put("serviceName", serviceName);
            map.put("podName", podMetadata.getName());
            map.put("labels", podMetadata.getLabels());
            List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
            map.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
            map.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
            map.put("lanIp", pod.getStatus().getHostIP());
            map.put("podIp", pod.getStatus().getPodIP());
            map.put("startTime", pod.getStatus().getStartTime());
            map.put("status", containerStatuses.get(0).getReady());
            Ingress ingress = kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).get();
            if (ingress != null) {
                if (ingress.getSpec().getRules() != null && ingress.getSpec().getRules().size() > 0) {
                    map.put("host", ingress.getSpec().getRules().get(0).getHost());
                } else {
                    map.put("host", "");
                }
            }
        }
        return map;
    }

    /**
     * 创建新的namespace
     * @param name namespace名称
     * @param desc dev/test
     * @param owner
     * @return
     */
    public boolean createNewNamespace(String name, String desc, String owner) {
        try {
            Namespace namespace = new Namespace();
            ObjectMeta objectMeta = new ObjectMeta();
            Map<String, String> annotations = new HashMap<>();
            annotations.put("description", desc);
            annotations.put("owner", owner);
            objectMeta.setAnnotations(annotations);
            objectMeta.setName(name);

            NamespaceSpec namespaceSpec = new NamespaceSpec();
            List<String> finalizers = new ArrayList<>();
            finalizers.add("kubernetes");
            namespaceSpec.setFinalizers(finalizers);

            NamespaceStatus namespaceStatus = new NamespaceStatus();
            namespaceStatus.setPhase("Active");

            namespace.setApiVersion("v1");
            namespace.setKind("Namespace");
            namespace.setMetadata(objectMeta);
            namespace.setSpec(namespaceSpec);
            namespace.setStatus(namespaceStatus);
            log.info("创建新的环境: {}", namespace.toString());
            kubernetesClient.namespaces().create(namespace);
            createQcloudRegistryKeySecret(name);
            createTencenthHubKeySecret(name);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    public Secret createQcloudRegistryKeySecret(String namespace) {
        Secret secret = new Secret();

        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "qcloudregistrykey");
        objectMeta.setLabels(labels);
        objectMeta.setName("qcloudregistrykey");
        objectMeta.setNamespace(namespace);

        Map<String, String> data = new HashMap<>();
        data.put(".dockercfg", "eyJjY3IuY2NzLnRlbmNlbnR5dW4uY29tIjp7InVzZXJuYW1lIjoiMTAwMDA4NjMyMjY5IiwicGFzc3dvcmQiOiJ7QXBwbGljYXRpb25Ub2tlbjo0MTM1NWY2OWQyY2U3ZWUyYmE2YjQxZmMxNzJlNzM0MH0iLCJlbWFpbCI6IjEwMDAwODYzMjI2OUBxcS5jb20iLCJhdXRoIjoiTVRBd01EQTROak15TWpZNU9udEJjSEJzYVdOaGRHbHZibFJ2YTJWdU9qUXhNelUxWmpZNVpESmpaVGRsWlRKaVlUWmlOREZtWXpFM01tVTNNelF3ZlE9PSJ9fQ==");

        secret.setApiVersion("v1");
        secret.setKind("Secret");
        secret.setType("kubernetes.io/dockercfg");
        secret.setMetadata(objectMeta);
        secret.setData(data);
        log.info("{}环境，创建新的secret：qcloudregistrykey", namespace);
        return kubernetesClient.secrets().inNamespace(namespace).create(secret);
    }

    public Secret createTencenthHubKeySecret(String namespace) {
        Secret secret = new Secret();

        ObjectMeta objectMeta = new ObjectMeta();
        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "tencenthubkey");
        objectMeta.setLabels(labels);
        objectMeta.setName("tencenthubkey");
        objectMeta.setNamespace(namespace);

        Map<String, String> data = new HashMap<>();
        data.put(".dockercfg", "eyJodWIudGVuY2VudHl1bi5jb20iOnsidXNlcm5hbWUiOiIxMDAwMDg2MzIyNjkiLCJwYXNzd29yZCI6ImV5SmhiR2NpT2lKSVV6STFOaUlzSW5SNWNDSTZJa3BYVkNKOS5leUpsZUhBaU9qRTROell5T1RFME9EVXNJbXAwYVNJNklqVTVPVGRqTmpkaExUWmpaV1l0TkRGbE55MDVPRGsyTFdGaE16TTFPRGRpWW1ReE1TSXNJbWxoZENJNk1UVTJNRGt6TVRRNE5Td2libUptSWpveE5UWXdPVE14TkRnMUxDSnpkV0lpT2lJeE1EQXdNRGcyTXpJeU5qa2lmUS5ZWEl4TzBaWTg3QmNzSktLdW9lQ3gzYllLZ1RzMktNVElSeDd5dWVRdDZNIiwiZW1haWwiOiIxMDAwMDg2MzIyNjlAcXEuY29tIiwiYXV0aCI6Ik1UQXdNREE0TmpNeU1qWTVPbVY1U21oaVIyTnBUMmxLU1ZWNlNURk9hVWx6U1c1U05XTkRTVFpKYTNCWVZrTktPUzVsZVVwc1pVaEJhVTlxUlRST2VsbDVUMVJGTUU5RVZYTkpiWEF3WVZOSk5rbHFWVFZQVkdScVRtcGthRXhVV21wYVYxbDBUa1JHYkU1NU1EVlBSR3N5VEZkR2FFMTZUVEZQUkdScFdXMVJlRTFUU1hOSmJXeG9aRU5KTmsxVVZUSk5SR3Q2VFZSUk5FNVRkMmxpYlVwdFNXcHZlRTVVV1hkUFZFMTRUa1JuTVV4RFNucGtWMGxwVDJsSmVFMUVRWGROUkdjeVRYcEplVTVxYTJsbVVTNVpXRWw0VHpCYVdUZzNRbU56U2t0TGRXOWxRM2d6WWxsTFoxUnpNa3ROVkVsU2VEZDVkV1ZSZERaTiJ9fQ==");
        secret.setApiVersion("v1");
        secret.setKind("Secret");
        secret.setType("kubernetes.io/dockercfg");
        secret.setMetadata(objectMeta);
        secret.setData(data);
        log.info("{}环境，创建新的secret：qcloudregistrykey", namespace);
        return kubernetesClient.secrets().inNamespace(namespace).create(secret);
    }

    /**
     * 删除ingress
     * @param namespace 环境
     * @param serviceName 服务名
     * @return
     */
    public boolean deleteIngress(String namespace, String serviceName) {
        log.info("删除{}环境的{} Ingress", namespace, serviceName);
        return kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).delete();
    }

    /**
     * 删除Service
     * @param namespace 环境
     * @param serviceName 服务名
     * @return
     */
    public boolean deleteService(String namespace, String serviceName) {
        log.info("删除{}环境的{} Service", namespace, serviceName);
        return kubernetesClient.services().inNamespace(namespace).withName(serviceName).delete();
    }

    /**
     * 获取Deployment
     * @param namespace
     * @param serviceName
     * @return
     */
    public Deployment getDeployment(String namespace, String serviceName) {
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).get();
    }

    /**
     * 获取环境的所有k8s变量
     * @param namespace 环境
     */
    public Map<String, Object> getListEnvVars(String namespace) {
        Map<String, Object> envMap = new TreeMap<>();
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).list().getItems();
        List<Service> serviceList = kubernetesClient.services().inNamespace(namespace).list().getItems();
        // 遍历Pod列表
        for (Pod pod: podList) {
            String serviceName = pod.getMetadata().getLabels().get("qcloud-app");
            serviceName = serviceName.replace("-", "_").toUpperCase();
            envMap.put(serviceName + "_SERVICE_HOST", pod.getStatus().getHostIP());
            // MYSQL需要额外加个参数
            if (serviceName.equals("MYSQL")) {
                envMap.put("DB_SERVICE_HOST", pod.getStatus().getHostIP());
            }
        }
        // 遍历Service列表
        for (Service service: serviceList) {
            String serviceName = service.getMetadata().getName().toUpperCase().replaceAll("-", "_");
            List<ServicePort> servicePortList = service.getSpec().getPorts();
            if (service.getMetadata().getLabels().get("type").equals("base")) {
                if (servicePortList.get(0).getPort() != null) {
                    envMap.put(serviceName + "_SERVICE_PORT", servicePortList.get(0).getNodePort());
                    // MYSQL需要额外加个参数
                    if (serviceName.equals("MYSQL")) {
                        envMap.put("DB_SERVICE_PORT", servicePortList.get(0).getNodePort());
                    }
                }
                for (ServicePort servicePort: servicePortList) {
                    envMap.put(serviceName + "_SERVICE_PORT_" + servicePort.getPort(), servicePort.getNodePort());
                    // MYSQL需要额外加个参数
                    if (serviceName.equals("MYSQL")) {
                        envMap.put("DB_SERVICE_PORT_" + servicePort.getPort(), servicePort.getNodePort());
                    }
                }
            } else {
                if (servicePortList.get(0).getNodePort() != null) {
                    envMap.put(serviceName + "_SERVICE_PORT", servicePortList.get(0).getNodePort());
                }
                for (ServicePort servicePort: servicePortList) {
                    if (service.getSpec().getType().equals("NodePort")) {
                        envMap.put(serviceName + "_SERVICE_PORT_" + servicePort.getPort(), servicePort.getNodePort());
                    } else if (service.getSpec().getType().equals("ClusterIP")) {
                        envMap.put(serviceName + "_SERVICE_PORT_" + servicePort.getName(), servicePort.getPort());
                    }
                }
            }
        }
        return envMap;
    }

    public void podTest() {
        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "mysql");
        labels.put("type", "base");
        List<Pod> podList = kubernetesClient.pods().inNamespace("vcc").withLabels(labels).list().getItems();
        String host = null;
        for (Pod pod: podList) {
            if (pod.getStatus().getHostIP() != null) {
                host = pod.getStatus().getHostIP();
                break;
            }
        }
        System.out.println(podList.size());
        System.out.println(host);
    }

    public static void main(String[] args) {
        K8sService k8sService = new K8sService();
//        k8sService.createNewNamespace("test6", "dev", "黎博");
        k8sService.podTest();
//        k8sService.getPodList("k8s");
//        System.out.println(JSON.toJSONString(k8sService.getPodListNew("fe")));
//        if (!k8sService.queryIfServiceExistByName("fe", "lxq-ui")) {
//            k8sService.createUIAndNodeService("fe", "lxq-ui", "ui", "ui");
//        }

    }
}
