package cn.qg.holmes.service.effect.impl;

import cn.qg.holmes.service.effect.DatabaseSyncService;
import cn.qg.holmes.utils.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.*;
import java.util.ArrayList;
import java.util.List;

@Slf4j
@Service
public class DatabaseSyncServiceImpl implements DatabaseSyncService {

    @Autowired
    RedisUtils redisUtils;

    private final String dbSyncPrefix = "dbsync:";

    /**
     * 从中间库获取数据库相关信息
     * @param ip mysql ip
     * @param port mysql port
     * @param username  mysql用户名
     * @param password mysql密码
     * @param dbName 数据库名
     * @return
     */
    @Override
    public boolean getDbInfoFromSource(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName + "?zeroDateTimeBehavior=convertToNull";
        String createTableKey = dbSyncPrefix + dbName + ":create" ;
        String insertTableKey = dbSyncPrefix + dbName + ":insert";
        String dbRedisKey = dbSyncPrefix + dbName;

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        String dbRedisValue = "";
        String insertRedisValue = "";
        List<Object> createTableRedisValue = new ArrayList<>();
        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();
        List<String> commentList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);

            if (!redisUtils.hasKey(dbRedisKey)) {
                String dbSql = String.format("SHOW CREATE DATABASE %s", dbName);
                preparedStatement = connection.prepareStatement(dbSql);
                ResultSet dbResultSet = preparedStatement.executeQuery();
                while (dbResultSet.next()) {
                    dbRedisValue = dbResultSet.getString(2);
                }
                // 设置建库语句的redis值
                redisUtils.set(dbRedisKey, dbRedisValue, 43200);
            }

            if (!redisUtils.hasKey(createTableKey)) {
                DatabaseMetaData databaseMetaData = connection.getMetaData();
                // 获取所有表名
                ResultSet tableResultSet = databaseMetaData.getTables(null, null, null, new String[]{"TABLE"});
                while (tableResultSet.next()) {
                    String tableName = tableResultSet.getString("TABLE_NAME");

                    // 获取所有建表语句
                    String sql = String.format("SHOW CREATE TABLE %s", tableName);
                    preparedStatement = connection.prepareStatement(sql);
                    ResultSet pResultSet = preparedStatement.executeQuery();
                    while (pResultSet.next()) {
                        log.info("获取建表语句：{}", pResultSet.getString(2));
                        createTableRedisValue.add(pResultSet.getString(2));
                    }

                    if (!redisUtils.hasKey(insertTableKey)) {
                        ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                        while (columnResultSet.next()) {
                            // 字段名称
                            String columnName = columnResultSet.getString("COLUMN_NAME");
                            // 数据类型
                            String columnType = columnResultSet.getString("TYPE_NAME");
                            // 备注
                            String remarks = columnResultSet.getString("REMARKS");

                            columnNameList.add(columnName);
                            columnTypeList.add(columnType);
                            commentList.add(remarks);
                        }
                        String columnArrayStr = null;
                        for (String column : columnNameList) {
                            if (null == columnArrayStr) {
                                columnArrayStr = "`" + column + "`";
                            } else {
                                columnArrayStr = columnArrayStr + "," + "`" + column + "`";
                            }
                        }

                        String selectSQL = String.format("select %s from %s", columnArrayStr, tableName);
                        preparedStatement = connection.prepareStatement(selectSQL);
                        ResultSet selectResultSet = preparedStatement.executeQuery();
                        while (selectResultSet.next()) {
                            String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
                            String insertSql = String.format("insert into %s (%s) values(%s);", tableName, columnArrayStr,
                                    rowValues);
                            log.info(insertSql);
                            insertSql = insertSql.replaceAll("\n", "<br/>");
                            insertSql = insertSql + "\n";
                            insertRedisValue += insertSql;
                        }
                        columnNameList.clear();
                        columnTypeList.clear();
                        commentList.clear();
                    }
                }
                redisUtils.lSet(createTableKey, createTableRedisValue, 43200);
                redisUtils.set(insertTableKey, insertRedisValue, 43200);
            }

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    @Override
    public boolean syncDbToDest(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port;
        String createTableKey = dbSyncPrefix + dbName + ":create" ;
        String insertTableKey = dbSyncPrefix + dbName + ":insert";
        String dbRedisKey = dbSyncPrefix + dbName;

        Connection connection = null;
        Connection newConnection = null;
        PreparedStatement preparedStatement = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);

            // 删除原有数据库
            String dropDbSql = String.format("DROP DATABASE IF EXISTS %s", dbName);
            log.info("删除原有数据库：{}", dropDbSql);
            preparedStatement = connection.prepareStatement(dropDbSql);
            preparedStatement.execute();

            // 重新创建数据库
            String createDbSql = redisUtils.get(dbRedisKey).toString();
            log.info("创建数据库：{}", createDbSql);
            preparedStatement = connection.prepareStatement(createDbSql);
            preparedStatement.execute();
            connection.close();

            // 重新设置连接，并指定数据库
            newConnection = DriverManager.getConnection(url + "/" + dbName, username, password);
            newConnection.setAutoCommit(true);

            // 从redis中获取要同步的表结构
            List<Object> createTableRedisValue = redisUtils.lGet(createTableKey, 0, redisUtils.lGetListSize(createTableKey));
            for (Object sql: createTableRedisValue) {
                preparedStatement = newConnection.prepareStatement(sql.toString());
                preparedStatement.execute();
            }

            // 从redis中同步表数据
            String insertTableRedisValue = redisUtils.get(insertTableKey).toString();
            preparedStatement = newConnection.prepareStatement(insertTableRedisValue);
            preparedStatement.execute();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (newConnection != null) {
                    newConnection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    /**
     * 获取数据库列表
     * @param ip 同步库ip
     * @param port 同步库端口号
     * @param username 同步库用户名
     * @param password 同步库密码
     * @return
     */
    @Override
    public List<Object> getDatabaseList(String ip, String port, String username, String password) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port;

        List<Object> databaseList = new ArrayList<>();
        String dbListRedisKey = dbSyncPrefix + ":db:list";

        if (redisUtils.hasKey(dbListRedisKey)) {
            databaseList = redisUtils.lGet(dbListRedisKey, 0, redisUtils.lGetListSize(dbListRedisKey));
            log.info("从redis获取到的数据库列表为：{}", databaseList);
        } else {
            Connection connection = null;

            try {
                Class.forName(driver);
                connection = DriverManager.getConnection(url, username, password);
                connection.setAutoCommit(true);

                String showDatabasesSql = "SHOW DATABASES";
                PreparedStatement preparedStatement = connection.prepareStatement(showDatabasesSql);
                ResultSet dbListResultSet = preparedStatement.executeQuery();
                while (dbListResultSet.next()) {
                    databaseList.add(dbListResultSet.getString(1));
                }
                preparedStatement.close();
                // 设置数据库列表缓存，缓存6个小时
                log.info("从同步库获取到的数据库列表为：{}", databaseList);
                redisUtils.lSet(dbListRedisKey, databaseList, 21600);
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                try {
                    if (connection != null) {
                        connection.close();
                    }
                } catch (Exception e) {
                    connection = null;
                }
            }
        }
        return databaseList;
    }

    /**
     * 获取表数据一行的所有值
     * @param rs
     * @param size
     * @param columnTypeList
     * @return
     */
    private String getRowValues(ResultSet rs, int size, List<String> columnTypeList) {
        try {
            String rowValues = null;
            for (int i = 1; i <= size; i++) {
                String columnValue = null;

                // 获取字段值
                columnValue = getValue(rs, i, columnTypeList.get(i - 1));
                // 如果是空值不添加单引号
                if (null != columnValue) {
                    columnValue = "'" + columnValue + "'";
                }
                // 拼接字段值
                if (null == rowValues) {
                    rowValues = columnValue;
                } else {
                    rowValues = rowValues + "," + columnValue;
                }
            }

            return rowValues;
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("获取表数据一行的所有值异常");
            return null;
        }
    }

    /**
     * 根据类型获取字段值
     * @return
     * @author  2018年9月6日 上午11:16:00
     */
    private String getValue(ResultSet resultSet, Integer index, String columnType) {
        try {
            if ("int".equals(columnType) || "INT".equals(columnType)) {
                // 整数
                Object intValue = resultSet.getObject(index);

                if (null == intValue) {
                    return null;
                }
                return intValue + "";
            } else if ("bigint".equals(columnType) || "BIGINT".equals(columnType)) {

                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("smallint".equals(columnType) || "SMALLINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("tinyint".equals(columnType) || "TINYINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("mediumint".equals(columnType) || "MEDIUMINT".equals(columnType)) {
                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("integer".equals(columnType) || "INTEGER".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("float".equals(columnType) || "FLOAT".equals(columnType)) {

                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("double".equals(columnType) || "DOUBLE".equals(columnType)) {
                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("decimal".equals(columnType) || "DECIMAL".equals(columnType)) {
                // 浮点数-金额类型
                BigDecimal value = resultSet.getBigDecimal(index);
                if (null == value) {
                    return null;
                }
                return value.toString();
            } else if ("char".equals(columnType) || "CHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("varchar".equals(columnType) || "VARCHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("tinytext".equals(columnType) || "TINYTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("text".equals(columnType) || "TEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("mediumtext".equals(columnType) || "MEDIUMTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("longtext".equals(columnType) || "LONGTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("year".equals(columnType) || "YEAR".equals(columnType)) {
                // 时间类型：范围 1901/2155 格式 YYYY
                String year = resultSet.getString(index);
                if (null == year) {
                    return null;
                }
                // 只需要年的字符即可，
                return year.substring(0, 4);
            } else if ("date".equals(columnType) || "DATE".equals(columnType)) {
                // 时间类型：范围 '1000-01-01'--'9999-12-31' 格式 YYYY-MM-DD
                return resultSet.getString(index);
            } else if ("time".equals(columnType) || "TIME".equals(columnType)) {
                // 时间类型：范围 '-838:59:59'到'838:59:59' 格式 HH:MM:SS
                return resultSet.getString(index);
            } else if ("datetime".equals(columnType) || "DATETIME".equals(columnType)) {
                // 时间类型：范围 '1000-01-01 00:00:00'--'9999-12-31 23:59:59' 格式 YYYY-MM-DD HH:MM:SS
                return resultSet.getString(index);
            } else if ("timestamp".equals(columnType) || "TIMESTAMP".equals(columnType)) {
                // 时间类型：范围 1970-01-01 00:00:00/2037 年某时 格式 YYYYMMDD HHMMSS 混合日期和时间值，时间戳
                return resultSet.getString(index);
            } else {
                return null;
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.err.println("获取数据库类型值异常");
            return null;
        }
    }

}
