package cn.qg.holmes.controller.mock.keystone.jdbuy;

import cn.qg.holmes.bean.KeystoneService;
import cn.qg.holmes.entity.mock.keystone.ProductItem;
import cn.qg.holmes.entity.mock.keystone.jdbuy.*;
import cn.qg.holmes.utils.DateUtils;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.*;

@Slf4j
@RestController
public class OrderController {

    /**
     * 商品税率
     */
    private final BigDecimal TAX = new BigDecimal("0.13");
    /**
     * SKU分配，随便MOCK一个
     */
    private final Integer SKU_CATEGORY = 15924;
    @Autowired
    KeystoneService keystoneService;
    @Autowired
    RedisUtils redisUtils;
    private String QYG_ORDER_PREFIX = "qyg_order_";

    /**
     * 7.3 企业购提交订单
     *
     * @param submitOrderRequest 请求参数
     * @return
     */
    @PostMapping("/api/order/submitOrder")
    public JdSubmitOrderResponse qygSubmitOrder(SubmitOrderRequest submitOrderRequest) {
        log.info("收到企业购提交订单请求：{}", JSON.toJSONString(submitOrderRequest));
        JdSubmitOrderResponse response = new JdSubmitOrderResponse();
        JSONArray orderPriceSnapArray = JSON.parseArray(submitOrderRequest.getOrderPriceSnap());
        JSONArray skuInfoArray = JSON.parseArray(submitOrderRequest.getSku());
        List<SubmitOrderRequest.JdSkuPriceInfo> orderPriceSnaps = JSON.parseObject(orderPriceSnapArray.toJSONString(), new TypeReference<List<SubmitOrderRequest.JdSkuPriceInfo>>() {
        });
        List<SubmitOrderRequest.JdSkuInfo> skuInfos = JSON.parseObject(skuInfoArray.toJSONString(), new TypeReference<List<SubmitOrderRequest.JdSkuInfo>>() {
        });
        Map<String, BigDecimal> skuPriceMap = convertSkuPriceMap(orderPriceSnaps);
        // 计算订单总金额
        BigDecimal orderPrice = new BigDecimal(0);
        for (SubmitOrderRequest.JdSkuInfo jdSkuInfo : skuInfos) {
            BigDecimal skuNum = new BigDecimal(jdSkuInfo.getNum());
            String skuId = jdSkuInfo.getSkuId();
            BigDecimal price = skuPriceMap.get(skuId);
            orderPrice = orderPrice.add(price.multiply(skuNum));
        }
        response.setSuccess(true);
        response.setResultCode("0001");
        response.setResultMessage("下单成功");

        JdSubmitOrderResponse.JdSumbitOrderInfo jdSumbitOrderInfo = new JdSubmitOrderResponse.JdSumbitOrderInfo();
        String jdOrderId = RandomStringUtils.randomNumeric(12);
        jdSumbitOrderInfo.setJdOrderId(Long.valueOf(jdOrderId));
        jdSumbitOrderInfo.setFreight(calculateFreight(orderPrice));
        jdSumbitOrderInfo.setOrderPrice(orderPrice);
        BigDecimal orderNakedPrice = orderPrice.divide(TAX.add(new BigDecimal(1)), BigDecimal.ROUND_HALF_UP);
        jdSumbitOrderInfo.setOrderNakedPrice(orderNakedPrice);
        jdSumbitOrderInfo.setOrderTaxPrice(orderNakedPrice.multiply(TAX).setScale(2, BigDecimal.ROUND_HALF_UP));

        List<JdSubmitOrderResponse.JdBizSku> bizSkuList = new ArrayList<>();
        for (SubmitOrderRequest.JdSkuInfo sku : skuInfos) {
            JdSubmitOrderResponse.JdBizSku bizSku = new JdSubmitOrderResponse.JdBizSku();
            String skuId = sku.getSkuId();
            ProductItem productItem = keystoneService.getProductItemById(skuId, "jdbuy");
            bizSku.setSkuId(Long.valueOf(skuId));
            bizSku.setNum(sku.getNum());
            bizSku.setCategory(SKU_CATEGORY);
            BigDecimal price = skuPriceMap.get(skuId);
            bizSku.setPrice(price);
            bizSku.setName(productItem.getName());
            bizSku.setTax(TAX.multiply(new BigDecimal(102)));
            BigDecimal nakedPrice = price.divide(TAX.add(new BigDecimal(1)), BigDecimal.ROUND_HALF_UP);
            bizSku.setNakedPrice(nakedPrice);
            bizSku.setTaxPrice(price.subtract(nakedPrice));
            bizSku.setType(0);
            bizSku.setOid(0L);

            bizSkuList.add(bizSku);
        }
        jdSumbitOrderInfo.setSku(bizSkuList);

        response.setResult(jdSumbitOrderInfo);
        // 设置缓存，方便后面查询京东订单详情接口使用
        redisUtils.set(QYG_ORDER_PREFIX + jdOrderId, JSON.toJSONString(response), 604800);
        return response;
    }

    /**
     * 7.5 确认预占库存订单
     *
     * @return
     */
    @PostMapping("/api/order/confirmOrder")
    public JSONObject qygConfirmOrder(JdConfirmOrderRequest request) {
        JSONObject response = new JSONObject();
        response.put("success", true);
        response.put("resultMessage", "确认下单成功");
        response.put("resultCode", "0003");
        response.put("result", true);
        return response;
    }

    /**
     * 7.6 取消未确认订单
     */
    @PostMapping("/api/order/cancel")
    public JSONObject qygOrderCancel(JdCancelOrderRequest request) {
        JSONObject resposne = new JSONObject();
        resposne.put("success", true);
        resposne.put("resultMessage", "取消订单成功");
        resposne.put("resultCode", "0002");
        resposne.put("result", true);
        return resposne;
    }

    /**
     * 7.7 查询订单详情
     */
    @PostMapping("/api/order/selectJdOrder")
    public JdOrderDetailResponse qygOrderDetail(JdOrderDetailRequest request) {
        String jdOrderId = request.getJdOrderId();
        String queryExts = request.getQueryExts();
        String jdOrderInfoRedisValue = (String) redisUtils.get(QYG_ORDER_PREFIX + jdOrderId);
        JdSubmitOrderResponse orderInfo = null;
        if (!StringUtils.isEmpty(jdOrderInfoRedisValue)) {
            orderInfo = JSONObject.parseObject(jdOrderInfoRedisValue, JdSubmitOrderResponse.class);
        }
        JdOrderDetailResponse response = new JdOrderDetailResponse();
        response.setSuccess(true);
        response.setResultMessage("");
        response.setResultCode("0000");

        JSONObject result = new JSONObject();
        result.put("pOrder", 0);
        result.put("orderState", 1);
        result.put("jdOrderId", Long.valueOf(jdOrderId));
        result.put("state", 1);
        result.put("submitState", 1);
        result.put("type", 2); // 1-父订单，2-子订单
        if (orderInfo != null) {
            result.put("freight", orderInfo.getResult().getFreight());
            result.put("orderPrice", orderInfo.getResult().getOrderPrice());
            result.put("orderNakedPrice", orderInfo.getResult().getOrderNakedPrice());
            result.put("orderTaxPrice", orderInfo.getResult().getOrderTaxPrice());
            result.put("sku", orderInfo.getResult().getSku());
        } else {
            // 如果缓存里取不到值，就随便给个值
            result.put("freight", 0);
            result.put("orderPrice", 10);
            result.put("orderNakedPrice", 8);
            result.put("orderTaxPrice", 2);
            result.put("sku", new JSONObject());
        }
        if (queryExts != null && queryExts.contains("orderType")) {
            result.put("orderType", 1);
        }
        if (queryExts != null && queryExts.contains("createOrderTime")) {
            result.put("createOrderTime", DateUtils.convertDate(new Date(), "yyyy-MM-dd hh:mm:ss"));
        }
        if (queryExts != null && queryExts.contains("finishTime")) {
            result.put("finishTime", DateUtils.convertDate(new Date(), "yyyy-MM-dd hh:mm:ss"));
        }
        if (queryExts != null && queryExts.contains("jdOrderState")) {
            // 16-等待确认收货
            result.put("jdOrderState", 16);
        }
        result.put("paymentType", 4);

        response.setResult(result);
        return response;
    }

    /**
     * 7.8 查询配送信息
     */
    @PostMapping("/api/order/orderTrack")
    public JdOrderTrackResponse getOrderTrack(JdOrderTrackRequest request) throws ParseException {
        String jdOrderId = request.getJdOrderId();
        JdOrderTrackResponse response = new JdOrderTrackResponse();
        response.setSuccess(true);
        response.setResultMessage("");
        response.setResultCode("0000");

        JdOrderTrackResponse.OrderTrackInfo orderTrackInfo = new JdOrderTrackResponse.OrderTrackInfo();
        List<JdOrderTrackResponse.OrderTrack> orderTrackList = new ArrayList<>();
        List<JdOrderTrackResponse.WaybillCode> waybillCodeList = new ArrayList<>();

        JdOrderTrackResponse.OrderTrack orderTrack1 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack2 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack3 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack4 = new JdOrderTrackResponse.OrderTrack();
        JdOrderTrackResponse.OrderTrack orderTrack5 = new JdOrderTrackResponse.OrderTrack();

        JdOrderTrackResponse.WaybillCode waybillCode = new JdOrderTrackResponse.WaybillCode();

        orderTrack1.setContent("您提交了订单，请等待系统确认");
        orderTrack1.setMsgTime(DateUtils.convertStr("2022-03-01 23:27:13", "yyyy-MM-dd hh:mm:ss"));
        orderTrack1.setOperator("客户");

        orderTrack2.setContent("温馨提示：您的订单将由京东指定厂家为您发货，请您耐心等待");
        orderTrack2.setMsgTime(DateUtils.convertStr("2022-03-01 23:27:45", "yyyy-MM-dd hh:mm:ss"));
        orderTrack2.setOperator("系统");

        orderTrack3.setContent("您的订单正在出库，请耐心等待");
        orderTrack3.setMsgTime(DateUtils.convertStr("2022-03-02 15:22:34", "yyyy-MM-dd hh:mm:ss"));
        orderTrack3.setOperator("系统");

        orderTrack4.setContent("您的订单已交付京东快递, 运单号为JDVC12992534651");
        orderTrack4.setMsgTime(DateUtils.convertStr("2022-03-02 15:25:01", "yyyy-MM-dd hh:mm:ss"));
        orderTrack4.setOperator("系统");

        orderTrack5.setContent("您的订单开始配送");
        orderTrack5.setMsgTime(DateUtils.convertStr("2022-03-03 10:22:34", "yyyy-MM-dd hh:mm:ss"));
        orderTrack5.setOperator("京东快递");

        orderTrackList.add(orderTrack1);
        orderTrackList.add(orderTrack2);
        orderTrackList.add(orderTrack3);
        orderTrackList.add(orderTrack4);
        orderTrackList.add(orderTrack5);

        waybillCode.setOrderId(Long.valueOf(jdOrderId));
        waybillCode.setCarrier("京东快递");
        waybillCode.setDeliveryOrderId("JDVD02626228685");
        waybillCode.setParentId(0L);
        waybillCodeList.add(waybillCode);

        orderTrackInfo.setJdOrderId(jdOrderId);
        orderTrackInfo.setOrderTrack(orderTrackList);
        orderTrackInfo.setWaybillCode(waybillCodeList);

        response.setResult(orderTrackInfo);
        return response;
    }

    /**
     * 9.1 查询商品售后属性
     */
    @PostMapping("/api/afterSaleNew/getGoodsAttributes")
    public AfsGoodsAttributesResponse getAfterSaleNewGoodsAttributes(String token, String param) {
        AfsGoodsAttributesRequest.AfterOrderInfo afterOrderInfo = JSON.parseObject(param, AfsGoodsAttributesRequest.AfterOrderInfo.class);
        List<Long> wareIdList = afterOrderInfo.getWareIds();

        AfsGoodsAttributesResponse response = new AfsGoodsAttributesResponse();
        response.setSuccess(true);
        response.setResultMessage("操作成功");
        response.setResultCode("0000");

        List<AfsGoodsAttributesResponse.SupportedInfo> supportedInfoList = new ArrayList<>();
        for (Long wareId: wareIdList) {
            AfsGoodsAttributesResponse.SupportedInfo supportedInfo = new AfsGoodsAttributesResponse.SupportedInfo();
            // 支持的服务类型。10退货，20换货，30维修预期
            List<Integer> customerExpect = new ArrayList<>();
            customerExpect.add(10);
            customerExpect.add(20);
            customerExpect.add(30);

            // 支持的返回京东方式。4上门取件， 7客户送货， 40客户发货。
            List<Integer> pickupWareType = new ArrayList<>();
            pickupWareType.add(4);
            pickupWareType.add(7);
            pickupWareType.add(40);

            supportedInfo.setWareId(wareId);
            supportedInfo.setWareNum(1);
            supportedInfo.setCustomerExpect(customerExpect);
            supportedInfo.setPickupWareType(pickupWareType);
            supportedInfoList.add(supportedInfo);
        }

        response.setResult(supportedInfoList);
        return response;
    }

    /**
     * 9.2 申请售后
     */
    @PostMapping("/api/afterSaleNew/createAfsApply")
    public JdAfterSaleApplyResponse createAfsApply(String token, String param) {
        AfsApplyRequest.ApplyParam applyParam = JSON.parseObject(param, AfsApplyRequest.ApplyParam.class);
        String orderId = applyParam.getOrderId();
        // 设置缓存，给后续接口使用
        redisUtils.set("afs:apply:" + orderId, JSON.toJSONString(applyParam));
        JdAfterSaleApplyResponse response = new JdAfterSaleApplyResponse();
        response.setSuccess(true);
        response.setResultCode("0000");
        response.setResultMessage("操作成功");
        response.setResult(null);
        return response;
    }

    /**
     * 9.3 查询售后概要
     * @param queryExts canCancel 申请单是否可取消售后, canConfirm 申请单是否可以确认, canSendSku 申请单是否可以填写发运信息
     */
    @PostMapping("/api/afterSaleNew/getAfsOutline")
    public AfsOutlineResponse getAfsOutline(String token, String param, String queryExts) {
        AfsOutlineRequest.AfsOutlineParam afsOutlineParam = JSON.parseObject(param, AfsOutlineRequest.AfsOutlineParam.class);
        AfsOutlineResponse response = new AfsOutlineResponse();
        response.setSuccess(true);
        response.setResultMessage("操作成功");
        response.setResultCode("0000");

        AfsOutlineResponse.AfsOutlinePage afsOutlinePage = new AfsOutlineResponse.AfsOutlinePage();
        afsOutlinePage.setPageNo(1);
        afsOutlinePage.setPageNum(1L);
        afsOutlinePage.setPageSize(20);
        afsOutlinePage.setTotal(1L);

        List<AfsOutlineResponse.ApplyInfo> applyInfoList = new ArrayList<>();
        AfsOutlineResponse.ApplyInfo applyInfo = new AfsOutlineResponse.ApplyInfo();

        applyInfo.setOrderId(afsOutlineParam.getOrderId());
        applyInfo.setOriginalOrderId(afsOutlineParam.getOrderId());
        applyInfo.setThirdApplyId(afsOutlineParam.getThirdApplyId());
        applyInfo.setIsOffline(false);
        applyInfo.setNewPickWareType(40);
        applyInfo.setPin("北京量化派VOP");
        applyInfo.setApplyTime(DateUtils.convertDate(DateUtils.getBeforeDay(new Date(), 5), "yyyy-mm-dd hh:MM:ss"));
        applyInfo.setCompleteTime(DateUtils.convertDate(DateUtils.getBeforeDay(new Date(), 1), "yyyy-mm-dd hh:MM:ss"));
        applyInfo.setApplyStep(60); //10申请中待审核；20审核完成待收货；30收货完成待处理；40处理完成（如需退款则等待退款）；50待用户确认，60用户确认完成，70取消
        applyInfo.setApplyNum(1); // 原始申请数量
        applyInfo.setApplicationNum(0); // 申请中数量
        applyInfo.setWareServiceNum(0);
        applyInfo.setConfirmNum(0);
        applyInfo.setWareCompleteNum(1);
        applyInfo.setWareCancelNum(0);
        applyInfo.setRefundNum(1);
        applyInfo.setShouldRefundAmount(new BigDecimal("10"));
        applyInfo.setNakedPriceAmount(new BigDecimal("10"));
        applyInfo.setRealRefundAmount(new BigDecimal("10"));
        applyInfo.setIsComplete(1);
        applyInfo.setCanConfirm(true);
        applyInfo.setCanSendSku(true);

        applyInfoList.add(applyInfo);
        afsOutlinePage.setData(applyInfoList);
        response.setResult(afsOutlinePage);

        redisUtils.set("afs:outline:" + afsOutlineParam.getOrderId(), JSON.toJSONString(afsOutlineParam));

        return response;
    }

    /**
     * 9.8 查询售后申请单明细
     */
    @PostMapping("/api/afterSaleNew/getApplyDetailInfo")
    public AfsApplyDetailResponse getApplyDetailInfo(String token, String param) {
        AfsApplyDetailRequest.ApplyDetailParam applyDetailParam = new AfsApplyDetailRequest.ApplyDetailParam();
        AfsApplyDetailResponse response = new AfsApplyDetailResponse();
        response.setSuccess(true);
        response.setResultMessage("操作成功");
        response.setResultCode("0000");

        AfsApplyDetailResponse.AfsApplyDetailDto applyDetailDto = new AfsApplyDetailResponse.AfsApplyDetailDto();
        applyDetailDto.setRefundInfoResultDtos(null);

        List<AfsApplyDetailResponse.AddressInfoDto> addressInfoDtoList = new ArrayList<>();
        AfsApplyDetailResponse.AddressInfoDto addressInfoDto = new AfsApplyDetailResponse.AddressInfoDto();

        // 设置AddressInfoDto
        addressInfoDto.setCustomerPin("北京量化派VOP");
        addressInfoDto.setNewOrderId(null);
        addressInfoDto.setThirdApplyId(applyDetailParam.getThirdApplyId());
        addressInfoDto.setOriginalOrderId(applyDetailParam.getOrderId());
        addressInfoDto.setOrderId(applyDetailParam.getOrderId());
        addressInfoDto.setAfterserviceType(30);
        addressInfoDto.setAfterserviceProvince(0);
        addressInfoDto.setAfterserviceCity(0);
        addressInfoDto.setAfterserviceCounty(0);
        addressInfoDto.setAfterserviceVillage(0);
        addressInfoDto.setAfterserviceAddress("北京市市辖区海淀区中国电子大厦B座");
        addressInfoDto.setAfterserviceReceiver("WTJD售后组");
        addressInfoDto.setAfterserviceTel("13691060671");
        addressInfoDto.setAfterserviceZipcode("");
        addressInfoDto.setAfterservicePhone("");

        addressInfoDtoList.add(addressInfoDto);

        applyDetailDto.setAddressInfoDtos(addressInfoDtoList);

        response.setResult(applyDetailDto);
        return response;
    }

    /**
     * 11.1 查询推送消息
     *
     * @return
     */
    @PostMapping("/api/message/get")
    public JdCompanyMsgResponse qygMessageGet(JdCompanyMsgRequest request) {
        String type = request.getType();
        JdCompanyMsgResponse response = new JdCompanyMsgResponse();
        response.setSuccess(true);
        response.setResultMessage("");
        response.setResultCode("0000");
        List<Map<String, Object>> result = new ArrayList<>();
        switch (type) {
            case "10":
                // TODO
                break;
            case "31":
                // TODO
                break;
            case "104":
                Set<String> set = redisUtils.keys("afs:outline:*");
                if (!set.isEmpty()) {
                    for (String afsOutlineStr: set) {
                        AfsOutlineRequest.AfsOutlineParam afsOutlineParam = JSON.parseObject(afsOutlineStr, AfsOutlineRequest.AfsOutlineParam.class);
                        Map<String, Object> map = new HashMap<>();
                        map.put("id", RandomStringUtils.randomNumeric(16));
                        map.put("time", DateUtils.convertDate(new Date(), "yyyy-mm-dd hh:MM:ss"));
                        map.put("type", 104);
                        Map<String, Object> childMap = new HashMap<>();
                        childMap.put("expectationChanged", true);
                        childMap.put("thirdApplyId", afsOutlineParam.getThirdApplyId());
                        childMap.put("pin", "北京量化派VOP");
                        childMap.put("stepPassType", 10);
                        childMap.put("isOffline", false);
                        childMap.put("applyStep", 60);
                        childMap.put("contractNumber", "VOP-20201209182245");
                        childMap.put("orderId", afsOutlineParam.getOrderId());
                        map.put("result", childMap);

                        // 处理完就删除
                        redisUtils.del("afs:outline:" + afsOutlineParam.getOrderId());
                        result.add(map);
                    }
                }
                break;
            default:
                break;
        }
        response.setResult(result);
        return response;
    }


    /**
     * 列表转Map，key为skuId，value是price
     *
     * @param orderPriceSnaps JdSkuPriceInfo列表
     * @return
     */
    public Map<String, BigDecimal> convertSkuPriceMap(List<SubmitOrderRequest.JdSkuPriceInfo> orderPriceSnaps) {
        Map<String, BigDecimal> map = new HashMap<>();
        for (SubmitOrderRequest.JdSkuPriceInfo jdSkuPriceInfo : orderPriceSnaps) {
            map.put(jdSkuPriceInfo.getSkuId(), jdSkuPriceInfo.getPrice());
        }
        return map;
    }

    /**
     * 计算运费
     *
     * @param totalPrice 总价格
     * @return
     */
    public BigDecimal calculateFreight(BigDecimal totalPrice) {
        if (totalPrice.compareTo(new BigDecimal("99")) >= 0) {
            return new BigDecimal("0");
        } else {
            return new BigDecimal("6");
        }
    }
}
