package cn.qg.holmes.aspect;

import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.core.JsonProcessingException;
import org.apache.commons.io.IOUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Map;

/**
 * 日志切面
 */
@Aspect
@Component
public class HttpLogAspect {

    private static final Logger logger = LoggerFactory.getLogger(HttpLogAspect.class);

    @Pointcut("execution(public * cn.qg.holmes.controller.mock.*(..))")
    public void httpRequestLog() {}

    /**
     * 前置通知：在连接点之前执行的通知
     *
     * @param joinPoint
     * @throws Throwable
     */
    @Before("httpRequestLog()")
    public void doBefore(JoinPoint joinPoint) throws JsonProcessingException {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();
        logger.info("请求发起ip地址: {}, 请求url: {}, 请求参数：{}", request.getRemoteAddr(), request.getRequestURI(), getQuery(request));
    }

    /**
     * 后置通知：在连接点之后执行的通知
     *
     * @param ret
     * @throws Throwable
     */
    @AfterReturning(returning = "ret", pointcut = "httpRequestLog()")
    public void doAfterReturning(Object ret) throws Throwable {
        // 处理完请求，返回内容
        logger.info("response:" + JSON.toJSONString(ret));
    }

    private String getQuery(HttpServletRequest request) {
        if (isPostPutRequest(request)) {
            return getParameters(request) + getRequestBody(request);
        } else {
            return getParameters(request);
        }
    }

    private boolean isPostPutRequest(HttpServletRequest request) {
        return "POST".equalsIgnoreCase(request.getMethod()) || "PUT".equalsIgnoreCase(request.getMethod());
    }

    private String getParameters(HttpServletRequest request) {
        StringBuilder sb = new StringBuilder();
        Map<String, String[]> params = request.getParameterMap();
        for (String key : params.keySet()) {
            sb.append('[')
                    .append(key)
                    .append('=')
                    .append(request.getParameter(key))
                    .append("] ");
        }

        return sb.toString();
    }

    private String getRequestBody(HttpServletRequest request) {
        try {
            return IOUtils.toString(request.getInputStream(), String.valueOf(Charset.defaultCharset()));
        } catch (IOException e) {
            return "";
        }
    }
}
