package cn.qg.holmes.controller.monitor;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.monitor.SqlMonitorDataSource;
import cn.qg.holmes.entity.monitor.SqlMonitorTask;
import cn.qg.holmes.service.monitor.SqlMonitorDataSourceService;
import cn.qg.holmes.service.monitor.SqlMonitorTaskService;
import cn.qg.holmes.utils.DingdingUtils;
import cn.qg.holmes.utils.JdbcUtils;
import cn.qg.holmes.utils.QuartzUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.apache.commons.lang3.StringUtils;
import org.quartz.Scheduler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 定时任务Controller
 */
@CrossOrigin
@RestController
@RequestMapping("/sql/monitor")
public class SqlMonitorTaskController {

    @Autowired
    SqlMonitorTaskService sqlMonitorTaskService;

    @Autowired
    SqlMonitorDataSourceService sqlMonitorDataSourceService;

    @Autowired
    JdbcTemplate jdbcTemplate;

    @Autowired
    private Scheduler scheduler;

    /**
     * 获取监控列表，带搜索和分页
     * @param pageNum 第几页
     * @param pageSize 每页多少个
     * @param serviceName 服务名
     * @param taskName 监控名称
     * @return
     */
    @GetMapping("/list")
    public JsonResult getMonitorTaskList(@RequestParam(defaultValue = "1") Integer pageNum,
                                         @RequestParam(defaultValue = "10") Integer pageSize,
                                         @RequestParam String serviceName,
                                         @RequestParam String taskName) {
        QueryWrapper<SqlMonitorTask> queryWrapper = new QueryWrapper<>();
        if (!StringUtils.isEmpty(serviceName)) {
            queryWrapper.eq("service_name", serviceName);
        }
        if (!StringUtils.isEmpty(taskName)) {
            queryWrapper.eq("task_name", taskName);
        }
        queryWrapper.orderByDesc("id");
        IPage<SqlMonitorTask> pageParam = new Page<>(pageNum, pageSize);
        IPage<SqlMonitorTask> monitorTaskIPage = sqlMonitorTaskService.page(pageParam, queryWrapper);
        List<SqlMonitorTask> sqlMonitorTaskList = monitorTaskIPage.getRecords();
        for (SqlMonitorTask sqlMonitorTask : sqlMonitorTaskList) {
            sqlMonitorTask.setDsName(sqlMonitorDataSourceService.getById(sqlMonitorTask.getDsId()).getName());
        }
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("list", sqlMonitorTaskList);
        resultMap.put("total", monitorTaskIPage.getTotal());
        return JsonResult.buildSuccessResult(resultMap);
    }

    /**
     * 新增监控
     *
     * @param sqlMonitorTask
     * @return
     */
    @PostMapping("/add")
    public JsonResult addMonitorTask(@RequestBody SqlMonitorTask sqlMonitorTask) {
        QueryWrapper<SqlMonitorTask> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("task_name", sqlMonitorTask.getTaskName());
        if (sqlMonitorTaskService.getOne(queryWrapper) != null) {
            return JsonResult.buildErrorStateResult("监控名称重复", false);
        }
        boolean result = sqlMonitorTaskService.save(sqlMonitorTask);
        if (result) {
            QuartzUtils.createSqlScheduleJob(scheduler, sqlMonitorTask);
            return JsonResult.buildSuccessResult("监控创建成功！");
        }
        return JsonResult.buildErrorStateResult("监控创建失败！", false);
    }

    /**
     * 修改监控
     *
     * @param sqlMonitorTask
     * @return
     */
    @PostMapping("/edit")
    public JsonResult editMonitorTask(@RequestBody SqlMonitorTask sqlMonitorTask) {
        QueryWrapper<SqlMonitorTask> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("task_name", sqlMonitorTask.getTaskName());
        SqlMonitorTask sqlMonitorTaskDb = sqlMonitorTaskService.getOne(queryWrapper);
        if (sqlMonitorTaskDb != null && !sqlMonitorTaskDb.getId().equals(sqlMonitorTask.getId())) {
            return JsonResult.buildErrorStateResult("监控名称重复", false);
        }
        boolean updateResult = sqlMonitorTaskService.saveOrUpdate(sqlMonitorTask);
        if (updateResult) {
            // 如果状态发生了变更，则根据改变之后的状态来决定是暂停任务还是重启任务
            if (!sqlMonitorTask.getStatus().equals(sqlMonitorTaskDb.getStatus())) {
                // 如果变更后的状态为0
                if (sqlMonitorTask.getStatus() == 0) {
                    QuartzUtils.pauseScheduleJob(scheduler, sqlMonitorTask.getTaskName());
                }
                // 如果变更后的状态为1
                if (sqlMonitorTask.getStatus() == 1) {
                    QuartzUtils.resumeScheduleJob(scheduler, sqlMonitorTask.getTaskName());
                }
            }
            return JsonResult.buildSuccessResult("监控修改成功！");
        }
        return JsonResult.buildErrorStateResult("监控修改失败", false);
    }

    /**
     * 删除监控
     *
     * @param sqlMonitorTask
     * @return
     */
    @PostMapping("/delete")
    public JsonResult deleteMonitorTask(@RequestBody SqlMonitorTask sqlMonitorTask) {
        Integer taskId = sqlMonitorTask.getId();
        if (sqlMonitorTaskService.getById(taskId) == null) {
            return JsonResult.buildErrorStateResult("定时任务不存在!", false);
        }
        boolean delResult = sqlMonitorTaskService.removeById(taskId);
        if (delResult) {
            QuartzUtils.deleteScheduleJob(scheduler, sqlMonitorTask.getTaskName());
            return JsonResult.buildSuccessResult("删除成功!");
        } else {
            return JsonResult.buildErrorStateResult("删除失败!", false);
        }
    }

    /**
     * 立即执行SQL
     *
     * @param dsId 数据源id
     * @param sql  SQL
     * @return
     */
    @PostMapping("/execute")
    public JsonResult executeSql(@RequestParam Integer dsId, @RequestParam String sql) {
        SqlMonitorDataSource sqlMonitorDataSource = sqlMonitorDataSourceService.getById(dsId);
        List<Map<String, Object>> resultList = JdbcUtils.queryForList(sqlMonitorDataSource.getHost(), sqlMonitorDataSource.getPort(), sqlMonitorDataSource.getUsername(),
                sqlMonitorDataSource.getPassword(), null, sql);
        return JsonResult.buildSuccessResult(resultList);
    }

    /**
     * 发送测试，立即测试下定时任务发送消息是否准确
     *
     * @param sqlMonitorTask 监控任务
     * @return
     */
    @PostMapping("/test")
    public JsonResult testMonitorTask(@RequestBody SqlMonitorTask sqlMonitorTask) {
        try {
            Integer dsId = sqlMonitorTask.getDsId();
            SqlMonitorDataSource sqlMonitorDataSource = sqlMonitorDataSourceService.getById(dsId);
            List<Map<String, Object>> resultList = JdbcUtils.queryForList(sqlMonitorDataSource.getHost(), sqlMonitorDataSource.getPort(),
                    sqlMonitorDataSource.getUsername(), sqlMonitorDataSource.getPassword(), null, sqlMonitorTask.getSql());
            String dingUrl = sqlMonitorTask.getDingUrl();
            String dingText = sqlMonitorTask.getDingText();
            dingText = dingText.replaceAll("\\{count}", String.valueOf(resultList.size()));
            String markdownMsg = DingdingUtils.buildMarkdownMsg("监控测试", dingText + "\n" + JSON.toJSONString(resultList), false);
            boolean result = DingdingUtils.sendToDingding(markdownMsg, dingUrl);
            return JsonResult.buildSuccessResult(result ? "发送成功！" : "发送失败！", true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("发送异常！", false);
        }
    }

    /**
     * 定时任务执行策略
     * @return
     */
    @GetMapping("/strategy")
    public JsonResult getSqlMonitorStrategy() {
        List<Map<String, String>> strategyList = new ArrayList<>();
        Map<String, String> dailyMap = new HashMap<>();
        Map<String, String> hourlyMap = new HashMap<>();
        hourlyMap.put("value", "0 0 * * * ?");
        hourlyMap.put("label", "每小时执行一次");
        dailyMap.put("value", "0 0 10 * * ?");
        dailyMap.put("label", "每天10点执行一次");
        Map<String, String> weeklyMap = new HashMap<>();
        weeklyMap.put("value", "0 0 10 ? * 2");
        weeklyMap.put("label", "每周一上午10点执行一次");
        strategyList.add(hourlyMap);
        strategyList.add(dailyMap);
        strategyList.add(weeklyMap);
        return JsonResult.buildSuccessResult(strategyList);
    }

    /**
     * 启用/禁用定时任务
     * @param taskId 定时任务id
     * @param status 状态，0-禁用，1-启用
     * @return
     */
    @PostMapping("/enable")
    public JsonResult modifyTaskStatus(Integer taskId, Integer status) {
        SqlMonitorTask sqlMonitorTask = sqlMonitorTaskService.getById(taskId);
        if (sqlMonitorTask == null) {
            return JsonResult.buildErrorStateResult("监控不存在！", false);
        }
        if (status == 1) {
            if (sqlMonitorTask.getStatus() == 1) {
                return JsonResult.buildErrorStateResult("监控已是启用状态", false);
            } else {
                sqlMonitorTask.setStatus(1);
                sqlMonitorTaskService.updateById(sqlMonitorTask);
                QuartzUtils.resumeScheduleJob(scheduler, sqlMonitorTask.getTaskName());
                return JsonResult.buildSuccessResult("启用成功！", true);
            }
        }
        if (status == 0) {
            if (sqlMonitorTask.getStatus() == 0) {
                return JsonResult.buildErrorStateResult("监控已是禁用状态", false);
            } else {
                sqlMonitorTask.setStatus(0);
                sqlMonitorTaskService.updateById(sqlMonitorTask);
                QuartzUtils.pauseScheduleJob(scheduler, sqlMonitorTask.getTaskName());
                return JsonResult.buildSuccessResult("禁用成功！", true);
            }
        }
        return JsonResult.buildErrorStateResult("status只能传0或1", false);
    }
}
