package cn.qg.holmes.service.auto.impl;

import cn.qg.holmes.service.auto.AutoUtilsService;
import cn.qg.holmes.utils.HttpClientUtils;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author libo
 * created at 2021-03-30
 */
@Service
@Slf4j
public class AutoUtilsServiceImpl implements AutoUtilsService {

    @Autowired
    RedisUtils redisUtils;

    /**
     * 参数替换-新
     * @param parameters headers或者parameters
     * @param uuid 唯一标识
     */
    @Override
    public Map<String, String> replaceVariables(String parameters, UUID uuid) {
        Map<String, String> parameterMap = JSON.parseObject(parameters, Map.class);
        log.info("替换之前的参数：{}", parameterMap);
        for (String key: parameterMap.keySet()) {
            String value = parameterMap.get(key);
            if (value.startsWith("$")) {
                parameterMap.put(key, redisUtils.get(uuid + "_" + value.substring(1)).toString());
            }
        }
        log.info("替换之后的参数：{}", parameterMap);
        return parameterMap;
    }

    /**
     * 新的参数替换，不再从redis里取值，而是从全局Map中取值
     * @param parameters
     * @param globalMap
     * @return
     */
    @Override
    public Map<String, String> replaceVariablesNew(String parameters, Map<String, Object> globalMap) {
        Map<String, String> parameterMap = JSON.parseObject(parameters, Map.class);
        log.info("替换之前的参数：{}", parameterMap);
        for (String key: parameterMap.keySet()) {
            String value = parameterMap.get(key);
            if (value.startsWith("$")) {
                parameterMap.put(key, globalMap.get(value.substring(1)).toString());
            }
        }
        log.info("替换之后的参数：{}", parameterMap);
        return parameterMap;
    }

    /**
     * 将响应中的值解析出来并存储到redis值
     * @param response 响应json
     * @param extract 解析的字段及保存到redis的key
     * @return true-解析成功，false-解析失败
     */
    public boolean extractResponse(String response, String extract, UUID uuid) {
        Map<String, String> extractMap = JSON.parseObject(extract, Map.class);
        try {
            for (String key: extractMap.keySet()) {
                String value = extractMap.get(key);
                String redisKey = uuid + "_" + key;
                String redisValue = "";
                if (value.startsWith("$")) {
                    redisValue = JsonPath.read(response, value);
                } else {
                    Pattern pattern = Pattern.compile(value);
                    Matcher matcher = pattern.matcher(response);
                    if (matcher.find()) {
                        redisValue = matcher.group(1);
                    } else {
                        return false;
                    }
                }
//                redisUtils.set(redisKey, JsonPath.read(response, value));
                redisUtils.set(redisKey, redisValue);
                redisUtils.expire(redisKey, 120);
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    @Override
    public Map<String, Object> extractResponseNew(String response, String extract) {
        Map<String, Object> resultMap = new HashMap<>();
        try {
            Map<String, String> extractMap = JSON.parseObject(extract, Map.class);
            for (String key: extractMap.keySet()) {
                String value = extractMap.get(key);
                String extractValue = "";
                // 如果以$开头，则使用JsonPath解析
                if (value.startsWith("$")) {
                    extractValue = JsonPath.read(response, value);
                } else {
                    Pattern pattern = Pattern.compile(value);
                    Matcher matcher = pattern.matcher(response);
                    if (matcher.find()) {
                        extractValue = matcher.group(1);
                    } else {
                        return null;
                    }
                }
                resultMap.put(key, extractValue);
            }
            return resultMap;
        } catch (Exception e) {
            e.printStackTrace();
            return resultMap;
        }
    }

    /**
     * 响应断言
     * @return
     */
    public boolean assertResponse(String response, List<Map> validateList) {
        for (Map<String, String> map: validateList) {
            String comparator = map.get("comparator");
            String checkpoint = JsonPath.read(response, map.get("check")).toString();
            String expectValue = map.get("expect");
            boolean result = singleAssertResponse(comparator, checkpoint, expectValue);
            if (!result) {
                return false;
            }
        }
        return true;
    }

    /**
     *
     * @param namespace
     * @param preString
     * @return
     */
    @Override
    public Map<String, Object> handlePreCondition(String namespace, String preString, Map<String, Object> globalMap) {
        log.info("开始处理前置动作！");
        Map<String, Object> resultMap = new HashMap<>();
        try {
            List<Map> preList = JSON.parseArray(preString, Map.class);
            for (Map<String, Object> map: preList) {
                String db = map.get("db").toString();
                String sql = map.get("sql").toString();
                String key = null;
                String type = null;
                if (map.get("key") != null) {
                    key = map.get("key").toString();
                }
                if (map.get("type") != null) {
                    type = map.get("type").toString();
                }
                // 确认SQL中是否需要变量替换
                String regex = "(\\{\\{[A-Za-z0-9]+}})";
                Pattern pattern = Pattern.compile(regex);
                Matcher matcher = pattern.matcher(sql);
                while (matcher.find()) {
                    String replace = matcher.group();
                    // 去掉首位的{{和}}
                    String tempKey = replace.substring(2, replace.length() -2);
                    sql = sql.replace(replace, "'" + globalMap.get(tempKey).toString() + "'");
                }
                Map<String, String> params = new HashMap<>();
                params.put("namespace", namespace);
                params.put("database", db);
                params.put("sql", sql);
                params.put("type", type);
                String url = "https://qa-platform-yxm.liangkebang.net/sql/execute";
                String result = HttpClientUtils.doGet(url, params);
                // 只有select才需要加入Map
                if (sql.startsWith("select") || sql.startsWith("SELECT")) {
                    if (type.equals("map")) {
                        resultMap.put(key, JsonPath.read(result, "$.data." + key));
                    }  else if (type.equals("list")) {
                        resultMap.put(key, JsonPath.read(result, "$.data."));
                    }
                }
            }
            log.info("前置动作获取到的值：{}", resultMap);
            return resultMap;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     *
     * @param comparator
     * @param checkpoint
     * @param expectValue
     * @return
     */
    public boolean singleAssertResponse(String comparator, String checkpoint, String expectValue) {
        switch (comparator) {
            case "eq":
                return checkpoint.equals(expectValue);
            case "gt":
                return Integer.parseInt(checkpoint) > Integer.parseInt(expectValue);
            case "lt":
                return Integer.parseInt(checkpoint) < Integer.parseInt(expectValue);
            case "neq":
                return !checkpoint.equals(expectValue);
            default:
                log.info("暂不支持该比较符号：{}", comparator);
                break;
        }
        return false;
    }
}
