package cn.qg.holmes.controller.effect;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.service.effect.DatabaseSyncService;
import cn.qg.holmes.service.k8s.K8sService;
import cn.qg.holmes.utils.JenkinsService;
import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.expression.JdbcNamedParameter;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/dbsync")
public class DbSyncController {

    @Autowired
    DatabaseSyncService databaseSyncService;

    @Autowired
    JenkinsService jenkinsService;

    @Value("${dbsync.mysql.ip}")
    private String ip;

    @Value("${dbsync.mysql.port}")
    private String port;

    @Value("${dbsync.mysql.username}")
    private String username;

    @Value("${dbsync.mysql.password}")
    private String password;

    @Autowired
    K8sService k8sService;

    /**
     * 同步数据库
     * @param namespace 环境
     * @param dbName 数据库名称
     * @param tableName 表名称
     * @param businessData 是否保留业务数据
     * @return
     */
    @GetMapping("/new")
    public JsonResult syncDatabase(@RequestParam String namespace, @RequestParam String dbName, @RequestParam String tableName, @RequestParam boolean businessData) {
        try {
            long startTime = System.currentTimeMillis();
            Map<String, String> map = k8sService.getMysqlAddressByNamespace(namespace);
            String destIp = map.get("host");
            String destPort = map.get("port");
            log.info("获取到{}环境的Mysql地址为：{}", namespace, destIp + ":" + destPort);
            if (!StringUtils.isEmpty(dbName)) {
                // 同步所有数据库
                if (dbName.equalsIgnoreCase("all")) {
                    List<String> dbList = databaseSyncService.getDatabaseList(ip, port, username, password);
                    databaseSyncService.getDbInfoFromSource(ip, port, username, password, dbName);
                    databaseSyncService.syncDbToDest(destIp, destPort, "qa", "qatest", dbName, namespace, businessData);
                } else {
                    // 同步单个数据库或单张表
                    if (tableName.equalsIgnoreCase("all") || tableName.equals("")) {
                        log.info("开始同步{}库下所有表到{}环境，保留业务数据：{}", dbName, namespace, businessData);
                        databaseSyncService.getDbInfoFromSource(ip, port, username, password, dbName);
                        databaseSyncService.syncDbToDest(destIp, destPort, "qa", "qatest", dbName, namespace, businessData);
                    } else {
                        log.info("开始同步{}库下{}表到{}环境，保留业务数据：{}", dbName, tableName, namespace, businessData);
                        databaseSyncService.getSingleTableFromSource(ip, port, username, password, dbName, tableName);
                        databaseSyncService.syncSingleTableToDest(destIp, destPort, "qa", "qatest", dbName, tableName, businessData);
                    }
                }
            }
            long endTime = System.currentTimeMillis();
            long elapsedTime = (endTime - startTime) / 1000;
            return JsonResult.buildSuccessResult(elapsedTime + "s");
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("同步失败！", false);
        }
    }

    /**
     * 获取数据库列表
     * @return
     */
    @GetMapping("/dbs")
    public JsonResult getDatabaseList() {
        return JsonResult.buildSuccessResult(databaseSyncService.getDatabaseList(ip, port, username, password));
    }

    /**
     * 获取某个库下的表列表
     * @param dbName 数据库名
     * @return
     */
    @GetMapping("/tables")
    public JsonResult getTableList(@RequestParam String dbName) {
        List<String> tableList = databaseSyncService.getTableListByDb(ip, port, username, password, dbName);
        tableList.add(0, "all");
        return JsonResult.buildSuccessResult(tableList);
    }

    /**
     * 跟数据库同步脚本使用的接口
     * @param database 数据库名
     * @return
     */
    @GetMapping("/tableList")
    public String getTableListForSync(@RequestParam String database) {
        List<String> tableList = databaseSyncService.getTableListByDb(ip, port, username, password, database);
        return StringUtils.join(tableList, " ");
    }

    /**
     * 同步表结构
     * @param ip
     * @param port
     * @param database
     * @param table
     * @return
     */
    @GetMapping("/table/structure")
    public JsonResult handleTableStructure(@RequestParam String ip, @RequestParam String port, @RequestParam String database, @RequestParam String table) {
        try {
            databaseSyncService.getSourceDbStructure(ip, port, username, password, database, table);
            databaseSyncService.syncDbStructureToDest(ip, port, "qa", "qatest", database, table);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("创建表结构失败！", false);
        }
        return JsonResult.buildSuccessResult("创建" + ip + ":" + port + "环境表结构成功！");
    }

    /**
     * 老数据库同步，调用Jenkins job sync_database_schema
     * @param dbName 数据库名
     * @param host 数据库
     * @param port 端口号
     * @param replaceDomain 是否替换域名
     * @param notDeleteBusinessData 是否删除业务数据
     * @param useCache 是否使用两个小时之前的备份
     * @return
     */
    @PostMapping("/old")
    public JsonResult dbSyncOld(String dbName, String host, String port, boolean replaceDomain, boolean notDeleteBusinessData, boolean useCache) {
        int buildNum = jenkinsService.buildDbSyncJob(dbName, host, port, replaceDomain, notDeleteBusinessData,  useCache);
        if (buildNum == 0)  {
            return JsonResult.buildErrorStateResult("数据库同步失败！", 0);
        } else {
            return JsonResult.buildSuccessResult("开始同步数据库！", buildNum);
        }
    }

    @GetMapping("/query")
    public JsonResult queryDbSyncResult(String jobName, int buildNum) {
        try {
            Map<String, String> result = jenkinsService.getBuildInfo(jobName, buildNum);
            return JsonResult.buildSuccessResult(result);
        }catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("查询失败", false);
        }
    }
}
