package cn.qg.holmes.service.k8s;

import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import cn.qg.holmes.utils.DateUtils;
import cn.qg.holmes.utils.FileUtils;
import cn.qg.holmes.utils.TkeUtils;
import cn.qg.holmes.utils.YamlUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.fabric8.kubernetes.api.model.*;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.extensions.Ingress;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;

import java.io.IOException;
import java.util.*;

/**
 * TKE qa集群k8s相关操作
 */
@Slf4j
@Component
public class TkeService {

    @Value("${no.healthcheck.service}")
    private String noHealthCheckService;

    @Value("${gyl.healthcheck.service}")
    private String gylHealthCheckService;

    @Autowired
    DockerProjectService dockerProjectService;

    private KubernetesClient kubernetesClient;

    public TkeService() {
        try {
            String configYAML = FileUtils.readFileFromClassPathResource("tke/kube-config.yml");
            Config config = Config.fromKubeconfig(configYAML);
            kubernetesClient = new DefaultKubernetesClient(config);
            String configCrt = FileUtils.readFileFromClassPathResource("tke/tke-cluster-ca.crt");
            config.setCaCertData(configCrt);
            log.info("TKE qa集群k8s连接成功！");
        } catch (Exception e) {
            log.info("TKE qa集群k8s连接失败！");
            e.printStackTrace();
        }
    }

    /**
     * 创建新的namespace
     *
     * @param name  namespace名称
     * @param desc  dev/test
     * @param owner
     * @return
     */
    public boolean createNewNamespace(String name, String desc, String owner) {
        try {
            ObjectMeta objectMeta = new ObjectMeta();
            Map<String, String> annotations = new HashMap<>();
            annotations.put("description", desc);
            annotations.put("owner", owner);
            objectMeta.setAnnotations(annotations);
            objectMeta.setName(name);

            NamespaceSpec namespaceSpec = new NamespaceSpec();
            List<String> finalizers = new ArrayList<>();
            finalizers.add("kubernetes");
            namespaceSpec.setFinalizers(finalizers);

            NamespaceStatus namespaceStatus = new NamespaceStatus();
            namespaceStatus.setPhase("Active");

            Namespace namespace = new NamespaceBuilder()
                    .withApiVersion("v1")
                    .withKind("Namespace")
                    .withMetadata(objectMeta)
                    .withSpec(namespaceSpec)
                    .withStatus(namespaceStatus)
                    .build();
            log.info("开始创建新的环境: {}", namespace.toString());
            Namespace namespaceResponse = kubernetesClient.namespaces().create(namespace);
            log.info("创建环境返回结果：{}", namespaceResponse.toString());
            createQcloudRegistryKeySecret(name);
            createTencenthHubKeySecret(name);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    /**
     * 创建TKE qcloudregistrykey类型 Secret
     * @param namespace
     * @return
     */
    public Secret createQcloudRegistryKeySecret(String namespace) {

        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "qcloudregistrykey");

        ObjectMeta objectMeta = new ObjectMetaBuilder()
                .withLabels(labels)
                .withName("tencenthubkey")
                .withNamespace(namespace)
                .build();

        Map<String, String> data = new HashMap<>();
        data.put(".dockercfg", "eyJjY3IuY2NzLnRlbmNlbnR5dW4uY29tIjp7InVzZXJuYW1lIjoiMTAwMDA4NjMyMjY5IiwicGFzc3dvcmQiOiJ7QXBwbGljYXRpb25Ub2tlbjo0MTM1NWY2OWQyY2U3ZWUyYmE2YjQxZmMxNzJlNzM0MH0iLCJlbWFpbCI6IjEwMDAwODYzMjI2OUBxcS5jb20iLCJhdXRoIjoiTVRBd01EQTROak15TWpZNU9udEJjSEJzYVdOaGRHbHZibFJ2YTJWdU9qUXhNelUxWmpZNVpESmpaVGRsWlRKaVlUWmlOREZtWXpFM01tVTNNelF3ZlE9PSJ9fQ==");

        Secret secret = new SecretBuilder()
                .withApiVersion("v1")
                .withKind("Secret")
                .withType("kubernetes.io/dockercfg")
                .withMetadata(objectMeta)
                .withData(data)
                .build();
        log.info("{}环境，创建新的secret：qcloudregistrykey", namespace);
        return kubernetesClient.secrets().inNamespace(namespace).create(secret);
    }

    /**
     * 创建TKE tencenthubkey 类型 Secret
     * @param namespace
     * @return
     */
    public Secret createTencenthHubKeySecret(String namespace) {

        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "tencenthubkey");
        ObjectMeta objectMeta = new ObjectMetaBuilder()
                .withLabels(labels)
                .withName("tencenthubkey")
                .withNamespace(namespace)
                .build();

        Map<String, String> data = new HashMap<>();
        data.put(".dockercfg", "eyJodWIudGVuY2VudHl1bi5jb20iOnsidXNlcm5hbWUiOiIxMDAwMDg2MzIyNjkiLCJwYXNzd29yZCI6ImV5SmhiR2NpT2lKSVV6STFOaUlzSW5SNWNDSTZJa3BYVkNKOS5leUpsZUhBaU9qRTROell5T1RFME9EVXNJbXAwYVNJNklqVTVPVGRqTmpkaExUWmpaV1l0TkRGbE55MDVPRGsyTFdGaE16TTFPRGRpWW1ReE1TSXNJbWxoZENJNk1UVTJNRGt6TVRRNE5Td2libUptSWpveE5UWXdPVE14TkRnMUxDSnpkV0lpT2lJeE1EQXdNRGcyTXpJeU5qa2lmUS5ZWEl4TzBaWTg3QmNzSktLdW9lQ3gzYllLZ1RzMktNVElSeDd5dWVRdDZNIiwiZW1haWwiOiIxMDAwMDg2MzIyNjlAcXEuY29tIiwiYXV0aCI6Ik1UQXdNREE0TmpNeU1qWTVPbVY1U21oaVIyTnBUMmxLU1ZWNlNURk9hVWx6U1c1U05XTkRTVFpKYTNCWVZrTktPUzVsZVVwc1pVaEJhVTlxUlRST2VsbDVUMVJGTUU5RVZYTkpiWEF3WVZOSk5rbHFWVFZQVkdScVRtcGthRXhVV21wYVYxbDBUa1JHYkU1NU1EVlBSR3N5VEZkR2FFMTZUVEZQUkdScFdXMVJlRTFUU1hOSmJXeG9aRU5KTmsxVVZUSk5SR3Q2VFZSUk5FNVRkMmxpYlVwdFNXcHZlRTVVV1hkUFZFMTRUa1JuTVV4RFNucGtWMGxwVDJsSmVFMUVRWGROUkdjeVRYcEplVTVxYTJsbVVTNVpXRWw0VHpCYVdUZzNRbU56U2t0TGRXOWxRM2d6WWxsTFoxUnpNa3ROVkVsU2VEZDVkV1ZSZERaTiJ9fQ==");

        Secret secret = new SecretBuilder()
                .withApiVersion("v1")
                .withKind("Secret")
                .withType("kubernetes.io/dockercfg")
                .withMetadata(objectMeta)
                .withData(data)
                .build();
        log.info("{}环境，创建新的secret：tencenthubkey", namespace);
        return kubernetesClient.secrets().inNamespace(namespace).create(secret);
    }

    /**
     * 获取namespace列表
     * @param env all-所有，dev-开发环境，test-测试环境
     * @return
     */
    public List<Map<String, Object>> getNamespaceList(String env) {
        List<Map<String, Object>> resultList = new ArrayList<>();
        List<Namespace> allNamespaceList = kubernetesClient.namespaces().list().getItems();
        List<Namespace> namespaceList = new ArrayList<>();
        if (StringUtils.equals(env, "all")) {
            for (Namespace namespace: allNamespaceList) {
                // 剔除一些系统创建的namespace
                if (namespace.getMetadata().getAnnotations() != null) {
                    namespaceList.add(namespace);
                }
            }
        } else {
            for (Namespace namespace: allNamespaceList) {
                if (namespace.getMetadata().getAnnotations() != null && namespace.getMetadata().getAnnotations().get("description").equals(env)) {
                    namespaceList.add(namespace);
                }
            }
        }
        for (Namespace namespace : namespaceList) {
            String name = namespace.getMetadata().getName();
            // 获取Service
            Service service = kubernetesClient.services().inNamespace(name).withName("mysql").get();
            // 获取Pod
            Map<String, String> labels = TkeUtils.StringArrayToMap(new String[]{"qcloud-app", "mysql", "type", "base"});
            List<Pod> podList = kubernetesClient.pods().inNamespace(name).withLabels(labels).list().getItems();
            Integer port = null;
            String host = null;
            if (service != null) {
                port = service.getSpec().getPorts().get(0).getNodePort();
            }
            for (Pod pod : podList) {
                if (StringUtils.equals(pod.getStatus().getPhase(), "Running")) {
                    host = podList.get(0).getStatus().getHostIP();
                }
            }
            Map<String, Object> map = new HashMap<>();
            map.put("name", name);
            map.put("description", namespace.getMetadata().getAnnotations().get("description"));
            map.put("owner", namespace.getMetadata().getAnnotations().get("owner"));
            map.put("status", namespace.getStatus().getPhase());
            map.put("mysqlHost", host);
            map.put("mysqlPort", port);
            try {
                map.put("createdAt", DateUtils.convertTZStr(namespace.getMetadata().getCreationTimestamp(), "yyyy-MM-dd HH:mm:ss"));
            } catch (Exception e) {
                log.info("时间解析异常！");
                e.printStackTrace();
                map.put("createdAt", namespace.getMetadata().getCreationTimestamp());
            }
            resultList.add(map);
        }
        return resultList;
    }

    /**
     * 获取pod列表
     * @param namespace
     * @return
     */
    public List<Map<String, Object>> getPodList(String namespace) {
        // 获取Pod列表
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).list().getItems();
        // 获取Service列表
        List<Service> serviceList = kubernetesClient.services().inNamespace(namespace).list().getItems();
        // 获取ingress列表
        List<Ingress> ingressList = kubernetesClient.extensions().ingresses().inNamespace(namespace).list().getItems();
        Map<String, Map<String, Object>> podServiceMap = new HashMap<>();
        // 遍历Pod列表
        for (Pod pod: podList) {
            if (!pod.getStatus().getPhase().equals("Failed")) {
                String serviceName = pod.getMetadata().getLabels().get("qcloud-app");
                podServiceMap.put(serviceName, formatPodInfo(pod));
            }
        }
        // 遍历Service列表
        for (Service service: serviceList) {
            String serviceName = service.getMetadata().getName();
            if (StringUtils.equals(serviceName, "kafka-" + namespace)) {
                serviceName = "kafka";
            }
            podServiceMap = assign(serviceName, podServiceMap, formatServiceInfo(service));
        }
        // 遍历Ingress列表
        for (Ingress ingress: ingressList) {
            String serviceName = ingress.getMetadata().getName();
            podServiceMap = assign(serviceName, podServiceMap, formatIngressInfo(ingress));
        }
        // 对象转列表
        List<Map<String, Object>> resultList = new ArrayList<>();
        for (Map<String, Object> map: podServiceMap.values()) {
            if ((map != null) && (map.get("serviceName") != null))  {
                resultList.add(map);
            }
        }
        // 根据serviceName对列表进行排序
        Collections.sort(resultList, (m1, m2) -> {
            String serviceName1 = m1.get("serviceName").toString();
            String serviceName2 = m2.get("serviceName").toString();
            return serviceName1.compareTo(serviceName2);
        });
        return resultList;
    }

    public Map<String, Map<String, Object>> assign(String serviceName, Map<String, Map<String, Object>> podServiceMap, Map<String, Object> otherMap) {
        Map<String, Object> podService = podServiceMap.get(serviceName);
        for (Map.Entry<String, Object> entry: otherMap.entrySet()) {
            if (podService != null) {
                podService.put(entry.getKey(), entry.getValue());
            }
        }
        podServiceMap.put(serviceName, podService);
        return podServiceMap;
    }

    /**
     * 处理pod相关信息
     * @param pod
     * @return
     */
    public Map<String, Object> formatPodInfo(Pod pod) {
        Map<String, Object> podMap = new HashMap<>();
        String serviceName = pod.getMetadata().getLabels().get("qcloud-app");
        podMap.put("serviceName", serviceName);
        podMap.put("podName", pod.getMetadata().getName());
        podMap.put("status", getPodStatus(pod));
        podMap.put("podIp", pod.getStatus().getPodIP());
        podMap.put("lanIp", pod.getStatus().getHostIP());
        // 特殊处理，格式化创建时间
        try {
            if (pod.getStatus().getStartTime() != null) {
                podMap.put("createdAt", DateUtils.convertTZStr(pod.getStatus().getStartTime(), "yyyy-MM-dd HH:mm:ss"));
            } else {
                podMap.put("createdAt", DateUtils.convertDate(new Date(), "yyyy-MM-dd HH:mm:ss"));
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.info("时间转换异常！");
            podMap.put("createdAt", pod.getStatus().getStartTime());
        }
        if (pod.getStatus().getContainerStatuses().size() > 0) {
            podMap.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
            podMap.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
        } else {
            podMap.put("image", null);
            podMap.put("imageId", null);
        }
        if (pod.getMetadata().getLabels() != null) {
            podMap.put("labels", pod.getMetadata().getLabels());
            // 对于非基础服务，新增服务描述，方便前端展示
            if (!StringUtils.equals(pod.getMetadata().getLabels().get("type"), "base")) {
                QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
                queryWrapper.eq("project_name", serviceName);
                DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
                if (dockerProject != null) {
                    podMap.put("desc", dockerProject.getDesc());
                }
            }
        }
        return podMap;
    }

    /**
     * 格式化pod状态
     * @param pod
     * @return
     */
    public String getPodStatus(Pod pod) {
        if (pod.getMetadata().getDeletionTimestamp() != null) {
            return  "Terminating";
        } else {
            String status = "";
            List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
            if (containerStatuses.size() == 0) {
                status = "Pending";
            } else {
                if (containerStatuses.get(0).getState().getWaiting() != null) {
                    status = "ContainerCreating";
                }
                if (containerStatuses.get(0).getState().getRunning() != null && !containerStatuses.get(0).getReady()) {
                    status = "Waiting";
                }
                if (containerStatuses.get(0).getState().getRunning() != null && containerStatuses.get(0).getReady()) {
                    status = "Normal";
                }
            }
            return status;
        }
    }

    /**
     * 格式化k8s Service
     * @param service
     * @return
     */
    public Map<String, Object> formatServiceInfo(Service service) {
        Map<String, Object> serviceMap = new HashMap<>();
        serviceMap.put("serviceType", service.getSpec().getType());
        List<Map<String, Object>> portMappingList = new ArrayList<>();
        List<ServicePort> servicePortList = service.getSpec().getPorts();
        if (servicePortList.size() > 0) {
            for (ServicePort servicePort : servicePortList) {
                if (servicePort.getNodePort() != null) {
                    serviceMap.put("port_" + servicePort.getName(), servicePort.getNodePort());
                }
                Map<String, Object> portMap = new HashMap<>();
                portMap.put("name", servicePort.getName());
                portMap.put("nodePort", servicePort.getNodePort());
                portMap.put("port", servicePort.getPort());
                portMap.put("protocol", servicePort.getProtocol());
                portMap.put("targetPort", servicePort.getTargetPort());
                portMappingList.add(portMap);
            }
        }
        if (portMappingList.size() > 0) {
            serviceMap.put("portMappings", portMappingList);
        }
        return serviceMap;
    }

    /**
     * 格式化k8s Ingress
     * @param ingress
     * @return
     */
    public Map<String, Object> formatIngressInfo(Ingress ingress) {
        Map<String, Object> ingressMap = new HashMap<>();
        if (ingress.getSpec().getRules() != null && ingress.getSpec().getRules().size() > 0) {
            ingressMap.put("host", ingress.getSpec().getRules().get(0).getHost());
        } else {
            ingressMap.put("host", "");
        }
        return ingressMap;
    }

    /**
     * 获取Service详情
     * @param namespace 环境
     * @param serviceName service名称
     * @return
     */
    public Service getServiceDetail(String namespace, String serviceName) {
        // kafka Service name不能直接叫kafka，需要特殊处理一下
        if (StringUtils.equals(serviceName, "kafka")) {
            serviceName = "kafka-" + namespace;
        }
        return kubernetesClient.services().inNamespace(namespace).withName(serviceName).get();
    }

    /**
     * 获取pod和service的一些信息
     * @param namespace
     * @param serviceType
     * @param serviceName
     * @return
     */
    public Map<String, Object> getPodAndServiceInfo(String namespace, String serviceType, String serviceName) {
        Service service = kubernetesClient.services().inNamespace(namespace).withName(serviceName).get();
        Pod pod = getPodDetail(namespace, serviceType, serviceName);
        Map<String, Object> map = new HashMap<>();
        if (pod != null && pod.getStatus().getPhase().equals("Running")) {
            // 端口映射
            List<Map<String, Object>> portMappingList = new ArrayList<>();
            ObjectMeta podMetadata = pod.getMetadata();
            if (service != null) {
                List<ServicePort> servicePortList = service.getSpec().getPorts();
                if (servicePortList.size() > 0) {
                    for (ServicePort servicePort : servicePortList) {
                        if (servicePort.getNodePort() != null) {
                            map.put("port_" + servicePort.getName(), servicePort.getNodePort());
                        }
                        Map<String, Object> portMap = new HashMap<>();
                        portMap.put("name", servicePort.getName());
                        portMap.put("nodePort", servicePort.getNodePort());
                        portMap.put("port", servicePort.getPort());
                        portMap.put("protocol", servicePort.getProtocol());
                        portMap.put("targetPort", servicePort.getTargetPort());
                        portMappingList.add(portMap);
                    }
                }
                if (portMappingList.size() > 0) {
                    map.put("portMappings", portMappingList);
                }
            }
            map.put("createdAt", podMetadata.getCreationTimestamp());
            map.put("serviceName", serviceName);
            map.put("podName", podMetadata.getName());
            map.put("labels", podMetadata.getLabels());
            List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
            map.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
            map.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
            map.put("lanIp", pod.getStatus().getHostIP());
            map.put("podIp", pod.getStatus().getPodIP());
            map.put("startTime", pod.getStatus().getStartTime());
            map.put("status", containerStatuses.get(0).getReady());
            Ingress ingress = kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).get();
            if (ingress != null) {
                if (ingress.getSpec().getRules() != null && ingress.getSpec().getRules().size() > 0) {
                    map.put("host", ingress.getSpec().getRules().get(0).getHost());
                } else {
                    map.put("host", "");
                }
            }
        }
        return map;
    }

    public Pod getPodDetail(String namespace, String serviceType, String serviceName) {
        Map<String, String> labelMap = TkeUtils.StringArrayToMap(new String[] {"type", serviceType, "qcloud-app", serviceName});
        LabelSelector labelSelector = new LabelSelectorBuilder().withMatchLabels(labelMap).build();
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).withLabelSelector(labelSelector).list().getItems();
        for (Pod pod: podList) {
            // pod可能有多个，返回Running状态的pod
            if (pod.getStatus().getPhase().equals("Running")) {
                return pod;
            }
        }
        return null;
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName   podName
     * @return
     */
    public boolean resetPod(String namespace, String podName) {
        return kubernetesClient.pods().inNamespace(namespace).withName(podName).delete();
    }

    /**
     * 获取Deployment
     *
     * @param namespace
     * @param serviceName
     * @return
     */
    public Deployment getDeployment(String namespace, String serviceName) {
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).get();
    }

    /**
     * 删除一个Deployment
     *
     * @param namespace   环境
     * @param serviceName 服务名
     * @return
     */
    public boolean deleteDeployment(String namespace, String serviceName) {
        log.info("开始删除{}环境{}的Deployment.", namespace, serviceName);
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).delete();
    }

    public boolean deleteService(String namespace, String serviceName) {
        log.info("开始删除{}环境{}的Service.", namespace, serviceName);
        // kafka Service需要特殊处理一下
        if (StringUtils.equals("kafka", serviceName)) {
            return kubernetesClient.services().inNamespace(namespace).withName("kafka-" + namespace).delete();
        }
        return kubernetesClient.services().inNamespace(namespace).withName(serviceName).delete();
    }

    public boolean deleteIngress(String namespace, String serviceName) {
        log.info("开始删除{}环境{}的Ingress.", namespace, serviceName);
         return kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).delete();
    }

    /**
     * 通过.yml文件创建Deployment
     * @param serviceCreateVo
     * @return
     */
    public Deployment createDeploymentByYaml(ServiceCreateVo serviceCreateVo) {
        String label = serviceCreateVo.getLabel();
        String serviceName = serviceCreateVo.getServiceName();
        String namespace = serviceCreateVo.getNamespace();
        String deploymentYaml;
        try {
            switch (label) {
                case "base":
                    String yamlFilePath = "tke/template/" + StringUtils.capitalize(serviceName) + "-Deployment.yml";
                    deploymentYaml = FileUtils.readFileFromClassPathResource(yamlFilePath);
                    deploymentYaml = TkeUtils.replaceBasicYaml(deploymentYaml, serviceCreateVo);
                    // kafka还需替换下zookeeper地址和本身的Service port
                    if (StringUtils.equals(serviceName, "kafka")) {
                        Map<String, Object> zookeeperInfo = getPodAndServiceInfo(namespace, "base", "zookeeper");
                        deploymentYaml = deploymentYaml.replaceAll("\\{\\{ZOOKEEPER_HOST}}", zookeeperInfo.get("lanIp").toString());
                        deploymentYaml = deploymentYaml.replaceAll("\\{\\{ZOOKEEPER_PORT}}", zookeeperInfo.get("port_2181").toString());
                        int count = 10;
                        Service kafkaService;
                        while (count > 0) {
                            try {
                                kafkaService = getServiceDetail(namespace, "kafka-" + namespace);
                                if (kafkaService != null) {
                                    deploymentYaml = deploymentYaml.replaceAll("\\{\\{KAFKA_SERVICE_PORT}}", String.valueOf(kafkaService.getSpec().getPorts().get(0).getNodePort()));
                                    break;
                                }
                                Thread.currentThread().sleep(1000);
                            } catch (Exception e) {
                                e.printStackTrace();
                                break;
                            }
                            count -= 1;
                        }
                    }
                    break;
                case "java":
                    DockerProject javaProject = dockerProjectService.getOne(Wrappers.lambdaQuery(DockerProject.class)
                            .eq(DockerProject::getProjectName, serviceCreateVo.getServiceName()));
                    if (javaProject == null) {
                        log.info("该项目未在后台配置，暂不创建Java Deployment：{}", JSON.toJSONString(serviceCreateVo));
                        return null;
                    }
                    deploymentYaml = FileUtils.readFileFromClassPathResource("tke/template/Java-Deployment.yml");
                    deploymentYaml = TkeUtils.replaceBusinessYaml(deploymentYaml, serviceCreateVo, javaProject);
                    break;
                case "ui":
                case "node":
                    DockerProject uiOrNodeProject = dockerProjectService.getOne(Wrappers.lambdaQuery(DockerProject.class)
                            .eq(DockerProject::getProjectName, serviceCreateVo.getServiceName()));
                    if (uiOrNodeProject == null) {
                        log.info("该项目未在后台配置，暂不创建UI Deployment：{}", JSON.toJSONString(serviceCreateVo));
                        return null;
                    }
                    deploymentYaml = FileUtils.readFileFromClassPathResource("tke/template/UI-Node-Deployment.yml");
                    deploymentYaml = TkeUtils.replaceBusinessYaml(deploymentYaml, serviceCreateVo, uiOrNodeProject);
                    break;
                case "go":
                    DockerProject goProject = dockerProjectService.getOne(Wrappers.lambdaQuery(DockerProject.class)
                            .eq(DockerProject::getProjectName, serviceCreateVo.getServiceName()));
                    if (goProject == null) {
                        log.info("该项目未在后台配置，暂不创建go Deployment：{}", JSON.toJSONString(serviceCreateVo));
                        return null;
                    }
                    deploymentYaml = FileUtils.readFileFromClassPathResource("tke/template/Go-Deployment.yml");
                    deploymentYaml = TkeUtils.replaceBusinessYaml(deploymentYaml, serviceCreateVo, goProject);
                    break;
                default:
                    log.info("暂不支持创建{}类型的deployment", label);
                    return null;
            }
            Yaml yaml = new Yaml(new Constructor(Deployment.class));
            Deployment deployment = yaml.load(deploymentYaml);
            // 健康检查白名单里的,去掉健康检查
            List<String> noHealthCheckServiceList = Arrays.asList(noHealthCheckService.split(","));
            if (noHealthCheckServiceList.contains(serviceName)) {
                deployment.getSpec().getTemplate().getSpec().getContainers().forEach(container -> container.setReadinessProbe(null));
                deployment.getSpec().getTemplate().getSpec().getContainers().forEach(container -> container.setLivenessProbe(null));
            }
            // 供应链健康检查单独处理下
            List<String> gylHealthCheckServiceList = Arrays.asList(gylHealthCheckService.split(","));
            if (gylHealthCheckServiceList.contains(serviceName)) {
                Probe readinessProbe = createHttpReadinessProbe("/actuator/health/readiness", 80);
                Probe livelinessProbe = createHttpLivenessProbe("/actuator/health/readiness", 80);
                deployment.getSpec().getTemplate().getSpec().getContainers().forEach(container -> container.setReadinessProbe(readinessProbe));
                deployment.getSpec().getTemplate().getSpec().getContainers().forEach(container -> container.setLivenessProbe(livelinessProbe));
            }
            log.info("开始在{}环境创建{}的k8s Deployment: \n{}", namespace, serviceName, YamlUtils.JsonToYamlStr(JSON.toJSONString(deployment)));
            return kubernetesClient.apps().deployments().inNamespace(namespace).createOrReplace(deployment);
        } catch (IOException e) {
            log.info("创建Deployment失败！");
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 通过.yml文件创建Service
     * @param serviceCreateVo
     * @return
     */
    public Service createServiceByYaml(ServiceCreateVo serviceCreateVo) {
        String label = serviceCreateVo.getLabel();
        String serviceName = serviceCreateVo.getServiceName();
        String namespace = serviceCreateVo.getNamespace();
        String serviceYaml;
        try {
            switch (label) {
                case "base":
                    String yamlFilePath = "tke/template/" + StringUtils.capitalize(serviceName) + "-Service.yml";
                    serviceYaml = FileUtils.readFileFromClassPathResource(yamlFilePath);
                    serviceYaml = TkeUtils.replaceBasicYaml(serviceYaml, serviceCreateVo);
                    break;
                case "java":
                    serviceYaml = FileUtils.readFileFromClassPathResource("tke/template/Java-Service.yml");
                    serviceYaml = TkeUtils.replaceBusinessYaml(serviceYaml, serviceCreateVo, null);
                    break;
                case "node":
                case "ui":
                    serviceYaml = FileUtils.readFileFromClassPathResource("tke/template/UI-Node-Service.yml");
                    serviceYaml = TkeUtils.replaceBusinessYaml(serviceYaml, serviceCreateVo, null);
                    break;
                case "go":
                    serviceYaml = FileUtils.readFileFromClassPathResource("tke/template/Go-Service.yml");
                    serviceYaml = TkeUtils.replaceBusinessYaml(serviceYaml, serviceCreateVo, null);
                    break;
                default:
                    log.info("暂不支持创建{}类型的k8s Service.", label);
                    return null;
            }
            Yaml yaml = new Yaml(new Constructor(Service.class));
            Service service = yaml.load(serviceYaml);
            // 设置下调试模式
            if (serviceCreateVo.getDebug() == 1) {
                service.getSpec().setType("NodePort");
            }
            log.info("开始在{}环境创建{}服务的k8s Service: \n{}", namespace, serviceName, YamlUtils.JsonToYamlStr(JSON.toJSONString(service)));
            return kubernetesClient.services().inNamespace(namespace).createOrReplace(service);
        } catch (IOException e) {
            log.info("Service创建失败.");
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 通过.yml文件创建PVC
     * @param serviceCreateVo
     * @return
     */
    public PersistentVolumeClaim createPvcByYaml(ServiceCreateVo serviceCreateVo) {
        String namespace = serviceCreateVo.getNamespace();
        String serviceName = serviceCreateVo.getServiceName();
        if (serviceName.contains("redis")) {
            log.info("redis不需要创建pvc.");
            return null;
        }
        try {
            String pvcFilePath = "tke/template/" + StringUtils.capitalize(serviceName) + "-Pvc.yml";
            String pvcYaml = FileUtils.readFileFromClassPathResource(pvcFilePath);
            pvcYaml = TkeUtils.replaceBasicYaml(pvcYaml, serviceCreateVo);
            log.info("开始在{}环境创建{}服务的k8s Pvc：\n{}", namespace, serviceName, pvcYaml);
            Yaml yaml = new Yaml(new Constructor(PersistentVolumeClaim.class));
            PersistentVolumeClaim persistentVolumeClaim = yaml.load(pvcYaml);
            return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).create(persistentVolumeClaim);
        } catch (IOException e) {
            log.info("PVC创建失败！");
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 通过Yaml文件创建Ingress
     * @param serviceCreateVo
     * @return
     */
    public Ingress createIngressByYaml(ServiceCreateVo serviceCreateVo) {
        try {
            String ingressYaml = FileUtils.readFileFromClassPathResource("tke/template/Ingress.yml");
            ingressYaml = TkeUtils.replaceBusinessYaml(ingressYaml, serviceCreateVo, null);
            Yaml yaml = new Yaml(new Constructor(Ingress.class));
            Ingress ingress = yaml.load(ingressYaml);
            IntOrString servicePort = new IntOrStringBuilder().withIntVal(80).build();
            ingress.getSpec().getRules().get(0).getHttp().getPaths().get(0).getBackend().setServicePort(servicePort);
            log.info("开始在{}环境创建{}服务的k8 Ingress:\n{}", serviceCreateVo.getNamespace(), serviceCreateVo.getServiceName(), YamlUtils.JsonToYamlStr(JSON.toJSONString(ingress)));
            return kubernetesClient.extensions().ingresses().inNamespace(serviceCreateVo.getNamespace()).createOrReplace(ingress);
        } catch (IOException e) {
            log.info("Ingress 创建失败!");
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 查看某个namespace中Deployment是否已存在
     * @param namespace 环境名
     * @param serviceName 服务名
     * @return
     */
    public boolean checkDeploymentExistence(String namespace, String serviceName) {
        return kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).get() != null;
    }

    /**
     * 查看某个namespace中pvc是否已存在
     * @param namespace 环境名
     * @param serviceName 服务名
     * @return
     */
    public boolean checkPvcExistence(String namespace, String serviceName) {
        return kubernetesClient.persistentVolumeClaims().inNamespace(namespace).withName(serviceName + "-" + namespace).get() != null;
    }

    /**
     * 创建Http类型的存活检查
     * @param path
     * @param port
     * @return
     */
    public Probe createHttpLivenessProbe(String path, Integer port) {
        HTTPGetAction httpGetAction = new HTTPGetActionBuilder()
                .withPath(path)
                .withPort(new IntOrStringBuilder().withIntVal(port).build())
                .build();

        Probe livenessProbe = new ProbeBuilder()
                .withHttpGet(httpGetAction)
                .withInitialDelaySeconds(600)
                .withTimeoutSeconds(5)
                .withPeriodSeconds(60)
                .withSuccessThreshold(1)
                .withFailureThreshold(3)
                .build();
        return livenessProbe;
    }

    /**
     * 创建Http类型的就绪检查
     * @param path
     * @param port
     * @return
     */
    public Probe createHttpReadinessProbe(String path, Integer port) {
        HTTPGetAction httpGetAction = new HTTPGetActionBuilder()
                .withPath(path)
                .withPort(new IntOrStringBuilder().withIntVal(port).build())
                .build();

        Probe readinessProbe = new ProbeBuilder()
                .withHttpGet(httpGetAction)
                .withInitialDelaySeconds(10)
                .withTimeoutSeconds(5)
                .withPeriodSeconds(10)
                .withSuccessThreshold(1)
                .withFailureThreshold(3)
                .build();
        return readinessProbe;
    }

    public void clearRubbish(String env) {
        List<Map<String, Object>> mapList = getNamespaceList(env);
        for (Map<String, Object> map: mapList) {
            String namespace = map.get("name").toString();
            log.info("开始清理{}环境的没有Deployment的Ingress和Service", namespace);

            // 循环删除某个namespace下没有Deployment的Service和Ingress
            List<Service> serviceList = kubernetesClient.services().inNamespace(namespace).list().getItems();
            for (Service service: serviceList) {
                String serviceName = service.getMetadata().getName();
                if (!serviceName.startsWith("kafka") && !checkDeploymentExistence(namespace, serviceName)) {
                    log.info("{}环境{}Deployment不存在，删除Service", namespace, serviceName);
                    deleteService(namespace, serviceName);
                }
            }
            // 循环删除某个namespace下没有Deployment的Ingress
            List<Ingress> ingressList = kubernetesClient.extensions().ingresses().inNamespace(namespace).list().getItems();
            for (Ingress ingress: ingressList) {
                String serviceName = ingress.getMetadata().getName();
                if (!checkDeploymentExistence(namespace, serviceName)) {
                    log.info("{}环境{}Deployment不存在，删除Ingress", namespace, serviceName);
                    deleteIngress(namespace, serviceName);
                }
            }
        }
    }

    /**
     * 获取运行中的pod列表
     *
     * @param namespace 环境
     * @return
     */
    public Map<String, String> getMysqlAddressByNamespace(String namespace) {
        // 获取Service
        Service service = kubernetesClient.services().inNamespace(namespace).withName("mysql").get();
        Map<String, String> labels = new HashMap<>();
        labels.put("qcloud-app", "mysql");
        labels.put("type", "base");
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).withLabels(labels).list().getItems();
        String port = null;
        String host = null;
        if (service != null) {
            port = String.valueOf(service.getSpec().getPorts().get(0).getNodePort());
        }
        if (podList.size() == 1) {
            host = podList.get(0).getStatus().getHostIP();
        }
        if (podList.size() >= 2) {
            for (Pod pod : podList) {
                if (pod.getStatus().getHostIP() != null) {
                    host = pod.getStatus().getHostIP();
                    break;
                }
            }
        }
        Map<String, String> resultMap = new HashMap<>();
        resultMap.put("host", host);
        resultMap.put("port", port);
        return resultMap;
    }

    /**
     * 修改服务指向
     * @param namespace
     * @param sourceServiceName
     * @param targetServiceName
     * @return
     */
    public boolean modifyServicePointing(String namespace, String sourceServiceName, String targetServiceName) {
        Service service = kubernetesClient.services().inNamespace(namespace).withName(sourceServiceName).get();
        service.getSpec().setClusterIP(null);
        service.getSpec().getPorts().forEach(servicePort -> servicePort.setTargetPort(null));
        service.getSpec().getSelector().put("qcloud-app", targetServiceName);
        return kubernetesClient.services().inNamespace(namespace).createOrReplace(service) != null;
    }

    public boolean modifyK8sService(Service service) {
        return kubernetesClient.services().inNamespace(service.getMetadata().getNamespace()).createOrReplace(service) != null;
    }
}
