package cn.qg.holmes.controller.quality;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.config.annotation.RequiresPermissions;
import cn.qg.holmes.entity.quality.DingRobot;
import cn.qg.holmes.entity.quality.SendScheduleVo;
import cn.qg.holmes.entity.quality.SendSmokingResultVo;
import cn.qg.holmes.service.quality.DingRobotService;
import cn.qg.holmes.service.quality.JiraBugPoolService;
import cn.qg.holmes.service.quality.JiraService;
import cn.qg.holmes.service.quality.WebhookService;
import cn.qg.holmes.utils.DingdingUtils;
import com.alibaba.fastjson.JSON;
import com.atlassian.jira.rest.client.api.domain.BasicProject;
import com.atlassian.jira.rest.client.api.domain.Issue;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/jira")
public class JiraController {

    @Autowired
    JiraService jiraService;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    @Autowired
    DingRobotService dingRobotService;

    @Autowired
    WebhookService webhookService;

    /**
     * jira webhook, 给测试环境提BUG用
     * @param request
     */
    @PostMapping("/webhook")
    public void jiraWebhook(HttpServletRequest request) {
        try {
            BufferedReader br = request.getReader();
            StringBuilder jiraData = new StringBuilder();
            String str = "";
            while((str = br.readLine()) != null){
                jiraData.append(str);
            }
            br.close();
            webhookService.handleTestJiraBugCommit(String.valueOf(jiraData));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * jira webhook，给生产俩项目提BUG使用
     * @param request
     */
    @PostMapping("/prod/webhook")
    public void prodWebhook(HttpServletRequest request) {
        try {
            BufferedReader br = request.getReader();
            StringBuilder jiraData = new StringBuilder();
            String str = "";
            while((str = br.readLine()) != null){
                jiraData.append(str);
            }
            br.close();
            log.info("收到线上JIRA BUG推送：{}", jiraData);
            webhookService.handleProdJiraBugUpdate(String.valueOf(jiraData));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取Jira项目列表
     * @return
     * @throws Exception
     */
    @GetMapping("/list/project")
    public JsonResult getJiraProjectKey() throws Exception {
        List<Map<String, String>> jiraKeyList = new ArrayList<>();
        Iterable<BasicProject> projects = jiraService.getJiraProjectList();
        for (BasicProject project: projects) {
            Map<String, String> map = new HashMap<>();
            map.put("id", project.getId().toString());
            map.put("key", project.getKey());
            map.put("name", project.getName());
            jiraKeyList.add(map);
        }
        return JsonResult.buildSuccessResult(jiraKeyList);
    }

    /**
     * 项目机器人列表
     * @param pageNum 第几页
     * @param pageSize 每页多少个
     * @return
     */
    @GetMapping("/list/robot")
    public JsonResult getProjectRobotList(Integer pageNum, Integer pageSize) {
        IPage<DingRobot> page = new Page<>(pageNum, pageSize);
        QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper();
        dingRobotQueryWrapper.orderByDesc("id");
        IPage<DingRobot> projectRobotIPage = dingRobotService.page(page, dingRobotQueryWrapper);
        Map<String, Object> map = new HashMap<>();
        map.put("total", projectRobotIPage.getTotal());
        map.put("list", projectRobotIPage.getRecords());
        return JsonResult.buildSuccessResult(map);
    }

    /**
     * 新增项目机器人
     * @param dingRobot 项目机器人实体
     * @return
     */
    @PostMapping("/add/robot")
    public JsonResult addProjectRobot(@RequestBody DingRobot dingRobot) {
        if (dingRobot.getProjectName() == null) {
            return JsonResult.buildErrorStateResult("项目名称不能为空！", false);
        }
        if (dingRobot.getDingUrl() == null) {
            return JsonResult.buildErrorStateResult("钉钉url不能为空！", false);
        }
        return JsonResult.buildSuccessResult(dingRobotService.save(dingRobot));
    }

    /**
     * 修改项目机器人
     * @param dingRobot 项目机器人实体
     * @return
     */
    @PostMapping("/modify/robot")
    public JsonResult editProjectRobot(@RequestBody DingRobot dingRobot) {
        if (dingRobot.getId() == null) {
            return JsonResult.buildErrorStateResult("id不能为空！", false);
        }
        if (dingRobot.getProjectName() == null) {
            return JsonResult.buildErrorStateResult("项目名称不能为空！", false);
        }
        if (dingRobot.getDingUrl() == null) {
            return JsonResult.buildErrorStateResult("钉钉url不能为空！", false);
        }
        return JsonResult.buildSuccessResult(dingRobotService.updateById(dingRobot));
    }

    /**
     * 删除项目机器人
     * @param projectRobotId 项目机器人实体id
     * @return
     */
    @GetMapping("/delete/robot")
    public JsonResult deleteProjectRobot(Integer projectRobotId) {
        if (dingRobotService.getById(projectRobotId) == null) {
            return JsonResult.buildErrorStateResult("项目机器人不存在！", false);
        }
        return JsonResult.buildSuccessResult(dingRobotService.removeById(projectRobotId));
    }

    @GetMapping("/report/project")
    public void sendReportToDingdingGroup(String project, String webhook) throws Exception {
        Iterable<Issue> issues = jiraService.getUnsolvedIssueListByProject(project);
        String markdownMsg = DingdingUtils.buildMarkdownMsgForUnsolvedIssueList(issues);
        DingdingUtils.sendToDingding(markdownMsg, webhook);
    }

    /**
     * 发送测试进度
     * @return
     * @throws Exception
     */
//    @RequiresPermissions("qa:process:schedule")
    @PostMapping("/send/schedule")
    public JsonResult sendTestReport(@RequestBody SendScheduleVo sendScheduleVo) throws Exception {
        String JQL = "project = " + sendScheduleVo.getJiraProjectKey() + " AND resolution = Unresolved AND text ~ \"" + sendScheduleVo.getTestProjectName() +  "\" ORDER BY priority DESC, updated DESC";
        Iterable<Issue> issues = jiraService.getIssueListByJQL(JQL);
        String markdownMsg = DingdingUtils.buildTestScheduleMarkdownMsg(issues, sendScheduleVo.getProgress(), sendScheduleVo.getRisk());
        return JsonResult.buildSuccessResult(DingdingUtils.sendToDingding(markdownMsg, sendScheduleVo.getWebhook()));
    }

    /**
     * 发送冒烟测试进度
     * @param sendSmokingResultVo 发送冒烟进度实体
     * @return
     */
//    @RequiresPermissions("qa:process:smoke")
    @PostMapping("/send/smoking")
    public JsonResult sendSmokingResult(@RequestBody SendSmokingResultVo sendSmokingResultVo) throws Exception {
        String JQL = "project = " + sendSmokingResultVo.getJiraProjectKey() + " AND resolution = Unresolved AND text ~ \""
                + sendSmokingResultVo.getTestProjectName() +  "\" ORDER BY priority DESC, updated DESC";
        Iterable<Issue> issues = jiraService.getIssueListByJQL(JQL);
        String markdownMsg = DingdingUtils.buildSmokingResultMarkdownMsg(issues, sendSmokingResultVo.getResult(), sendSmokingResultVo.getRisk());
        return JsonResult.buildSuccessResult(DingdingUtils.sendToDingding(markdownMsg, sendSmokingResultVo.getWebhook()));
    }

    /**
     * 根据JIRA项目名称和项目关键字获取未解决BUG
     * @param jiraProjectKey
     * @param projectName
     * @return
     * @throws Exception
     */
    @GetMapping("/issue/list")
    public JsonResult getUnsolvedByProject(@RequestParam String jiraProjectKey,
                                           @RequestParam String projectName,
                                           @RequestParam boolean unsolved) throws Exception {
        String JQL = null;
        if (!unsolved) {
            JQL = "project = " + jiraProjectKey + " AND text ~ \"" + projectName +  "\" ORDER BY priority DESC, updated DESC";
        } else {
            JQL = "project = " + jiraProjectKey + " AND resolution = Unresolved AND text ~ \"" + projectName +  "\" ORDER BY priority DESC, updated DESC";
        }
        Iterable<Issue> issues = jiraService.getIssueListByJQL(JQL);
        List<Map<String, Object>> mapList = new ArrayList<>();
        for (Issue issue: issues) {
            Map<String, Object> map = new HashMap<>();
            map.put("key", issue.getKey());
            map.put("summary", issue.getSummary());
            map.put("reporter", issue.getReporter().getDisplayName());
            if (issue.getAssignee() != null) {
                map.put("assignee", issue.getAssignee().getDisplayName());
            } else {
                map.put("assignee", "未分配");
            }
            map.put("priority", issue.getPriority().getName());
            map.put("status", issue.getStatus().getName());
            mapList.add(map);
        }
        return JsonResult.buildSuccessResult(mapList);
    }
}
