package cn.qg.holmes.utils;

import cn.qg.holmes.entity.monitor.MonitorTask;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;

@Slf4j
public class QuartzUtils {

    public static void createScheduleJob(Scheduler scheduler, MonitorTask monitorTask) {
        try {
            Class<? extends Job> jobClass = (Class<? extends Job>) Class.forName("cn.qg.holmes.quartz." + monitorTask.getTaskClass());
            JobDetail jobDetail = JobBuilder
                    .newJob(jobClass)
                    .withIdentity(monitorTask.getTaskName())
                    .build();
            CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(monitorTask.getTaskCron());
            CronTrigger cronTrigger = TriggerBuilder
                    .newTrigger()
                    .withIdentity(monitorTask.getTaskName())
                    .withSchedule(scheduleBuilder)
                    .build();
            scheduler.scheduleJob(jobDetail, cronTrigger);
        } catch (ClassNotFoundException e) {
            log.info("定时任务类: {}路径出错：请输入类的绝对路径", monitorTask.getTaskClass());
        } catch (SchedulerException e) {
            log.info("创建定时任务出错:" + e.getMessage());
        }
        log.info("创建定时任务: {}", monitorTask.getTaskName());
    }

    /**
     * 根据任务名称暂停定时任务
     * @param scheduler  调度器
     * @param jobName    定时任务名称
     * @throws SchedulerException
     */
    public static void pauseScheduleJob(Scheduler scheduler, String jobName) {
        JobKey jobKey = JobKey.jobKey(jobName);
        try {
            scheduler.pauseJob(jobKey);
        } catch (SchedulerException e) {
            log.info("暂停定时任务出错：" + e.getMessage());
        }
    }

    /**
     * 根据任务名称恢复定时任务
     * @param scheduler  调度器
     * @param jobName    定时任务名称
     * @throws SchedulerException
     */
    public static void resumeScheduleJob(Scheduler scheduler, String jobName) {
        JobKey jobKey = JobKey.jobKey(jobName);
        try {
            scheduler.resumeJob(jobKey);
        } catch (SchedulerException e) {
            log.info("启动定时任务出错：" + e.getMessage());
        }
    }

    /**
     * 根据任务名称立即运行一次定时任务
     * @param scheduler     调度器
     * @param jobName       定时任务名称
     * @throws SchedulerException
     */
    public static void runOnce(Scheduler scheduler, String jobName) {
        JobKey jobKey = JobKey.jobKey(jobName);
        try {
            scheduler.triggerJob(jobKey);
        } catch (SchedulerException e) {
            log.info("运行定时任务出错：" + e.getMessage());
        }
    }

    /**
     * 更新定时任务
     *
     * @param scheduler   调度器
     * @param monitorTask 定时任务信息类
     * @throws SchedulerException
     */
    public static void updateScheduleJob(Scheduler scheduler, MonitorTask monitorTask) {
        try {
            //获取到对应任务的触发器
            TriggerKey triggerKey = TriggerKey.triggerKey(monitorTask.getTaskName());
            //设置定时任务执行方式
            CronScheduleBuilder scheduleBuilder = CronScheduleBuilder.cronSchedule(monitorTask.getTaskCron());
            //重新构建任务的触发器trigger
            CronTrigger trigger = (CronTrigger) scheduler.getTrigger(triggerKey);
            trigger = trigger
                    .getTriggerBuilder()
                    .withIdentity(triggerKey)
                    .withSchedule(scheduleBuilder)
                    .build();
            //重置对应的job
            scheduler.rescheduleJob(triggerKey, trigger);
        } catch (SchedulerException e) {
            System.out.println("更新定时任务出错：" + e.getMessage());
        }
    }

    /**
     * 根据定时任务名称从调度器当中删除定时任务
     *
     * @param scheduler 调度器
     * @param jobName   定时任务名称
     * @throws SchedulerException
     */
    public static void deleteScheduleJob(Scheduler scheduler, String jobName) {
        JobKey jobKey = JobKey.jobKey(jobName);
        try {
            scheduler.deleteJob(jobKey);
        } catch (SchedulerException e) {
            log.info("删除定时任务出错：" + e.getMessage());
        }
    }
}
