package cn.qg.holmes.service.quality.impl;

import cn.qg.holmes.entity.quality.DingRobot;
import cn.qg.holmes.entity.quality.JiraBugPool;
import cn.qg.holmes.mapper.quality.DingRobotMapper;
import cn.qg.holmes.service.quality.JiraBugPoolService;
import cn.qg.holmes.service.quality.WebhookService;
import cn.qg.holmes.utils.DingdingUtils;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.List;

@Slf4j
@Service
public class WebhookServiceImpl implements WebhookService {

    @Autowired
    DingRobotMapper dingRobotMapper;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    /**
     * 电商线上故障处理群钉钉机器人地址
     */
    @Value("${ds.prod.ding.url}")
    private String dsProdDingUrl;

    /**
     * 金融线上故障处理群钉钉机器人地址
     */
    @Value("${jr.prod.ding.url}")
    private String jrProdDingUrl;

    /**
     * 处理线上jira bug状态变化
     * @param jiraData
     */
    @Override
    public void handleProdJiraBugUpdate(String jiraData) {
        String webhook = "";
        String webhookEvent = JsonPath.read(jiraData, "$.webhookEvent");
        String eventTypeName = JsonPath.read(jiraData, "$.issue_event_type_name");
        String creator = JsonPath.read(jiraData, "$.issue.fields.creator.displayName");
        String assignee = "";
        if (JsonPath.read(jiraData, "$.issue.fields.assignee") == null) {
            assignee = "未分配";
        } else {
            assignee = JsonPath.read(jiraData, "$.issue.fields.assignee.displayName");
        }
        String key = JsonPath.read(jiraData, "$.issue.key");
        String summary = JsonPath.read(jiraData, "$.issue.fields.summary");
        String priority = JsonPath.read(jiraData, "$.issue.fields.priority.name");
        String description = JsonPath.read(jiraData, "$.issue.fields.description");
        List<String> attachmentList = JsonPath.read(jiraData, "$.issue.fields.attachment[*].content");
        if (StringUtils.equals(key.split("-")[0], "YFGDZZ")) {
            // webhook赋值为金融群的机器人地址
            webhook = jrProdDingUrl;
        } else if (StringUtils.equals(key.split("-")[0], "YXMXS")) {
            // webhook赋值为电商群的机器人地址
            webhook = dsProdDingUrl;
        } else if (StringUtils.equals(key.split("-")[0], "CSXT")) {
            webhook = "https://oapi.dingtalk.com/robot/send?access_token=be220b4bd17f311d70365a0ee81a659b54f06d52f1ee8d4d7d1051a183a266e1";
        }
        // BUG创建的时候发送通知
        if (StringUtils.equals(webhookEvent, "jira:issue_created")) {
            String title = "线上研发工单提交";
            String content = "📣" + creator + "提交了线上研发工单：" + key + "\n\n";
            content += "概要：[" + summary + "](" + "http://jira2.quantgroup.cn/browse/" + key + ")" + "\n\n"
                    + "- 优先级：" + priority + "\n"
                    + "- 经办人：" + assignee + "\n\n";
            if (description != null) {
                String[] descList = description.split("\n");
                for (String desc: descList) {
                    content += "> " + desc + "\n\n";
                }
            }
            // 暂时不用
//            if (attachmentList.size() > 0) {
//                for (String attachment: attachmentList) {
//                    content += "> ![](" + attachment + ")\n" ;
//                }
//            }
            DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, true), webhook);
        }
        // BUG更新的时候发送通知
        if (StringUtils.equals(webhookEvent, "jira:issue_updated")) {
            // 钉钉消息title
            String title = "研发工单更新";
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + key;
            // 钉钉消息内容
            String content = "";
            switch (eventTypeName) {
                case "issue_work_started":
                    // BUG状态变为【处理中】
                    content += "📣研发工单 " + key + " 状态更新:" + "\n\n";
                    content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n";
                    content += "状态变更为：【处理中】\n\n";
                    DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, false), webhook);
                    break;
                case "issue_resolved":
                    // BUG状态变为【已解决】
                    content += "📣研发工单 " + key + " 状态更新:" + "\n\n";
                    content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n";
                    content += "状态变更为：【已解决】\n\n";
                    // content += "状态变更为：【已解决】, 解决结果为：【" + JsonPath.read(jiraData, "$.changelog.items[0].toString") + "】\n\n";
                    content += "请相关同学跟进验收🙏";
                    DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, false), webhook);
                    break;
                case "issue_reopened":
                    // BUG状态变为【重新打开】
                    content += "📣研发工单 " + key + " 状态更新:" + "\n\n";
                    content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n";
                    content += "状态变更为：【重新打开】\n\n";
                    content += "请相关同学再跟进一下🙏";
                    DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, false), webhook);
                    break;
                case "issue_closed":
                    // BUG状态变为【关闭】
                    // 暂不处理
                    break;
                case "issue_commented":
                    // 新增BUG备注
                    content += "📣研发工单 " + key + " 新增备注:" + "\n\n";
                    content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n";
                    content += "新增备注：" + JsonPath.read(jiraData, "$.comment.body");
                    DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, false), webhook);
                    break;
                case "issue_updated":
                    // BUG更新
                    String changedField = JsonPath.read(jiraData, "$.changelog.items[0].field");
                    if (StringUtils.equals(changedField, "assignee")) {
                        content += "📣研发工单 " + key + " 经办人变更:" + "\n\n";
                        content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n";
                        content += "- 原经办人：" + JsonPath.read(jiraData, "$.changelog.items[0].fromString") + "\n";
                        content += "- 新经办人：" + JsonPath.read(jiraData, "$.changelog.items[0].toString") + "\n";
                        DingdingUtils.sendToDingding(DingdingUtils.buildMarkdownMsg(title, content, false), webhook);
                    }
                    break;
                default:
                    break;
            }
        }
    }

    /**
     * 处理测试环境jira bug 提交
     * @param jiraData
     */
    @Override
    public Boolean handleTestJiraBugCommit(String jiraData) {
        log.info("收到jira bug提交：{}", jiraData);
        String webhookEvent = JsonPath.read(jiraData, "$.webhookEvent");
        if (StringUtils.equals(webhookEvent, "jira:issue_created")) {
            String creator = JsonPath.read(jiraData, "$.issue.fields.creator.displayName");
            String assignee = "";
            if (JsonPath.read(jiraData, "$.issue.fields.assignee") == null) {
                assignee = "未分配";
            } else {
                assignee = JsonPath.read(jiraData, "$.issue.fields.assignee.displayName");
            }
            String key = JsonPath.read(jiraData, "$.issue.key");
            String summary = JsonPath.read(jiraData, "$.issue.fields.summary");
            String priority = JsonPath.read(jiraData, "$.issue.fields.priority.name");
            String module = JsonPath.read(jiraData, "$.issue.fields.components[0].name");
            String robotUrl = null;
            Integer dingRobotId = null;
            Integer status = null;
            List<DingRobot> dingRobots = dingRobotMapper.selectList(null);
            for (DingRobot dingRobot : dingRobots) {
                String projectName = "【" + dingRobot.getProjectName() + "】";
                if (summary.contains(projectName)) {
                    robotUrl = dingRobot.getDingUrl();
                    dingRobotId = dingRobot.getId();
                    status = dingRobot.getStatus();
                    break;
                }
            }
            // 只有创建了dingRobot的项目的BUG才会被放到BUG池子里
            if (robotUrl != null && dingRobotId != null) {
                // 首次提交的BUG放入bug池子
                JiraBugPool jiraBugPool = new JiraBugPool();
                jiraBugPool.setKey(key);
                jiraBugPool.setSummary(summary);
                jiraBugPool.setPriority(priority);
                jiraBugPool.setReporter(creator);
                jiraBugPool.setAssignee(assignee);
                jiraBugPool.setUrl("http://jira2.quantgroup.cn/browse/" + key);
                jiraBugPool.setDingUrl(robotUrl);
                jiraBugPool.setDingRobotId(dingRobotId);
                jiraBugPoolService.save(jiraBugPool);
            }
            // 只有项目是进行中才会发送通知
            if (robotUrl != null && status == 1) {
                String markdownMsg = DingdingUtils.buildBugCommitMarkdownMsg(key, summary, creator, assignee, priority, module);
                return DingdingUtils.sendToDingding(markdownMsg, robotUrl);
            }
        }
        return false;
    }
}
