package cn.qg.holmes.service.auto.impl;

import cn.qg.holmes.entity.auto.AutoModule;
import cn.qg.holmes.entity.auto.Interface;
import cn.qg.holmes.entity.auto.Testcase;
import cn.qg.holmes.entity.auto.TestcaseReport;
import cn.qg.holmes.mapper.auto.TestcaseMapper;
import cn.qg.holmes.service.auto.*;
import cn.qg.holmes.utils.HttpClientUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author libo
 * created at 2021-03-30
 */
@Service
@Slf4j
public class TestcaseServiceImpl extends ServiceImpl<TestcaseMapper, Testcase> implements TestcaseService {

    @Autowired
    TestcaseMapper testcaseMapper;

    @Autowired
    InterfaceService interfaceService;

    @Autowired
    AutoModuleService autoModuleService;

    @Autowired
    AutoUtilsService autoUtilsService;

    @Autowired
    TestcaseReportService testcaseReportService;

    @Override
    public IPage<Testcase> getTestcaseList(Integer moduleId, String testcaseName, Integer pageNum, Integer pageSize) {
        IPage<Testcase> page = new Page<>(pageNum, pageSize);
        return testcaseMapper.getTestcaseList(page, moduleId, testcaseName);
    }

    /**
     * 执行单接口用例
     * @param namespace 环境
     * @param testcaseId 用例id
     * @return
     */
    @Override
    public String singleTestcaseExecutor(String namespace, Integer testcaseId, String batch) {
        Testcase testcase = testcaseMapper.selectById(testcaseId);
        Interface anInterface = interfaceService.getById(testcase.getInterfaceId());
        AutoModule autoModule = autoModuleService.getById(anInterface.getModuleId());
        String domain = autoModule.getDomain();
        String url = domain.replace("${NAMESPACE}", namespace) + anInterface.getUrl();
        String method = anInterface.getMethod().toUpperCase();
        String paramType = anInterface.getParamType();
        String headers = testcase.getHeaders();
        String parameters = testcase.getParameters();
        String testcaseName = testcase.getName();
//        String variables = testcase.getVariables();
        String extract = testcase.getExtract();
        String validate = testcase.getValidate();
        Map<String, String> headersMap = new HashMap<>();
        Map<String, String> parameterMap = new HashMap<>();
        // 参数准备
        if (!StringUtils.isEmpty(headers)) {
            headersMap = JSON.parseObject(headers, Map.class);
        }
        if (!StringUtils.isEmpty(parameters)) {
            parameterMap = JSON.parseObject(parameters, Map.class);
        }
        // 使用变量替换headers
//        if (variables != null && !variables.isEmpty() && headers != null && !headers.isEmpty()) {
//            headersMap = autoUtilsService.replaceHeaders(headers, variables);
//        }
//        // 使用变量替换参数
//        if (variables != null && !variables.isEmpty() && parameters != null && !parameters.isEmpty()) {
//            parameterMap = autoUtilsService.replaceParameters(parameters, variables);
//        }
        // 创建断言列表
        List<Map> validateList = new ArrayList<>();
        if (validate != null && !validate.isEmpty()) {
            validateList = JSON.parseArray(validate, Map.class);
        }
        String response = "";
        Long startTime = System.currentTimeMillis();
        // 发起http请求
        switch (method) {
            case "GET":
                response = HttpClientUtils.doGet(url, headersMap, parameterMap);
                break;
            case "POST":
                if (paramType.equals("json")) {
                    response = HttpClientUtils.doPostJson(url, headersMap, JSON.toJSONString(parameterMap));
                } else if (paramType.equals("form")) {
                    response = HttpClientUtils.doPost(url, headersMap, parameterMap);
                } else if (paramType.equals("params")) {
                    response = HttpClientUtils.doPost(url, headersMap, parameterMap);
                }
                break;
            default:
                log.info("暂不支持该请求方法：{}", method);
                break;
        }
        Long endTime = System.currentTimeMillis();
        Long elapsedTime = endTime - startTime;
        log.info("用例:{}, 返回为：{}", testcaseName, response);
        log.info("本次接口执行时间：{}", elapsedTime);
        // 进行断言
        boolean assertResult = true;
        if (!validateList.isEmpty()) {
            assertResult = autoUtilsService.assertResponse(response, validateList);
        }
        // 解析响应
//        if (extract != null && !extract.isEmpty()) {
//            autoUtilsService.extractResponse(response, extract);
//        }
        Map<String, Object> map = new HashMap<>();
        map.put("url", url);
        map.put("name", testcaseName);
        map.put("headers", headersMap);
        map.put("parameters", parameterMap);
        map.put("assertResult", assertResult);
        // 兼容response不是json的情况
        try {
            map.put("response", JSON.parseObject(response, Map.class));
        } catch (Exception e) {
            e.printStackTrace();
            map.put("response", response);
        }
        map.put("elapsedTime", elapsedTime);

        // 插入测试报告
        TestcaseReport testcaseReport = new TestcaseReport();
        testcaseReport.setTestcaseId(testcaseId);
        testcaseReport.setHeaders(JSON.toJSONString(headersMap));
        testcaseReport.setParameters(JSON.toJSONString(parameterMap));
        testcaseReport.setResponse(response);
        testcaseReport.setValidate(validate);
        if (assertResult) {
            testcaseReport.setStatus("pass");
        } else {
            testcaseReport.setStatus("fail");
        }
        testcaseReport.setElapsedTime(elapsedTime);
        testcaseReport.setBatch(batch);
        testcaseReport.setModuleId(testcase.getModuleId());
        testcaseReport.setNamespace(namespace);
        testcaseReport.setUrl(url);
        testcaseReportService.save(testcaseReport);
        return JSON.toJSONString(map);
    }

    /**
     * 执行某个模块下所有用例
     * @param namespace 环境
     * @param moduleId 模块id
     * @param batch 执行批次
     * @return
     */
    @Override
    public String executeTestcaseByModuleId(String namespace, Integer moduleId, String batch) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("module_id", moduleId);
        List<Testcase> testcaseList = testcaseMapper.selectList(queryWrapper);
        List<Map> resultList = new ArrayList<>();
        for (Testcase testcase: testcaseList) {
            String response = singleTestcaseExecutor(namespace, testcase.getId(), batch);
            resultList.add(JSON.parseObject(response, Map.class));
        }
        return JSON.toJSONString(resultList);
    }
}
