package cn.qg.holmes.utils;

import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * HttpClient 工具类
 * @author libo
 */
public class HttpClientUtils {

    // 编码格式。发送编码格式统一用UTF-8
    private static final String ENCODING = "UTF-8";

    // 设置连接超时时间，单位毫秒。
    private static final int CONNECT_TIMEOUT = 6000;

    // 请求获取数据的超时时间(即响应时间)，单位毫秒。
    private static final int SOCKET_TIMEOUT = 6000;

    /**
     * http get 请求
     * @param url
     * @param headers
     * @param params
     * @return
     */
    public static String doGet(String url, Map<String, String> headers, Map<String, String> params) {

        CloseableHttpClient httpClient = HttpClients.createDefault();
        CloseableHttpResponse httpResponse = null;

        String resultString = "";

        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            // 设置请求参数
            if (params != null) {
                for (String key: params.keySet()) {
                    uriBuilder.addParameter(key, params.get(key));
                }
            }
            URI uri = uriBuilder.build();

            // 创建Http Get 请求
            HttpGet httpGet = new HttpGet(uri);

            // 设置请求超时时间
            RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
            httpGet.setConfig(requestConfig);

            // 设置请求头
            setHeader(headers, httpGet);

            // 执行http请求
            httpResponse = httpClient.execute(httpGet);
            resultString = EntityUtils.toString(httpResponse.getEntity(), ENCODING);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                // 释放资源
                if (httpResponse != null) {
                    httpResponse.close();
                }
                httpClient.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return resultString;
    }

    // 不带参数的get请求
    public static String doGet(String url) {
        return doGet(url, null, null);
    }

    // 不带headers的get请求
    public static String doGet(String url, Map<String, String> params) {
        return doGet(url, null, params);
    }

    /**
     * 发送POST请求，携带map格式的参数
     * @param url 请求地址
     * @param headers 请求头
     * @param params 请求参数，Map格式
     * @return
     */
    public static String doPost(String url, Map<String, String> headers, Map<String, String> params) {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        CloseableHttpResponse httpResponse = null;
        String resultString = "";

        try {
            // 创建 HttpPost
            HttpPost httpPost = new HttpPost(url);

            // 设置请求超时时间
            RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
            httpPost.setConfig(requestConfig);

            // 设置请求头
            setHeader(headers, httpPost);

            // 设置请求参数
            if (params != null && !params.isEmpty()) {
                List<NameValuePair> nameValuePairs = new ArrayList<>();
                for (String key: params.keySet()) {
                    nameValuePairs.add(new BasicNameValuePair(key, params.get(key)));
                }
                httpPost.setEntity(new UrlEncodedFormEntity(nameValuePairs, ENCODING));
            }
            // 执行http请求
            httpResponse = httpClient.execute(httpPost);
            resultString = EntityUtils.toString(httpResponse.getEntity(), ENCODING);

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                // 释放资源
                if (httpResponse != null) {
                    httpResponse.close();
                }
                httpClient.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return resultString;
    }

    // 不带参数的post请求
    public static String doPost(String url) {
        return doPost(url, null, null);
    }

    // 不带headers的post请求
    public static String doPost(String url, Map<String, String> params) {
        return doPost(url, null, params);
    }

    /**
     * 发送post请求，携带json类型数据
     * @param url 请求地址
     * @param headers 请求头
     * @param json json格式参数
     * @return
     */
    public static String doPostJson(String url, Map<String, String> headers, String json) {
        CloseableHttpClient httpClient = HttpClients.createDefault();
        CloseableHttpResponse httpResponse = null;
        String resultString = "";
        try {
            // 创建Http Post请求
            HttpPost httpPost = new HttpPost(url);
            // 创建请求内容
            StringEntity entity = new StringEntity(json, ContentType.APPLICATION_JSON);
            httpPost.setEntity(entity);

            // 设置请求头
            setHeader(headers, httpPost);

            httpResponse = httpClient.execute(httpPost);
            resultString = EntityUtils.toString(httpResponse.getEntity(), ENCODING);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                // 释放资源
                if (httpResponse != null) {
                    httpResponse.close();
                }
                httpClient.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return resultString;
    }

    /**
     * Description: 封装请求头
     * @param headers 请求头
     * @param httpRequestBase HttpRequestBase
     */
    public static void setHeader(Map<String, String> headers, HttpRequestBase httpRequestBase) {
        // 封装请求头
        if (headers != null && !headers.isEmpty()) {
            Set<String> keySet = headers.keySet();
            for (String key : keySet) {
                // 设置到请求头到HttpRequestBase对象中
                httpRequestBase.setHeader(key, headers.get(key));
            }
        }
    }
}

