package cn.qg.holmes.controller.quality;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.config.annotation.RequiresPermissions;
import cn.qg.holmes.entity.quality.SubmitTestInstruction;
import cn.qg.holmes.service.quality.JiraService;
import cn.qg.holmes.service.quality.SubmitTestInstructionService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.atlassian.jira.rest.client.api.domain.BasicProject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * 质量管理相关接口
 * @author libo
 * 2021-10-29
 */
@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/quality")
public class QualityController {

    @Autowired
    SubmitTestInstructionService instructionService;

    @Autowired
    JiraService jiraService;

    /**
     * 提测说明分页列表
     * @param projectName 项目名称
     * @param jiraProjectKey jira项目关键字
     * @param pageNum 第几页
     * @param pageSize 每页多少个
     * @return
     */
    @GetMapping("/testInstruction/list")
    public JsonResult getSubmitTestInstructionList(String projectName, String jiraProjectKey,
                                                   @RequestParam(defaultValue = "1") Integer pageNum,
                                                   @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<SubmitTestInstruction> queryWrapper = new QueryWrapper<>();
        if (!StringUtils.isEmpty(projectName)) {
            queryWrapper.like("project_name", projectName);
        }
        if (!StringUtils.isEmpty(jiraProjectKey)) {
            queryWrapper.like("jira_project_key", jiraProjectKey);
        }
        queryWrapper.orderByDesc("id");
        IPage<SubmitTestInstruction> instructionIPage = new Page<>(pageNum, pageSize);
        IPage<SubmitTestInstruction> page = instructionService.page(instructionIPage, queryWrapper);
        Map<String, Object> map = new HashMap<>();
//        map.put("list", page.getRecords());
        List<SubmitTestInstruction> submitTestInstructionList = page.getRecords();
        map.put("total", page.getTotal());
        // 获取jira项目Map
        Map<String, String> projectMap = new HashMap<>();
        try {
            Iterable<BasicProject> projects = jiraService.getJiraProjectList();
            for (BasicProject project: projects) {
                projectMap.put(project.getKey(), project.getName());
            }
        } catch (Exception e) {
            log.info("获取jira项目Map失败.");
            e.printStackTrace();
        }
        // 循环设置列表字段jiraProjectName
        for (SubmitTestInstruction submitTestInstruction: submitTestInstructionList) {
            submitTestInstruction.setJiraProjectName(projectMap.get(submitTestInstruction.getJiraProjectKey()));
        }
        map.put("list", submitTestInstructionList);
        return JsonResult.buildSuccessResult(map);
    }

    /**
     * 新增提测说明
     * @param submitTestInstruction 提测说明实体类
     * @return
     */
    @RequiresPermissions("qa:testDescription:add")
    @PostMapping("/testInstruction/save")
    public JsonResult saveSubmitTestInstruction(@RequestBody SubmitTestInstruction submitTestInstruction) {
        String projectName = submitTestInstruction.getProjectName();
        String jiraProjectKey = submitTestInstruction.getJiraProjectKey();
        QueryWrapper<SubmitTestInstruction> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_name", projectName);
        queryWrapper.eq("jira_project_key", jiraProjectKey);
        if (instructionService.getOne(queryWrapper) != null) {
            return JsonResult.buildErrorStateResult("项目已存在，无法创建！", false);
        }
        return JsonResult.buildSuccessResult("项目创建成功！", instructionService.save(submitTestInstruction));
    }

    /**
     * 更新提测说明
     * @param submitTestInstruction 提测说明实体
     * @return
     */
    @RequiresPermissions("qa:testDescription:edit")
    @PostMapping("/testInstruction/update")
    public JsonResult updateSubmitTestInstruction(@RequestBody SubmitTestInstruction submitTestInstruction) {
        Integer id = submitTestInstruction.getId();
        Date dateParam = submitTestInstruction.getUpdateTime();
        // 获取数据库中保存的实体
        SubmitTestInstruction submitTestInstructionDb = instructionService.getById(id);
        Date dateDb = submitTestInstructionDb.getUpdateTime();
        // 如果前端传的更新时间与数据库存的更新时间一致，那就直接更新
        if (!dateParam.equals(dateDb)) {
            // 多人修改时，处理服务与分支
            List<Map> serviceAndBranchParamList = JSONArray.parseArray(submitTestInstruction.getServiceAndBranch(), Map.class);
            List<Map> serviceAndBranchDbList = JSONArray.parseArray(submitTestInstructionDb.getServiceAndBranch(), Map.class);
            for (Map mapParam : serviceAndBranchParamList) {
                boolean flag = false;
                for (Map mapDb : serviceAndBranchDbList) {
                    if (mapParam.get("serviceName").toString().equals(mapDb.get("serviceName").toString())) {
                        flag = true;
                    }
                }
                if (!flag) {
                    serviceAndBranchDbList.add(mapParam);
                }
            }
            submitTestInstruction.setServiceAndBranch(JSON.toJSONString(serviceAndBranchDbList));

            // 多人修改时，处理Apollo
            List<Map> apolloParamList = JSONArray.parseArray(submitTestInstruction.getApollo(), Map.class);
            List<Map> apolloDbList = JSONArray.parseArray(submitTestInstructionDb.getApollo(), Map.class);
            for (Map mapParam: apolloParamList) {
                boolean flag = false;
                for (Map mapDb: apolloDbList) {
                    if (mapParam.get("projectName").toString().equals(mapDb.get("projectName").toString())
                            && mapParam.get("key").toString().equals(mapDb.get("key").toString())) {
                        flag = true;
                    }
                }
                if (!flag) {
                    apolloDbList.add(mapParam);
                }
            }
            submitTestInstruction.setApollo(JSON.toJSONString(apolloDbList));

            // 多人修改时，处理数据库
            List<Map> databaseParamList = JSONArray.parseArray(submitTestInstruction.getDatabase(), Map.class);
            List<Map> databaseDbList = JSONArray.parseArray(submitTestInstructionDb.getDatabase(), Map.class);
            for (Map mapParam: databaseParamList) {
                boolean flag = false;
                for (Map mapDb: databaseDbList) {
                    if (mapParam.get("db").toString().equals(mapDb.get("db").toString())) {
                        flag = true;
                    }
                }
                if (!flag) {
                    databaseDbList.add(mapParam);
                }
            }
            submitTestInstruction.setDatabase(JSON.toJSONString(databaseDbList));

            // 多人修改时，处理备注
            String scopeParam = submitTestInstruction.getScope();
            String scopeDb = submitTestInstructionDb.getScope();
            submitTestInstruction.setScope(scopeDb + "\n" + scopeParam);
        }
        submitTestInstruction.setUpdateTime(null);
        instructionService.saveOrUpdate(submitTestInstruction);

        return JsonResult.buildSuccessResult("更新成功!");
    }

    /**
     * 删除提测说明
     * @param submitTestInstruction 提测说明实体
     * @return
     */
    @RequiresPermissions("qa:testDescription:del")
    @PostMapping("/testInstruction/delete")
    public JsonResult deleteSubmitTestInstruction(@RequestBody SubmitTestInstruction submitTestInstruction) {
        return JsonResult.buildSuccessResult("删除成功！", instructionService.removeById(submitTestInstruction.getId()));
    }

    /**
     * 根据jira项目关键字获取项目名称
     * @param jiraProjectKey jira项目关键字，例如APP
     * @return
     */
    @GetMapping("/testInstruction/getProjectNameList")
    public JsonResult getTestInstructionListForQaProcess(@RequestParam String jiraProjectKey) {
        QueryWrapper<SubmitTestInstruction> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("jira_project_key", jiraProjectKey);
        return JsonResult.buildSuccessResult("获取项目列表成功", instructionService.list(queryWrapper));
    }

    /**
     * 获取详情
     * @param projectName 项目名称
     * @param jiraProjectKey jira项目关键字
     * @return
     */
    @GetMapping("/testInstruction/detail")
    public JsonResult getTestInstructionDetail(@RequestParam String projectName, @RequestParam String jiraProjectKey) {
        QueryWrapper<SubmitTestInstruction>  queryWrapper = new QueryWrapper<>();
        queryWrapper
                .eq("project_name", projectName)
                .eq("jira_project_key", jiraProjectKey);
        SubmitTestInstruction submitTestInstruction = instructionService.getOne(queryWrapper);
        if (submitTestInstruction == null) {
            return JsonResult.buildErrorStateResult("所选项目不存在", false);
        }
        // 设置jiraProjectName
        try {
            Iterable<BasicProject> projects = jiraService.getJiraProjectList();
            for (BasicProject project: projects) {
                if (project.getKey().equals(submitTestInstruction.getJiraProjectKey())) {
                    submitTestInstruction.setJiraProjectName(project.getName());
                }
            }
        } catch (Exception e) {
            log.info("获取jira项目Map失败.");
            e.printStackTrace();
        }
        return JsonResult.buildSuccessResult(submitTestInstruction);
    }
}
