package cn.qg.holmes.controller.k8s;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.config.annotation.RequiresPermissions;
import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.service.k8s.DockerProjectService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

@CrossOrigin
@RestController
@RequestMapping("/docker/project")
public class DockerProjectController {

    @Autowired
    DockerProjectService dockerProjectService;

    /**
     * 获取docker project列表
     *
     * @param pageNum  第几页
     * @param pageSize 每页多少个
     * @return
     */
    @GetMapping("/list")
    public JsonResult getDockerProjectList(String projectName, String projectType, Integer isActive,
                                           @RequestParam(defaultValue = "1") Integer pageNum,
                                           @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage projectIPage = dockerProjectService.getDockerProjectList(projectName, projectType, isActive, pageNum, pageSize);
        Map<String, Object> map = new HashMap<>();
        map.put("total", projectIPage.getTotal());
        map.put("list", projectIPage.getRecords());
        return JsonResult.buildSuccessResult(map);
    }

    /**
     * 新增docker project
     *
     * @param dockerProject docker项目实体类
     * @return
     */
    @RequiresPermissions("docker:proconfig:add")
    @PostMapping("/add")
    public JsonResult addDockerProject(@RequestBody DockerProject dockerProject) {
        return JsonResult.buildSuccessResult(dockerProjectService.saveOrUpdate(dockerProject));
    }

    /**
     * 编辑docker project
     *
     * @param dockerProject docker项目实体类
     * @return
     */
    @RequiresPermissions("docker:proconfig:edit")
    @PostMapping("/edit")
    public JsonResult editDockerProject(@RequestBody DockerProject dockerProject) {
        return JsonResult.buildSuccessResult(dockerProjectService.saveOrUpdate(dockerProject));
    }

    /**
     * 删除docker project
     *
     * @param dockerProjectId 删除id
     * @return
     */
    @GetMapping("/del")
    public JsonResult delDockerProject(@RequestParam Integer dockerProjectId) {
        return JsonResult.buildSuccessResult(dockerProjectService.removeById(dockerProjectId));
    }

    @GetMapping("/detail")
    public JsonResult getProjectDetail(@RequestParam String projectName) {
        QueryWrapper<DockerProject> dockerProjectQueryWrapper = new QueryWrapper<>();
        dockerProjectQueryWrapper.eq("project_name", projectName);
        return JsonResult.buildSuccessResult(dockerProjectService.getOne(dockerProjectQueryWrapper));
    }

    @GetMapping("/type")
    public JsonResult getType() {
        QueryWrapper<DockerProject> dockerProjectQueryWrapper = new QueryWrapper<>();
        dockerProjectQueryWrapper
                .groupBy("project_type")
                .orderByDesc("count(*)")
                .select("project_type as projectType", "count(*) as total");
        return JsonResult.buildSuccessResult(dockerProjectService.listMaps(dockerProjectQueryWrapper));
    }

    @GetMapping("/info")
    public JsonResult getInfo(@RequestParam String projectName) {
        QueryWrapper<DockerProject> dockerProjectQueryWrapper = new QueryWrapper<>();
        dockerProjectQueryWrapper.eq("project_name", projectName);
        DockerProject dockerProject = dockerProjectService.getOne(dockerProjectQueryWrapper);
        Map<String, Object> map = new HashMap<>();
        map.put("id", dockerProject.getId());
        map.put("type", dockerProject.getProjectType());
        map.put("git_path", dockerProject.getGitPath());
        map.put("auth", dockerProject.getOwner());
        map.put("desc", dockerProject.getDesc());
        map.put("is_active", dockerProject.getIsActive() == 1);
        map.put("deploy_to_testenv", dockerProject.getDeployToDocker() == 1);
        map.put("host_name", dockerProject.getHostName());
        map.put("git_lab", dockerProject.getGitPath().split(":")[0].split("@")[1]);
        map.put("git_path_group", dockerProject.getGitPathGroup());
        map.put("project_name", dockerProject.getProjectName());
        map.put("sonar", dockerProject.getSonar() == 1);
        map.put("build_command", dockerProject.getBuildCommand());
        map.put("createdAt", dockerProject.getCreateTime());
        map.put("updatedAt", dockerProject.getUpdateTime());
        return JsonResult.buildSuccessResult(map);
    }
}
