package cn.qg.holmes.controller.jira;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.jira.JiraBugPool;
import cn.qg.holmes.entity.jira.SendScheduleVo;
import cn.qg.holmes.service.jira.JiraBugPoolService;
import cn.qg.holmes.service.jira.JiraIssueService;
import cn.qg.holmes.utils.DingdingUtils;
import com.atlassian.jira.rest.client.api.domain.Issue;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@CrossOrigin
@RestController
@RequestMapping("/jira")
public class JiraController {

    @Autowired
    JiraIssueService jiraIssueService;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    @GetMapping("/report/project")
    public void sendReportToDingdingGroup(String project, String webhook) throws Exception {
        Iterable<Issue> issues = jiraIssueService.getUnsolvedIssueListByProject(project);
        String markdownMsg = DingdingUtils.buildMarkdownMsgForUnsolvedIssueList(issues);
        DingdingUtils.sendToDingding(markdownMsg, webhook);
    }

    /**
     * 发送测试进度
     * @return
     * @throws Exception
     */
    @PostMapping("/send/schedule")
    public JsonResult sendTestReport(@RequestBody SendScheduleVo sendScheduleVo) throws Exception {
        String JQL = "project = " + sendScheduleVo.getJiraProjectKey() + " AND resolution = Unresolved AND text ~ \"" + sendScheduleVo.getTestProjectName() +  "\" ORDER BY priority DESC, updated DESC";
        Iterable<Issue> issues = jiraIssueService.getIssueListByJQL(JQL);
        String markdownMsg = DingdingUtils.buildTestScheduleMarkdownMsg(issues, sendScheduleVo.getProgress(), sendScheduleVo.getRisk());
        return JsonResult.buildSuccessResult(DingdingUtils.sendToDingding(markdownMsg, sendScheduleVo.getWebhook()));
    }

    @GetMapping("/test")
    public JsonResult test() throws Exception{
        String JQL = "project = XCX AND text ~ KA乐信 ORDER BY key ASC, priority DESC, updated DESC";
        Iterable<Issue> issues = jiraIssueService.getIssueListByJQL(JQL);
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        List<Map<String, Object>> result = new ArrayList<>();
        for (Issue issue: issues) {
            Map<String, Object> map = new HashMap<>();
            map.put("标题", issue.getSummary());
            map.put("报告人", issue.getReporter().getDisplayName());
            map.put("经办人", issue.getAssignee().getDisplayName());
            map.put("优先级", issue.getPriority().getName());
            map.put("创建时间", simpleDateFormat.format(issue.getCreationDate().toDate()));
            if (issue.getStatus().getName().equals("关闭") || issue.getStatus().getName().equals("已解决")) {
                map.put("解决时间", simpleDateFormat.format(DateTime.parse(issue.getFieldByName("已解决").getValue().toString()).toDate()));
            } else {
                map.put("解决时间", null);
            }
            map.put("解决时长", jiraIssueService.calculateBugFixTime(issue.getCreationDate(), DateTime.parse(issue.getFieldByName("已解决").getValue().toString())).toString());
            result.add(map);
        }
        return JsonResult.buildSuccessResult(result);
    }

    @GetMapping("/bug/pool/caculate")
    public JsonResult bugPoolCaculateFixTime() throws Exception {
        QueryWrapper<JiraBugPool> jiraBugPoolQueryWrapper = new QueryWrapper<>();
        jiraBugPoolQueryWrapper.eq("enable", 1);
        List<JiraBugPool> jiraBugPoolList = jiraBugPoolService.list(jiraBugPoolQueryWrapper);
        List<Map<String, Object>> mapList = new ArrayList<>();
        for (JiraBugPool jiraBugPool : jiraBugPoolList) {
            Issue issue = jiraIssueService.getJiraIssueByKey(jiraBugPool.getKey());
            String resolveResult = issue.getStatus().getName();
            DateTime startDate = issue.getCreationDate();
            DateTime endDate;
            // 如果已解决或已关闭，仅修改状态
            if (resolveResult.equals("已解决") || resolveResult.equals("关闭")) {
                endDate = DateTime.parse(issue.getFieldByName("已解决").getValue().toString());
            } else {
                endDate = new DateTime();
            }
            int duration = jiraIssueService.calculateBugFixTime(startDate, endDate);
            Map<String, Object> map = new HashMap<>();
            map.put("title", issue.getSummary());
            map.put("持续时长", duration);
            mapList.add(map);
        }
        return JsonResult.buildSuccessResult(mapList);
    }
}
