package cn.qg.holmes.controller.k8s;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.auth.UserInfoVo;
import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.EnvTemplateDetail;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import cn.qg.holmes.service.auth.TokenService;
import cn.qg.holmes.service.k8s.*;
import cn.qg.holmes.utils.MongoUtils;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.tencentcloudapi.tcr.v20190924.models.RepoInfo;
import com.tencentcloudapi.tcr.v20190924.models.TagInfo;
import io.fabric8.kubernetes.api.model.*;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * k8及容器操作相关接口
 * @author libo
 */
@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/k8s")
public class K8sController {

    @Autowired
    TkeService tkeService;

    @Autowired
    ImageService imageService;

    @Autowired
    DockerProjectService dockerProjectService;

    @Autowired
    MqService mqService;

    @Autowired
    RedisService redisService;

    @Autowired
    TokenService tokenService;

    @Autowired
    EnvTemplateService envTemplateService;

    @Autowired
    EnvTemplateDetailService envTemplateDetailService;

    @Autowired
    RedisUtils redisUtils;

    @Value("${mongodb.sync.list}")
    private String mongodbSyncStr;

    /**
     * 获取namespace列表
     * @return
     */
    @GetMapping("/namespace/list")
    public JsonResult getNamespaceList(String env) {
        return JsonResult.buildSuccessResult(tkeService.getNamespaceList(env));
    }

    /**
     * 获取pod列表
     * @param namespace 环境
     * @return
     */
    @GetMapping("/service/list")
    public JsonResult getServiceList(@RequestParam String namespace) {
        List<Map<String, Object>> podList = tkeService.getPodList(namespace);
        return JsonResult.buildSuccessResult(podList);
    }

    @GetMapping("/service/detail")
    public JsonResult getServiceDetail(@RequestParam String namespace,
                                       @RequestParam String serviceType,
                                       @RequestParam String serviceName) {
        return JsonResult.buildSuccessResult(tkeService.getPodAndServiceInfo(namespace, serviceType, serviceName));
    }

    /**
     * 根据服务名称获取镜像列表
     * @param repoName 服务名称
     * @return
     */
    @GetMapping("/image/list")
    public JsonResult getImageListByServiceName(@RequestParam String repoName) {
        return JsonResult.buildSuccessResult(imageService.getImageListByService(repoName));
    }

    /**
     * 新增deployment, 如果service和pvc没有，也会同时创建
     * @param serviceCreateVo 创建实体类
     * @return
     */
    @PostMapping("/service/create")
    public JsonResult createPod(@RequestBody ServiceCreateVo serviceCreateVo) {
        try {
            String namespace = serviceCreateVo.getNamespace();
            String serviceName = serviceCreateVo.getServiceName();
            String label = serviceCreateVo.getLabel();
            if (StringUtils.equals(label, "base")) {
                // 基础服务，如果没有PVC则创建
                if (!tkeService.checkPvcExistence(namespace, serviceName)) {
                    tkeService.createPvcByYaml(serviceCreateVo);
                }
            } else {
                // 非基础服务，创建Ingress
                tkeService.createIngressByYaml(serviceCreateVo);
            }
            // 新增Service
            tkeService.createServiceByYaml(serviceCreateVo);
            // 新增Deployment
            tkeService.createDeploymentByYaml(serviceCreateVo);
        } catch (Exception e) {
            log.info("服务创建异常.", e);
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("服务创建异常", false);
        }
        return JsonResult.buildSuccessResult("服务创建成功");
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName podName
     * @return
     */
    @PostMapping("/service/redeploy")
    public JsonResult resetPodByName(String namespace, String podName) {
        return JsonResult.buildSuccessResult(tkeService.resetPod(namespace, podName));
    }

    /**
     * jenkins打包部署镜像接口，专门提供给global-jenkinsfile使用
     * @param serviceCreateVo
     * @return
     */
    @PostMapping("/service/modify")
    public JsonResult modifyPod(@RequestBody ServiceCreateVo serviceCreateVo) {
        log.info("接口/k8s/service/modify收到请求，请求参数为：{}", JSON.toJSONString(serviceCreateVo));
        String serviceName = serviceCreateVo.getServiceName();
        String namespace = serviceCreateVo.getNamespace();
        //  默认mock为0
        if (serviceCreateVo.getMock() ==  null) {
            serviceCreateVo.setMock(0);
        }
        // 默认debug为0
        if (serviceCreateVo.getDebug() == null) {
            serviceCreateVo.setDebug(0);
        }

        // 如果原来服务已存在，则按照原debug和mock的值，设置serviceCreateVo
        if (tkeService.checkDeploymentExistence(namespace, serviceName)) {
            Deployment deployment = tkeService.getDeployment(namespace, serviceName);
            List<Container> containers =  deployment.getSpec().getTemplate().getSpec().getContainers();
            if (containers != null && containers.size() > 0) {
                List<EnvVar> envVarList = containers.get(0).getEnv();
                // 根据原来的debug和mock值，来设置debug和mock
                for (EnvVar envVar: envVarList) {
                    if (envVar.getName().equals("DEBUG")) {
                        serviceCreateVo.setDebug(Integer.valueOf(envVar.getValue()));
                    }
                    if (envVar.getName().equals("MOCK")) {
                        serviceCreateVo.setMock(Integer.valueOf((envVar.getValue())));
                    }
                }
            }
        }
        Service service = tkeService.getServiceDetail(namespace, serviceName);
        if (service == null) {
            // 创建或替换Service
            tkeService.createServiceByYaml(serviceCreateVo);
            try {
                Thread.sleep(3000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        // 创建或替换Deployment
        tkeService.createDeploymentByYaml(serviceCreateVo);
        // 创建或替换Ingress
        tkeService.createIngressByYaml(serviceCreateVo);

        return JsonResult.buildSuccessResult("镜像部署成功");
    }

    /**
     * 更新pod
     * @param serviceCreateVo
     * @return
     */
    @PostMapping("/service/update")
    public JsonResult updatePod(@RequestBody ServiceCreateVo serviceCreateVo) {
        try {
            tkeService.createServiceByYaml(serviceCreateVo);
            tkeService.createDeploymentByYaml(serviceCreateVo);
            tkeService.createIngressByYaml(serviceCreateVo);
            return JsonResult.buildSuccessResult("更新成功！", true);
        } catch (Exception e) {
            e.printStackTrace();
            log.info("更新服务失败: {}", serviceCreateVo);
            return JsonResult.buildErrorStateResult("更新失败", false);
        }
    }

    /**
     * 删除pod
     * @param namespace 环境
     * @param serviceName 服务名称
     * @return
     */
    @PostMapping("/service/delete")
    public JsonResult deletePodByName(String namespace, String serviceName, String serviceType) {
        try {
            // 删除Service
            tkeService.deleteService(namespace, serviceName);
            // 删除Deployment
            tkeService.deleteDeployment(namespace, serviceName);
            if (!StringUtils.equals("base", serviceType)) {
                // 非基础服务，删除Ingress
                tkeService.deleteIngress(namespace, serviceName);
            }
            return JsonResult.buildSuccessResult("服务删除成功!", true);
        } catch (Exception e) {
            e.printStackTrace();
            log.info("删除服务异常：namespace: {}, serviceName: {}, serviceType: {}", namespace, serviceName, serviceType);
            return JsonResult.buildErrorStateResult("服务删除失败!", false);
        }
    }

    /**
     * 修改服务类型：ClusterIP 或者 NodePort
     * @param namespace
     * @param serviceName
     * @param type
     * @return
     */
    @PostMapping("/service/changeType")
    public JsonResult changeServiceType(String namespace, String serviceName, Integer type) {
        Service service = tkeService.getServiceDetail(namespace, serviceName);
        if (service == null) {
            return JsonResult.buildErrorStateResult("Service不存在!", false);
        }
        if (type == 1) {
            service.getSpec().setType("ClusterIP");
            service.getSpec().getPorts().forEach(servicePort -> servicePort.setNodePort(null));
            service.getSpec().setClusterIP(null);
            service.getSpec().setExternalTrafficPolicy(null);
        } else if (type == 2) {
            service.getSpec().setType("NodePort");
            service.getSpec().setClusterIP(null);
        } else {
            return JsonResult.buildErrorStateResult("暂不支持此类型!", false);
        }
        boolean modifyResult = tkeService.modifyK8sService(service);
        return modifyResult?JsonResult.buildSuccessResult("Service修改成功!"):JsonResult.buildErrorStateResult("Service修改失败!", false);
    }

    /**
     * 根据类型获取所有的镜像名
     * @param namespace qa-base，qa-ui，qa-java等
     * @return
     */
    @GetMapping("/getSingleTypeApp")
    public JsonResult getSingleTypeApp(@RequestParam String namespace, @RequestParam String type) {
        List<RepoInfo> repoInfoList = imageService.getRepoInfoList(namespace);
        Map<String, Object> resultMap =  new HashMap<>();
        if (type.equals("base")) {
            resultMap.put("RepoInfo", repoInfoList);
            resultMap.put("TotalCount", repoInfoList.size());
            return JsonResult.buildSuccessResult(resultMap);
        }
        if (repoInfoList.size() > 0) {
            List<RepoInfo> newRepoInfoList = new ArrayList<>();
            QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
            queryWrapper
                    .eq("project_type", type)
                    .eq("is_active", 1)
                    .eq("deploy_to_docker", 1);
            List<DockerProject> dockerProjectList = dockerProjectService.list(queryWrapper);
            for (RepoInfo repoInfo: repoInfoList) {
                String projectName = repoInfo.getRepoName().split("/")[1];
                for (DockerProject dockerProject: dockerProjectList) {
                    if (dockerProject.getProjectName().equals(projectName)) {
                        newRepoInfoList.add(repoInfo);
                    }
                }
            }
            resultMap.put("RepoInfo", newRepoInfoList);
            resultMap.put("TotalCount", newRepoInfoList.size());
            return JsonResult.buildSuccessResult(resultMap);
        } else {
            return JsonResult.buildSuccessResult(null);
        }
    }

    /**
     * 同步线上mq数据到测试
     * @param host 测试环境mq地址
     * @return
     */
    @PostMapping("/sync/mq")
    public JsonResult syncOnlineMqToTest(@RequestParam String host) {
        return JsonResult.buildSuccessResult(mqService.setDefinitions(host));
    }

    /**
     * 比较线上和测试mq
     * @param host
     * @return
     */
    @GetMapping("/mq/diff")
    public JsonResult getMqDiffResult(@RequestParam String host) {
        return JsonResult.buildSuccessResult(mqService.getMqDiff(host));
    }

    /**
     * 情况测试环境redis
     * @param namespace 环境
     * @return
     */
    @PostMapping("/redis/flush")
    public JsonResult flushRedis(@RequestParam String namespace) {
        try {
            Map<String, Object> redisInfo = tkeService.getPodAndServiceInfo(namespace, "base", "redis");
            String lanIp = redisInfo.get("lanIp").toString();
            List<Map<String, Object>> portMappings = (List<Map<String, Object>>) redisInfo.get("portMappings");
            for (Map<String, Object> portMap: portMappings) {
                redisService.flushRedis(lanIp, (Integer) portMap.get("nodePort"));
            }
            return JsonResult.buildSuccessResult("redid清除成功！", true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("redis清除失败！", false);
        }
    }

    /**
     * 创建新的namespace，同时需要根据模板构建对应服务
     */
    @PostMapping("/namespace/create")
    public JsonResult createNewNamespace(@RequestParam String name,
                                         @RequestParam String desc,
                                         @RequestParam String owner,
                                         @RequestParam String templateId) {
        boolean createNamespace = tkeService.createNewNamespace(name, desc, owner);
        if (!createNamespace) {
            return JsonResult.buildErrorStateResult("环境创建失败！", false);
        }
        QueryWrapper<EnvTemplateDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("template_id", templateId);
        List<EnvTemplateDetail> envTemplateDetailList = envTemplateDetailService.list(queryWrapper);
        for (EnvTemplateDetail envTemplateDetail: envTemplateDetailList) {
            String label = envTemplateDetail.getLabel();

            ServiceCreateVo serviceCreateVo = new ServiceCreateVo();
            serviceCreateVo.setServiceName(envTemplateDetail.getServiceName());
            serviceCreateVo.setNamespace(name);
            serviceCreateVo.setImage(envTemplateDetail.getTag());
            serviceCreateVo.setType(envTemplateDetail.getServiceType());
            serviceCreateVo.setDebug(0);
            serviceCreateVo.setMock(0);
            serviceCreateVo.setLabel(envTemplateDetail.getLabel());
            serviceCreateVo.setDomain(envTemplateDetail.getDomain());

            // 处理基础服务
            if (label.equals("base")) {
                tkeService.createPvcByYaml(serviceCreateVo);
                tkeService.createServiceByYaml(serviceCreateVo);
                tkeService.createDeploymentByYaml(serviceCreateVo);
            } else {
                // 业务服务
                tkeService.createServiceByYaml(serviceCreateVo);
                tkeService.createDeploymentByYaml(serviceCreateVo);
                tkeService.createIngressByYaml(serviceCreateVo);
            }
        }
        // 删除相关缓存
        redisUtils.del("k8s:namespace:all");
        redisUtils.del("k8s:namespace:test");
        redisUtils.del("k8s:namespace:dev");
        return JsonResult.buildSuccessResult("环境创建成功");
    }

    /**
     * 根据roleId获取有权限展示按钮的namespace
     */
    @GetMapping("/role/namespace")
    public JsonResult getNamespaceListByRoleId(@RequestHeader String token) {
        UserInfoVo userInfoVo = tokenService.getUserInfoFromCache(token);
        Integer roleId = userInfoVo.getRoleIds().get(0);
        List<String> nsStrList = new ArrayList<>();
        List<Map<String, Object>> nsMapList = new ArrayList<>();
        if (roleId == 1 || roleId == 2) {
            nsMapList = tkeService.getNamespaceList("all");
        }
        if (roleId == 3) {
            nsMapList = tkeService.getNamespaceList("dev");
        }
        for (Map map: nsMapList) {
            nsStrList.add(map.get("name").toString());
        }
        return JsonResult.buildSuccessResult(String.join(",", nsStrList));
    }

    /**
     * 给jenkins构建提供的namespace列表
     * @param env all、test、dev
     * @return
     */
    @GetMapping("/get_namespace_for_jenkins")
    public String getNamespaceForJenkins(@RequestParam String env) {
        List<Map<String, Object>> mapList = tkeService.getNamespaceList(env);
        List<String> resultList = new ArrayList<>();
        for (Map<String, Object> map: mapList) {
            resultList.add(map.get("name").toString());
        }
        resultList.sort(String::compareTo);
        return String.join("\n", resultList);
    }

    /**
     * 按照类型获取所有的服务
     * @return
     */
    @GetMapping("/repo/getAll")
    public JsonResult getAllRepos() {
        List<Map<String, Object>> resultMapList = new ArrayList<>();
        List<RepoInfo> repoInfoList = imageService.getRepoInfoList("qa-base");
        Map<String, Object> base = new HashMap<>();
        List<Map<String, String>> baseList = new ArrayList<>();
        for (RepoInfo repoInfo: repoInfoList) {
            String serviceName = repoInfo.getRepoName().split("/")[1];
            TagInfo[] tagInfos = imageService.getImageListByService(repoInfo.getRepoName()).getTagInfo();
            for (TagInfo tagInfo: tagInfos) {
                Map<String, String> map = new HashMap<>();
                map.put("serviceName", serviceName + ":" + tagInfo.getTagName());
                map.put("serviceType", serviceName);
                map.put("label", "base");
                map.put("tag", "qa-base/" + serviceName + ":" + tagInfo.getTagName());
                map.put("domain", serviceName + tagInfo.getTagName());
                baseList.add(map);
            }
        }
        base.put("type", "base");
        base.put("repos", baseList);
        resultMapList.add(base);

        QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
        queryWrapper.groupBy("project_type").select("project_type");
        List<DockerProject> projectList = dockerProjectService.list(queryWrapper);
        queryWrapper.clear();

        for (DockerProject dp: projectList) {
            String type = dp.getProjectType();
            Map<String, Object> serviceMap = new HashMap<>();
            List<Map<String, String>> serviceMapList = new ArrayList<>();
            queryWrapper
                    .eq("project_type", type)
                    .eq("is_active", 1)
                    .eq("deploy_to_docker", 1);
            List<DockerProject> dockerProjectList = dockerProjectService.list(queryWrapper);
            for (DockerProject dockerProject: dockerProjectList) {
                Map<String, String> map = new HashMap<>();
                map.put("serviceType", type);
                map.put("label", type);
                map.put("serviceName", dockerProject.getProjectName());
                map.put("domain", dockerProject.getHostName().split("\\.")[0]);
                map.put("tag", "qa-test/" + dockerProject.getProjectName() + ":latest");
                serviceMapList.add(map);
            }
            serviceMap.put("type", type);
            serviceMap.put("repos", serviceMapList);
            resultMapList.add(serviceMap);
            queryWrapper.clear();
        }

        return JsonResult.buildSuccessResult(resultMapList);
    }

    /**
     * 给打包用的，获取dubbo接口
     * @param namespace 环境
     * @param serviceType 服务类型
     * @param serviceName 服务名称
     * @param key port_20880
     * @return
     */
    @GetMapping("/dubbo/port")
    public String getDubboPort(@RequestParam String namespace,
                               @RequestParam String serviceType,
                               @RequestParam String serviceName,
                               @RequestParam String key) {
        List<ServicePort> servicePortList = tkeService.getServiceDetail(namespace, serviceName).getSpec().getPorts();
        Map<String, Object> serviceMap = new HashMap<>();
        List<Map<String, Object>> portMappingList = new ArrayList<>();
        if (servicePortList.size() > 0) {
            for (ServicePort servicePort : servicePortList) {
                if (servicePort.getNodePort() != null) {
                    serviceMap.put("port_" + servicePort.getName(), servicePort.getNodePort());
                }
                Map<String, Object> portMap = new HashMap<>();
                portMap.put("name", servicePort.getName());
                portMap.put("nodePort", servicePort.getNodePort());
                portMap.put("port", servicePort.getPort());
                portMap.put("protocol", servicePort.getProtocol());
                portMap.put("targetPort", servicePort.getTargetPort());
                portMappingList.add(portMap);
            }
        }
        serviceMap.put("portMappings", portMappingList);
        serviceMap.put("serviceName", serviceName);
        serviceMap.put("namespace", namespace);
        if (serviceMap.containsKey(key)) {
            return serviceMap.get(key).toString();
        } else {
            return JSON.toJSONString(serviceMap);
        }
    }

    /**
     * 同步dbList
     * @param host mongodb host
     * @param port mongodb port
     * @return
     */
    @GetMapping("/sync/mongo")
    public JsonResult syncMongodb(@RequestParam String host, @RequestParam int port) {
        Map<String, Object> map = new HashMap<>();
        List<String> mongodbSyncList = Arrays.asList(mongodbSyncStr.split(","));
        for (String dbName: mongodbSyncList) {
            String result = MongoUtils.createDbWithAuthentication(host, port, dbName, "qa", "qatest");
            if (result != null) {
                map.put(dbName, "success");
            } else {
                map.put(dbName, "fail");
            }
        }
        return JsonResult.buildSuccessResult(map);
    }

    @GetMapping("/clear/is")
    public JsonResult clearIs(@RequestParam String env) {
        tkeService.clearRubbish(env);
        return JsonResult.buildSuccessResult(true);
    }

    /**
     * 修改service指向
     * @param namespace 环境
     * @param sourceServiceName 源Service
     * @param targetServiceName 目标Service
     * @return
     */
    @PostMapping("/service/redirect")
    public JsonResult modifyServiceDirect(String namespace, String sourceServiceName, String targetServiceName) {
        return JsonResult.buildSuccessResult(tkeService.modifyServicePointing(namespace, sourceServiceName, targetServiceName));
    }
}
