package cn.qg.holmes.utils;

import com.alibaba.fastjson.JSON;
import com.atlassian.jira.rest.client.api.domain.Issue;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Days;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 钉钉发送消息类
 * @author libo
 */
@Slf4j
public class DingdingUtils {

    /**
     * 发送钉钉消息基本方法
     * @param jsonString 发送的json字符串
     * @param webhook 钉钉webhook
     * @return 是否发送成功，true/false
     */
    public static boolean sendToDingding(String jsonString, String webhook) {
        Map<String, String> headers = new HashMap<>();
        headers.put("Content-Type", "application/json; charset=utf-8");
        String result = HttpClientUtils.doPostJson(webhook, headers, jsonString);
        log.info("发送给钉钉: {}, 内容：{}, 钉钉返回结果：{}", webhook, jsonString, result);
        Map<String, String> resultMap = JSON.parseObject(result, Map.class);
        return resultMap.get("errmsg").equals("ok");
    }

    /**
     * 构建markdown消息基础类
     * @param title 标题
     * @param content 内容
     * @param atAll 是否at所有人
     * @return markdown消息的json字符串
     */
    public static String buildMarkdownMsg(String title, String content, boolean atAll) {
        Map<String, Object> markdown = new HashMap<>();
        Map<String, String> contentMap = new HashMap<>();
        Map<String, Object> atMap = new HashMap<>();
        atMap.put("isAtAll", atAll);
        markdown.put("at", atMap);
        markdown.put("msgtype", "markdown");
        contentMap.put("title", title);
        contentMap.put("text", content);
        markdown.put("markdown", contentMap);
        return JSON.toJSONString(markdown);
    }

    /**
     * 构建@具体人的markdown消息，需要在text里加上@手机号，以及at下的atMobiles列表里添加需要被@的手机号
     * @param title 消息标题
     * @param content 内容
     * @param phoneNo 被@的手机号
     * @param atAll 是否@所有人
     * @return
     */
    public static String buildMarkdownMsgAtSpecifyPhone(String title, String content, String phoneNo, boolean atAll) {
        Map<String, Object> markdown = new HashMap<>();
        Map<String, String> contentMap = new HashMap<>();
        Map<String, Object> atMap = new HashMap<>();
        List<String> atMobileList = new ArrayList<>();
        atMobileList.add(phoneNo);
        atMap.put("atMobiles", atMobileList);
        atMap.put("isAtAll", atAll);
        markdown.put("at", atMap);
        markdown.put("msgtype", "markdown");
        contentMap.put("title", title);
        contentMap.put("text", content);
        markdown.put("markdown", contentMap);
        return JSON.toJSONString(markdown);
    }

    /**
     * 构建BUG刚提交时，发送给钉钉的markdown消息
     * @param key jira issue key
     * @param summary 标题
     * @param creator 创建人
     * @param assignee 经办人
     * @param priority 优先级
     * @param module 模块
     * @return json字符串类型的markdown消息
     */
    public static String buildBugCommitMarkdownMsg(String key, String summary, String creator, String assignee, String priority, String module) {
        String title = "测试提交了BUG：" + key;
        String issueUrl = "http://jira2.quantgroup.cn/browse/" + key;
        String content = creator + "提交了BUG：" + key + "\n\n";
        content += "概要：[" + summary + "](" + issueUrl + ")"+ "\n\n"
                + "- 优先级：" + priority + "\n"
                + "- 模块：" + module + "\n"
                + "- 经办人：" + assignee;
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * for 数据工单
     * 根据issueList构建未解决issueList的markdown格式
     * @param issues Issue的迭代器
     * @return
     */
    public static String buildMarkdownMsgForUnsolvedIssueList(Iterable<Issue> issues) {
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        String title =  "今日剩余未解决BUG统计";
        String content = "#### 今日剩余未解决BUG数：" + issueList.size() + "\n";
        int count = 0;
        for (Issue issue: issueList) {
            count = count + 1;
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
            content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
            content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
            content = content + "- 状态：" + issue.getStatus().getName() + "\n";
            content = content + "- 经办人：" + issue.getAssignee().getDisplayName() + "\n\n";
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建测试进度的markdown消息
     * @param issues issue迭代器
     * @param progress 测试进度
     * @param risk 风险点
     * @return 测试进度markdown消息的json字符串
     */
    public static String buildTestScheduleMarkdownMsg(Iterable<Issue> issues, String progress, String risk) {
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        Date date=new Date();
        DateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        String title =  format.format(date) + "今日测试进度汇报";
        String content = format.format(date) + "\n\n";
        content +=  "今日测试进度：" + progress + " \n\n";
        content = content + "剩余未解决BUG列表：" + "\n\n";
        if (issueList.size() == 0) {
            content += "暂无未解决BUG！\n\n";
        }
        int count = 0;
        for (Issue issue: issueList) {
            count += 1;
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
            content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
            content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
            content = content + "- 状态：" + issue.getStatus().getName() + "\n";
            String assignee = "";
            if (issue.getAssignee() == null) {
                assignee = "未分配";
            } else {
                assignee = issue.getAssignee().getDisplayName();
            }
            content = content + "- 经办人：" + assignee + "\n\n";
        }
        if (risk != null && !risk.isEmpty()) {
            content += "**风险：**\n\n";
            List<String> riskList = Arrays.asList(risk.split("\n"));
            for (String str: riskList) {
                content += "**" + str + "**\n\n";
            }
//            content =  content + "**风险：" + risk + "**\n";
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建BUG fix 提醒消息
     * @param issue Jira Issue
     * @return
     */
    public static String buildBugFixRemindMsg(Issue issue, Integer elapsed) {
        String duration = "";
        if (elapsed >= 4 && elapsed <8) {
            duration = elapsed.toString() + "小时";
        }
        if (elapsed >= 8) {
            duration = (elapsed / 8) + "天";
        }
        String title = "BUG修复提醒";
        String content = issue.getAssignee().getDisplayName() + "同学，你有一个BUG已经超过" + duration + "未解决，测试同学等得花儿都谢啦，麻烦赶紧修复哦！\n\n";
        String bugUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
        content += "BUG具体信息：\n\n";
        content += "- 标题：[" + issue.getSummary().replace("\"", "") + "](" + bugUrl + ")\n";
        content += "- 优先级：" + issue.getPriority().getName() + "\n";
        content += "- 经办人：" + issue.getAssignee().getDisplayName() + "\n\n";
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建pipeline钉钉消息
     * @param serviceName 服务名
     * @param serviceBranch 分支
     * @param creator 创建人
     * @param commitMsg 提交信息
     * @param date 日期
     * @param namespace 环境
     * @return
     */
    public static String buildPipelineMarkdownMsg(String serviceName, String serviceBranch, String serviceType,
                                                  String creator, String commitMsg, String date, String namespace, String buildNum) {
        String title = "测试环境自动构建提醒";
        String content = "> 项目名称：" + serviceName + "\n\n"
                + "> 分支名称：" + serviceBranch + "\n\n"
                + "> 部署空间：" + namespace + "\n\n"
                + "> 触发同学：" + creator + "\n\n"
                + "> 触发信息：" + commitMsg + "\n\n"
                + "> 触发时间：" + date + "\n\n"
                + "[查看详情](" + "http://172.29.1.15:8080/view/TKE/job/" + "tke-" + serviceType + "/" + buildNum + "/console"  + ")";
        return buildMarkdownMsg(title, content, false);
    }

    /**
     * 构建冒烟测试结果的钉钉消息
     * @param issues 未解决BUG列表
     * @param result 冒烟结果，true-冒烟测试通过，false-冒烟测试未通过
     * @param risk 风险点
     * @return
     */
    public static String buildSmokingResultMarkdownMsg(Iterable<Issue> issues, boolean result, String risk) {
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        String title = "";
        String content = "";
        DateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        String today =  format.format(new Date());
        content = today + "\n\n";
        if (result) {
            title = "冒烟测试通过！";
            content += "冒烟测试结果：✅\n\n亲爱的小伙伴们，经过冒烟测试，您的项目可以正式进入测试阶段，感谢您辛苦的努力，后面的bug还请多多支持呀！\n\n";
        } else {
            title = "冒烟测试未通过！";
            content += "冒烟测试结果：❌\n\n亲爱的小伙伴们，经过冒烟测试，您的项目无法正式进入测试阶段。因为有严重问题阻塞流程，请尽快修复后再次提测。由此耽误的时间算做您开发改bug的时间喔，测试时间会顺延到冒烟测试通过后的时间!\n\n";
        }
        if (issueList.size() > 0) {
            content += "冒烟测试BUG列表：\n\n";
            int count = 0;
            for (Issue issue: issueList) {
                count = count + 1;
                String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
                content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
                content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
                content = content + "- 状态：" + issue.getStatus().getName() + "\n";
                String assignee = "";
                if (issue.getAssignee() == null) {
                    assignee = "未分配";
                } else {
                    assignee = issue.getAssignee().getDisplayName();
                }
                content = content + "- 经办人：" + assignee + "\n\n";
            }
        }
        if (risk != null) {
            content += "**风险点：**" + risk;
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建发送给 项目上线后消息同步 的消息
     * @param projectName 项目名称
     * @param submitTestUrl 提测说明地址
     * @return
     */
    public static String buildTestSubmitInstructionMsg(String projectName, String submitTestUrl) {
        String title = "新项目提测通知";
        String content = "亲爱的数仓小伙伴，有新的项目提测啦。请关注：\n\n";
        content += "- 项目名：" + projectName + "\n";
        content += "- 项目详细信息：" + submitTestUrl + "\n\n";
        content += "测试已经开始接手项目，正在紧急的测试中，项目上线后还会触发一个上线机器人。请根据实际情况，关注项目上线后的情况~";
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建每日循环BUG池发送信息
     * @param issues
     * @return
     */
    public static String buildDailyCycleJiraBugPoolMsg(List<Issue> issues) {
        String title = "今日未解决BUG统计";
        String content = "📣今日未解决BUG统计：" + "\n\n";
        int count = 0;
        for (Issue issue: issues) {
            count = count + 1;
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
            content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
            content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
            content = content + "- 状态：" + issue.getStatus().getName() + "\n";
            String assignee = "";
            if (issue.getAssignee() == null) {
                assignee = "未分配";
            } else {
                assignee = issue.getAssignee().getDisplayName();
            }
            content = content + "- 经办人：" + assignee + "\n\n";
            DateTime startDate = issue.getCreationDate();
            DateTime curDate = new DateTime();
            int duration = calculateBugFixTime(startDate, curDate);
            content += "- 持续时间：" + duration + "小时\n\n";
        }
        content += "**截止当前，共有" + String.valueOf(count) + "个BUG未修复，为了不影响测试进度，请对应经办人尽快跟进解决一下～**";
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建每日线上未解决工单统计
     * @param issues jira bug 列表
     * @return
     */
    public static String buildDailyProdBugNotifyMsg(Iterable<Issue> issues) {
        String title = "线上未解决工单统计";
        String content = "📣线上未解决工单统计：" + "\n\n";
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        if (issueList.size() > 0) {
            int count = 0;
            for (Issue issue: issueList) {
                count = count + 1;
                String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
                content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
                content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
                content = content + "- 状态：" + issue.getStatus().getName() + "\n";
                String assignee = "";
                if (issue.getAssignee() == null) {
                    assignee = "未分配";
                } else {
                    assignee = issue.getAssignee().getDisplayName();
                }
                content = content + "- 经办人：" + assignee + "\n";
                DateTime startDate = issue.getCreationDate();
                DateTime curDate = new DateTime();
                int duration = calculateBugFixTime(startDate, curDate);
                content += "- 持续时间：" + duration + "小时\n\n";
            }
            content += "**截止当前，共有" + String.valueOf(count) + "个工单未解决，请对应经办人尽快跟进解决**";
        }
        if (issueList.size() == 0) {
            content += "**截止当前，所有工单都已经解决完毕，产研小伙伴最给力了👍🏻**";
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 计算BUG持续时间
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @return
     */
    public static Integer calculateBugFixTime(DateTime startDate, DateTime endDate) {
        int startHour = startDate.getHourOfDay();
        int endHour = endDate.getHourOfDay();
        int startMinute = startDate.getMinuteOfHour();
        int endMinute = endDate.getMinuteOfHour();
        int startSecond = startDate.getSecondOfMinute();
        int endSecond = endDate.getSecondOfMinute();
        boolean daySubFlag = true;
        if (endHour < startHour) {
            daySubFlag = false;
        }
        if (endHour == startHour && endMinute < startMinute) {
            daySubFlag = false;
        }
        if (endHour == startHour && endMinute == startMinute && endSecond < startSecond) {
            daySubFlag = false;
        }

        if (startHour >= 19) {
            startHour = 19;
        }
        if (startHour <= 10) {
            startHour = 10;
        }
        if (endHour >= 19) {
            endHour = 19;
        }
        if (endHour <= 10) {
            endHour = 10;
        }
        int hourDiff = endHour - startHour;
        int days = Days.daysBetween(startDate, endDate).getDays();
        return daySubFlag ? days * 8 + hourDiff : (days + 1) * 8 + hourDiff;
    }

    public static void main(String[] args) {
        String title = "测试同学提交BUG";
        String issueUrl = "http://jira2.quantgroup.cn/browse/CSXT-178";
        String content = "黎博提交了BUG：CSXT-178 \n\n";
        List<String> attachmentList = new ArrayList<>();
        String description = "姓名：张犇\n" +
                "手机号：15610309807\n" +
                "详情：客户表示申请借款无法操作，提示存在在途业务，无法申请借款，请处理，谢谢";
//        attachmentList.add("http://jira2.quantgroup.cn/secure/attachment/17004/%E5%BD%A9%E8%89%B2Q_%E7%94%BB%E6%9D%BF+1.jpg");
//        attachmentList.add("http://jira2.quantgroup.cn/secure/attachment/17013/image.png");
//        attachmentList.add("https://gw.alicdn.com/tfs/TB1ut3xxbsrBKNjSZFpXXcXhFXa-846-786.png");
        content += "概要：[" + "【定时推送】啦啦啦" + "](" + issueUrl + ")"+ "\n\n"
                + "- 优先级：" + "P3一般" + "\n"
                + "- 经办人：" + "@18311069063" + "\n\n";
        String[] descList = description.split("\n");
        for (String desc: descList) {
            content += "> " + desc + "\n\n";
        }
//        for (String attachment: attachmentList) {
//            content += "> ![](" + attachment + ")\n\n";
//        }
        String msg = buildMarkdownMsgAtSpecifyPhone(title, content, "18501033527", false);
        sendToDingding(msg, "https://oapi.dingtalk.com/robot/send?access_token=835663338d638e40daaf3ab358af741ef0680a826a962c91bedc53b149d85ee1");
    }
}
