package cn.qg.holmes.controller.gitlab;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.config.annotation.RequiresPermissions;
import cn.qg.holmes.entity.auth.UserInfoVo;
import cn.qg.holmes.entity.gitlab.DingRobotPipelineVo;
import cn.qg.holmes.entity.gitlab.Pipeline;
import cn.qg.holmes.entity.jira.DingRobot;
import cn.qg.holmes.service.auth.TokenService;
import cn.qg.holmes.service.gitlab.PipelineService;
import cn.qg.holmes.service.jira.DingRobotService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@CrossOrigin
@RequestMapping("/pipeline")
public class PipelineController {

    @Autowired
    PipelineService pipelineService;

    @Autowired
    DingRobotService dingRobotService;

    @Autowired
    HttpServletRequest servletRequest;

    @Autowired
    TokenService tokenService;

    /**
     * gitlab webhook
     * @param request
     */
    @PostMapping("/webhook")
    public void piplineWebhook(HttpServletRequest request) {
        StringBuilder piplineData = new StringBuilder();
        try {
            BufferedReader br = request.getReader();
            String str = "";
            while((str = br.readLine()) != null){
                piplineData.append(str);
            }
            br.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.info("收到gitlab推送来的数据：{}", piplineData);
        pipelineService.buildPipelineJob(String.valueOf(piplineData));
    }

    @GetMapping("/list")
    public JsonResult getDingRobotAndPipelineList(String jiraProjectKey, String projectName, String namespace, Integer status,
                                                  @RequestParam(defaultValue = "1") Integer pageNum,
                                                  @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<DingRobot> page = new Page<>(pageNum, pageSize);
        QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper();
        if (!StringUtils.isEmpty(jiraProjectKey)) {
            dingRobotQueryWrapper.eq("jira_project_key", jiraProjectKey);
        }
        if (!StringUtils.isEmpty(projectName)) {
            dingRobotQueryWrapper.like("project_name", projectName);
        }
        if (!StringUtils.isEmpty(namespace)) {
            dingRobotQueryWrapper.eq("namespace", namespace);
        }
        if (status != null) {
            dingRobotQueryWrapper.eq("status", status);
        }
        dingRobotQueryWrapper.orderByDesc("id");
        IPage<DingRobot> projectRobotIPage = dingRobotService.page(page, dingRobotQueryWrapper);
        List<DingRobot> dingRobotList = projectRobotIPage.getRecords();
        List<DingRobotPipelineVo> dingRobotPipelineVoList = new ArrayList<>();
        for (DingRobot dingRobot: dingRobotList) {
            DingRobotPipelineVo dingRobotPipelineVo = new DingRobotPipelineVo();
            dingRobotPipelineVo.setDingRobot(dingRobot);
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobot.getId());
            List<Pipeline> pipelineList = pipelineService.list(pipelineQueryWrapper);
            dingRobotPipelineVo.setPipelineList(pipelineList);
            dingRobotPipelineVoList.add(dingRobotPipelineVo);
        }
        Map<String, Object> map = new HashMap<>();
        map.put("total", projectRobotIPage.getTotal());
        map.put("list", dingRobotPipelineVoList);
        return JsonResult.buildSuccessResult(map);
    }

    /**
     * 新增ding robot 以及对应的pipeline
     * @return
     */
    @RequiresPermissions("qa:process:add")
    @PostMapping("/add")
    public JsonResult addRobotAndPipeline(@RequestBody DingRobotPipelineVo dingRobotPipelineVo) {
        try {
            DingRobot dingRobot = dingRobotPipelineVo.getDingRobot();
            // 设置创建人
            String token = servletRequest.getHeader("token");
            UserInfoVo userInfoVo = tokenService.getUserInfoFromCache(token);
            dingRobot.setCreator(userInfoVo.getChineseName());
            String projectName = dingRobot.getProjectName();
            String jiraProjectKey = dingRobot.getJiraProjectKey();
            QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper<>();
            dingRobotQueryWrapper.eq("jira_project_key", jiraProjectKey);
            dingRobotQueryWrapper.eq("project_name", projectName);
            // 根据jira project key和项目名判断是否已存在
            if (dingRobotService.getOne(dingRobotQueryWrapper) != null) {
                return JsonResult.buildErrorStateResult("项目已存在", false);
            }
            boolean dingRobotResult = dingRobotService.save(dingRobot);
            if (!dingRobotResult) {
                return JsonResult.buildErrorStateResult("服务器异常，保存失败！", false);
            }
            // 获取该robot id，并赋值给pipelineList
            DingRobot dingRobotNew = dingRobotService.getOne(dingRobotQueryWrapper);
            Integer dingRobotId = dingRobotNew.getId();
            List<Pipeline> pipelineList = dingRobotPipelineVo.getPipelineList();
            // 判断pipelineList是否为空
            if (pipelineList.size() > 0) {
                for (Pipeline pipeline: pipelineList) {
                    pipeline.setDingRobotId(dingRobotId);
                }
                pipelineService.saveBatch(pipelineList);
            }
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("新增失败", false);
        }
    }

    /**
     * 修改pipeline
     * @param dingRobotPipelineVo
     * @return
     */
    @RequiresPermissions("qa:process:edit")
    @PostMapping("/modify")
    public JsonResult modifyPipeline(@RequestBody DingRobotPipelineVo dingRobotPipelineVo) {
        try {
            DingRobot dingRobot = dingRobotPipelineVo.getDingRobot();
            boolean dingRobotResult = dingRobotService.saveOrUpdate(dingRobot);
            if (!dingRobotResult) {
                return JsonResult.buildErrorStateResult("修改失败！", false);
            }
            Integer dingRobotId = dingRobot.getId();
            List<Pipeline> pipelineList = dingRobotPipelineVo.getPipelineList();
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobotId);
            List<Pipeline> pipelineListInDb = pipelineService.list(pipelineQueryWrapper);
            // 判断是否有删除的部分
            if (pipelineListInDb.size() > 0) {
                for (Pipeline pipelineDb: pipelineListInDb) {
                    boolean flag = false;
                    if (pipelineList.size() > 0) {
                        for (Pipeline pipeline: pipelineList) {
                            if (pipelineDb.getId().equals(pipeline.getId())) {
                                flag = true;
                                break;
                            }
                        }
                    }
                    if (!flag) {
                        pipelineService.removeById(pipelineDb.getId());
                    }
                }
            }

            // 判断是否有新增的
            for (Pipeline pipeline: pipelineList) {
                if (pipeline.getId() == null) {
                    pipeline.setDingRobotId(dingRobotId);
                }
            }
            // 剩下的直接更新
            pipelineService.saveOrUpdateBatch(pipelineList);
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("修改失败！", false);
        }
    }

    /**
     * 删除pipeline
     * @param dingRobotId
     * @return
     */
    @RequiresPermissions("qa:process:del")
    @GetMapping("/delete")
    public JsonResult deletePipeline(@RequestParam Integer dingRobotId) {
        try {
            if (dingRobotService.getById(dingRobotId) != null) {
                dingRobotService.removeById(dingRobotId);
            }
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobotId);
            if (pipelineService.list(pipelineQueryWrapper) != null) {
                pipelineService.remove(pipelineQueryWrapper);
            }
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            return JsonResult.buildErrorStateResult("删除失败", false);
        }
    }

    /**
     * 更新状态
     * @param dingRobotId 机器人id
     * @param status 状态，1-进行中，2-已完成
     * @return
     */
    @RequiresPermissions("qa:process:finish")
    @PostMapping("/finish")
    public JsonResult finishPipeline(@RequestParam Integer dingRobotId, @RequestParam Integer status) {
        try {
            DingRobot dingRobot = dingRobotService.getById(dingRobotId);
            if (dingRobot == null) {
                return JsonResult.buildErrorStateResult("该测试流程不存在！", false);
            }
            // 设置状态为已完成
            dingRobot.setStatus(status);
            if (!dingRobotService.updateById(dingRobot)) {
                return JsonResult.buildErrorStateResult("更新状态失败！", false);
            }
            return JsonResult.buildSuccessResult("更新状态成功！", true);
        } catch (Exception e) {
            return JsonResult.buildErrorStateResult("更新状态失败！", false);
        }
    }

}
