package cn.qg.holmes.service.effect.impl;

import cn.qg.holmes.service.effect.DatabaseSyncService;
import cn.qg.holmes.utils.HttpClientUtils;
import cn.qg.holmes.utils.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.*;
import java.util.*;

@Slf4j
@Service
public class DatabaseSyncServiceImpl implements DatabaseSyncService {

    @Autowired
    RedisUtils redisUtils;

    @Value("${tke.host}")
    private String tkeHost;

    @Value("${cash_loan_flow.sql}")
    private String cashLoanFlowSql;

    @Value("${contract.sql}")
    private String contractSql;

    @Value("${payment_center.sql}")
    private String paymentCenterSql;

    private final String dbSyncPrefix = "dbsync:";

    /**
     * 从中间库获取数据库相关信息
     * @param ip 中间库 ip
     * @param port 中间库 port
     * @param username  中间库用户名
     * @param password 中间库密码
     * @param dbName 数据库名
     * @return
     */
    @Override
    public boolean getDbInfoFromSource(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;
        String dbRedisKey = dbSyncPrefix + dbName;
        String tableListKey = dbSyncPrefix + dbName + ":tables";
        Connection connection = null;
        PreparedStatement preparedStatement = null;

        // 创建数据库的redis值
        String dbRedisValue = "";
        // 表数据redis值
        String insertRedisValue = "";
        // 建表语句redis值
        String createTableRedisValue = "";
        // 表名列表redis值
        String tableListStr = "";

        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);

            if (!redisUtils.hasKey(dbRedisKey)) {
                String dbSql = String.format("SHOW CREATE DATABASE %s", dbName);
                preparedStatement = connection.prepareStatement(dbSql);
                ResultSet dbResultSet = preparedStatement.executeQuery();
                while (dbResultSet.next()) {
                    dbRedisValue = dbResultSet.getString(2);
                }
                // 设置建库语句的redis值
                redisUtils.set(dbRedisKey, dbRedisValue, 86400);
            }


            DatabaseMetaData databaseMetaData = connection.getMetaData();
            // 获取所有表名
            ResultSet tableResultSet = databaseMetaData.getTables(null, null, null, new String[]{"TABLE"});
            while (tableResultSet.next()) {
                String tableName = tableResultSet.getString("TABLE_NAME");
                tableListStr += tableName;
                tableListStr += "\n";

                long startTime = System.currentTimeMillis();
                log.info("开始获取表{}的数据", tableName);
                String createTableKey = dbSyncPrefix + dbName + ":" + tableName + ":create" ;
                String insertTableKey = dbSyncPrefix + dbName + ":" + tableName + ":insert" ;

                if (!redisUtils.hasKey(createTableKey)) {
                    // 获取所有建表语句
                    String sql = String.format("SHOW CREATE TABLE %s", tableName);
                    preparedStatement = connection.prepareStatement(sql);
                    ResultSet pResultSet = preparedStatement.executeQuery();
                    while (pResultSet.next()) {
                        createTableRedisValue = pResultSet.getString(2);
                    }
                    redisUtils.set(createTableKey, createTableRedisValue, 86400);
                    createTableRedisValue = "";
                }

                if (!redisUtils.hasKey(insertTableKey)) {
                    ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                    while (columnResultSet.next()) {
                        // 字段名称
                        String columnName = columnResultSet.getString("COLUMN_NAME");
                        // 数据类型
                        String columnType = columnResultSet.getString("TYPE_NAME");

                        columnNameList.add(columnName);
                        columnTypeList.add(columnType);
                    }
                    String columnArrayStr = null;
                    for (String column : columnNameList) {
                        if (null == columnArrayStr) {
                            columnArrayStr = "`" + column + "`";
                        } else {
                            columnArrayStr = columnArrayStr + "," + "`" + column + "`";
                        }
                    }

                    String selectSQL = String.format("select %s from %s", columnArrayStr, tableName);
                    preparedStatement = connection.prepareStatement(selectSQL);
                    ResultSet selectResultSet = preparedStatement.executeQuery();
                    while (selectResultSet.next()) {
                        String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
                        String insertSql = String.format("insert into %s (%s) values(%s);", tableName, columnArrayStr, rowValues);
                        insertSql = insertSql.replaceAll("\n", "<br/>");
                        insertSql = insertSql + "\n";
                        insertRedisValue += insertSql;
                    }
                    columnNameList.clear();
                    columnTypeList.clear();
                    redisUtils.set(insertTableKey, insertRedisValue, 86400);
                    long endTime = System.currentTimeMillis();
                    log.info("{}表数据获取完成，共花费{}秒", tableName, (endTime - startTime) / 1000);
                    insertRedisValue = "";
                }
            }
            redisUtils.set(tableListKey, tableListStr, 86400);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 同步数据库信息到目标地址
     * @param ip 目标库ip
     * @param port 目标库端口
     * @param username 目标库用户名
     * @param password 目标库密码
     * @param dbName 目标库名称
     * @return
     */
    @Override
    public boolean syncDbToDest(String ip, String port, String username, String password, String dbName, String namespace) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port;

        String dbRedisKey = dbSyncPrefix + dbName;
        String tableListKey = dbSyncPrefix + dbName + ":tables";

        Connection connection = null;
        Connection newConnection = null;
        PreparedStatement preparedStatement = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);

            // 删除原有数据库
            String dropDbSql = String.format("DROP DATABASE IF EXISTS %s", dbName);
            log.info("删除原有数据库：{}", dropDbSql);
            preparedStatement = connection.prepareStatement(dropDbSql);
            preparedStatement.execute();

            // 重新创建数据库
            String createDbSql = redisUtils.get(dbRedisKey).toString();
            log.info("重新创建数据库：{}", createDbSql);
            preparedStatement = connection.prepareStatement(createDbSql);
            preparedStatement.execute();
            connection.close();

            // 重新设置连接，并指定数据库
            newConnection = DriverManager.getConnection(url + "/" + dbName, username, password);
            newConnection.setAutoCommit(false);

            List<String> tableList = Arrays.asList(redisUtils.get(tableListKey).toString().split("\n"));
            Statement statement = newConnection.createStatement();
            // 循环处理每个表
            for (String tableName: tableList) {
                log.info("开始同步表：{}", tableName);
                long dataStartTime = System.currentTimeMillis();
                String createTableKey = dbSyncPrefix + dbName + ":" + tableName + ":create" ;
                String insertTableKey = dbSyncPrefix + dbName + ":" + tableName + ":insert" ;
                String createTableValue = redisUtils.get(createTableKey).toString();
                String insertTableValue = redisUtils.get(insertTableKey).toString();

                // 不为空时才执行建表语句
                if (!createTableValue.isEmpty()) {
                    statement.execute(createTableValue);
                }

                // 不为空时才插入数据
                if (!insertTableValue.isEmpty()) {
                    for (String insertSql: insertTableValue.split("\n")) {
                        statement.addBatch(insertSql);
                    }
                    statement.executeBatch();
                    statement.clearBatch();
                }
                long dataEndTime = System.currentTimeMillis();
                log.info("{}表同步完成，共花费{}秒", tableName, (dataEndTime - dataStartTime) / 1000);
            }

            // 判断是否需要update
            if (dbName.equals("cash_loan_flow")) {
                for (String sql: cashLoanFlowSql.split("\n")) {
                    log.info("执行update sql: {}", sql);
                    statement.addBatch(sql);
                }
                statement.executeBatch();
                statement.clearBatch();
            }
            if (dbName.equals("contract")) {
                for (String sql: contractSql.split("\n")) {
                    log.info("执行update sql: {}", sql);
                    statement.addBatch(sql);
                }
                statement.executeBatch();
                statement.clearBatch();
            }
            if (dbName.equals("payment_center")) {
                for (String sql: paymentCenterSql.split("\n")) {
                    log.info("执行update sql: {}", sql);
                    statement.addBatch(sql);
                }
                statement.executeBatch();
                statement.clearBatch();
            }
            newConnection.commit();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (newConnection != null) {
                    newConnection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public boolean getSingleTableFromSource(String ip, String port, String username, String password, String dbName, String tableName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        String tableCreateKey = dbSyncPrefix + dbName + ":" + tableName + ":create";
        String tableInsertKey = dbSyncPrefix + dbName + ":" + tableName + ":insert";
        String insertRedisValue = "";

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        List<String> columnNameList = new ArrayList<>();
        List<String> columnTypeList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(true);
            DatabaseMetaData databaseMetaData = connection.getMetaData();

            if (!redisUtils.hasKey(tableCreateKey)) {
                String sql = String.format("SHOW CREATE TABLE %s", tableName);
                preparedStatement = connection.prepareStatement(sql);
                ResultSet resultSet = preparedStatement.executeQuery();
                while (resultSet.next()) {
                    redisUtils.set(tableCreateKey, resultSet.getString(2), 600);
                }
            }

            if (!redisUtils.hasKey(tableInsertKey)) {
                ResultSet columnResultSet = databaseMetaData.getColumns(null, "%", tableName, "%");
                while (columnResultSet.next()) {
                    // 字段名称
                    String columnName = columnResultSet.getString("COLUMN_NAME");
                    // 数据类型
                    String columnType = columnResultSet.getString("TYPE_NAME");

                    columnNameList.add(columnName);
                    columnTypeList.add(columnType);;
                }
                String columnArrayStr = null;
                for (String column : columnNameList) {
                    if (null == columnArrayStr) {
                        columnArrayStr = "`" + column + "`";
                    } else {
                        columnArrayStr = columnArrayStr + "," + "`" + column + "`";
                    }
                }

                String selectSQL = String.format("select %s from %s", columnArrayStr, tableName);
                preparedStatement = connection.prepareStatement(selectSQL);
                ResultSet selectResultSet = preparedStatement.executeQuery();
                while (selectResultSet.next()) {
                    String rowValues = getRowValues(selectResultSet, columnNameList.size(), columnTypeList);
                    String insertSql = String.format("insert into %s (%s) values(%s);", tableName, columnArrayStr, rowValues);
                    insertSql = insertSql.replaceAll("\n", "<br/>");
                    insertSql = insertSql + "\n";
                    insertRedisValue += insertSql;
                }
                redisUtils.set(tableInsertKey, insertRedisValue, 600);
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    @Override
    public boolean syncSingleTableToDest(String ip, String port, String username, String password, String dbName, String tableName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        String tableCreateKey = dbSyncPrefix + dbName + ":" + tableName + ":create";
        String tableInsertKey = dbSyncPrefix + dbName + ":" + tableName + ":insert";

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        Statement statement = null;

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(false);

            String createTableSql = redisUtils.get(tableCreateKey).toString();
            String insertTableSql = redisUtils.get(tableInsertKey).toString();

            // 如果表存在，则首先删除表
            String dropTableSql = String.format("DROP TABLE if exists %s", tableName);
            preparedStatement = connection.prepareStatement(dropTableSql);
            preparedStatement.execute();

            // 重新创建表
            preparedStatement = connection.prepareStatement(createTableSql);
            preparedStatement.execute();

            statement = connection.createStatement();
            for (String insertSql: insertTableSql.split("\n")) {
                statement.addBatch(insertSql);
            }
            statement.executeBatch();
            statement.clearBatch();
            connection.commit();
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
                if (statement != null) {
                    statement.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return true;
    }

    /**
     * 获取数据库列表
     * @param ip 同步库ip
     * @param port 同步库端口号
     * @param username 同步库用户名
     * @param password 同步库密码
     * @return
     */
    @Override
    public List<Object> getDatabaseList(String ip, String port, String username, String password) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port;

        List<Object> databaseList = new ArrayList<>();
        String dbListRedisKey = dbSyncPrefix + ":db:list";

        if (redisUtils.hasKey(dbListRedisKey)) {
            databaseList = redisUtils.lGet(dbListRedisKey, 0, redisUtils.lGetListSize(dbListRedisKey));
            log.info("从redis获取到的数据库列表为：{}", databaseList);
        } else {
            Connection connection = null;

            try {
                Class.forName(driver);
                connection = DriverManager.getConnection(url, username, password);
                connection.setAutoCommit(true);

                String showDatabasesSql = "SHOW DATABASES";
                PreparedStatement preparedStatement = connection.prepareStatement(showDatabasesSql);
                ResultSet dbListResultSet = preparedStatement.executeQuery();
                while (dbListResultSet.next()) {
                    databaseList.add(dbListResultSet.getString(1));
                }
                preparedStatement.close();
                // 设置数据库列表缓存，缓存24个小时
                log.info("从同步库获取到的数据库列表为：{}", databaseList);
                redisUtils.lSet(dbListRedisKey, databaseList, 86400);
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                try {
                    if (connection != null) {
                        connection.close();
                    }
                } catch (Exception e) {
                    connection = null;
                }
            }
        }
        return databaseList;
    }

    /**
     * 根据namespace获取对应的mysql ip和端口
     * @param namespace
     * @return
     */
    @Override
    public Map<String, String> getMysqlInfoByNamespace(String namespace) {
        Map<String, String> headers = new HashMap<>();
        Map<String, String> params = new HashMap<>();
        headers.put("cluster", "qa");
        params.put("namespace", namespace);
        params.put("serviceName", "mysql");
        params.put("type", "base");
        String response = HttpClientUtils.doPostJson(tkeHost + "/service/details", headers, JSON.toJSONString(params));
        String mysqlIp = JsonPath.read(response, "$.data.lanIp").toString();
        String mysqlPort = JsonPath.read(response, "$.data.portMappings[0].nodePort").toString();
        Map<String, String> result = new HashMap<>();
        result.put("ip", mysqlIp);
        result.put("port", mysqlPort);
        return result;
    }

    /**
     * 获取某个数据库下表列表
     * @param ip mysql ip
     * @param port mysql port
     * @param username mysql 用户名
     * @param password mysql 密码
     * @param dbName 数据库名
     * @return
     */
    @Override
    public List<Object> getTableListByDb(String ip, String port, String username, String password, String dbName) {
        String driver = "com.mysql.jdbc.Driver";
        String url = "jdbc:mysql://" + ip + ":" + port + "/" + dbName;

        Connection connection = null;
        PreparedStatement preparedStatement = null;
        List<Object> tableList = new ArrayList<>();

        try {
            Class.forName(driver);
            connection = DriverManager.getConnection(url, username, password);
            connection.setAutoCommit(false);

            String tablesSql = "SHOW TABLES";
            preparedStatement = connection.prepareStatement(tablesSql);
            ResultSet tableListResultSet = preparedStatement.executeQuery();
            while (tableListResultSet.next()) {
                tableList.add(tableListResultSet.getString(1));
            }
            return tableList;

        } catch (Exception e) {
            e.printStackTrace();
            return null;
        } finally {
            try {
                if (preparedStatement != null) {
                    preparedStatement.close();
                }
                if (connection != null) {
                    connection.close();
                }
            } catch (Exception e) {
                connection = null;
            }
        }
    }

    /**
     * 获取表数据一行的所有值
     * @param rs
     * @param size
     * @param columnTypeList
     * @return
     */
    private String getRowValues(ResultSet rs, int size, List<String> columnTypeList) {
        try {
            String rowValues = null;
            for (int i = 1; i <= size; i++) {
                String columnValue = null;

                // 获取字段值
                columnValue = getValue(rs, i, columnTypeList.get(i - 1));
                // 如果是空值不添加单引号
                if (null != columnValue) {
//                    if (columnValue.contains("'")) {
//                        columnValue = "\"" + columnValue + "\"";
//                    } else {
//                        columnValue = "'" + columnValue + "'";
//                    }
                    columnValue = columnValue.replace("'", "\\'");
                    columnValue = columnValue.replace("\"", "\\\"");
                    columnValue = "'" + columnValue + "'";
                }
                // 拼接字段值
                if (null == rowValues) {
                    rowValues = columnValue;
                } else {
                    rowValues = rowValues + "," + columnValue;
                }
            }

            return rowValues;
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("获取表数据一行的所有值异常");
            return null;
        }
    }

    /**
     * 根据类型获取字段值
     * @return
     * @author  2018年9月6日 上午11:16:00
     */
    private String getValue(ResultSet resultSet, Integer index, String columnType) {
        try {
            if ("int".equals(columnType) || "INT".equals(columnType)) {
                // 整数
                Object intValue = resultSet.getObject(index);
                if (null == intValue) {
                    return null;
                }
                return intValue + "";
            } else if ("bigint".equals(columnType) || "BIGINT".equals(columnType)) {
                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ( "bigint unsigned".equals(columnType) || "BIGINT UNSIGNED".equals(columnType)) {
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("smallint".equals(columnType) || "SMALLINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("tinyint".equals(columnType) || "TINYINT".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("mediumint".equals(columnType) || "MEDIUMINT".equals(columnType)) {
                // 长整形
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("integer".equals(columnType) || "INTEGER".equals(columnType)) {
                // 整数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("float".equals(columnType) || "FLOAT".equals(columnType)) {

                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("double".equals(columnType) || "DOUBLE".equals(columnType)) {
                // 浮点数
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            } else if ("decimal".equals(columnType) || "DECIMAL".equals(columnType)) {
                // 浮点数-金额类型
                BigDecimal value = resultSet.getBigDecimal(index);
                if (null == value) {
                    return null;
                }
                return value.toString();
            } else if ("char".equals(columnType) || "CHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("varchar".equals(columnType) || "VARCHAR".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("tinytext".equals(columnType) || "TINYTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("text".equals(columnType) || "TEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("mediumtext".equals(columnType) || "MEDIUMTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("longtext".equals(columnType) || "LONGTEXT".equals(columnType)) {
                // 字符串类型
                String value = resultSet.getString(index);
                return value;
            } else if ("year".equals(columnType) || "YEAR".equals(columnType)) {
                // 时间类型：范围 1901/2155 格式 YYYY
                String year = resultSet.getString(index);
                if (null == year) {
                    return null;
                }
                // 只需要年的字符即可，
                return year.substring(0, 4);
            } else if ("date".equals(columnType) || "DATE".equals(columnType)) {
                // 时间类型：范围 '1000-01-01'--'9999-12-31' 格式 YYYY-MM-DD
                return resultSet.getString(index);
            } else if ("time".equals(columnType) || "TIME".equals(columnType)) {
                // 时间类型：范围 '-838:59:59'到'838:59:59' 格式 HH:MM:SS
                return resultSet.getString(index);
            } else if ("datetime".equals(columnType) || "DATETIME".equals(columnType)) {
                // 时间类型：范围 '1000-01-01 00:00:00'--'9999-12-31 23:59:59' 格式 YYYY-MM-DD HH:MM:SS
                return resultSet.getString(index);
            } else if ("timestamp".equals(columnType) || "TIMESTAMP".equals(columnType)) {
                // 时间类型：范围 1970-01-01 00:00:00/2037 年某时 格式 YYYYMMDD HHMMSS 混合日期和时间值，时间戳
                return resultSet.getString(index);
            } else if ("bit".equals(columnType) || "BIT".equals(columnType)) {
                String value = resultSet.getString(index);
                if (value.equals("false")) {
                    return "0";
                } else if (value.equals("true")) {
                    return "1";
                } else {
                    return value;
                }
            } else {
                Object value = resultSet.getObject(index);
                if (null == value) {
                    return null;
                }
                return value + "";
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.err.println("获取数据库类型值异常");
            return "0000-00-00 00:00:00";
        }
    }

    private String replaceDomain(String sql, String namespace) {
        if (sql.contains("xyqb.com")) {
            return sql.replace(".xyqb.com", "-" + namespace + ".liangkebang.net");
        } else if (sql.contains(".q-gp.com")) {
            return sql.replace(".q-gp.com", "-" + namespace + ".liangkebang.net");
        } else {
            return sql;
        }
    }
}
