package cn.qg.holmes.utils;

import io.fabric8.kubernetes.api.model.*;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Component
public class K8sService {

    private KubernetesClient kubernetesClient;

    public K8sService() {
        try {
            String configYAML = String.join("\n", readConfigFile("kube-config.yml"));
            Config config = Config.fromKubeconfig(configYAML);
            kubernetesClient = new DefaultKubernetesClient(config);
            String configCrt = String.join("\n", readConfigFile("tke-cluster-ca.crt"));
            config.setCaCertData(configCrt);
        } catch (Exception e) {
            log.info("k8s service 初始化失败！");
            e.printStackTrace();
        }

    }

    /**
     * 读取k8s配置文件
     * @param file 配置文件名
     * @return
     * @throws IOException
     */
    public static List<String> readConfigFile(String file) throws IOException {
        String str = "";
        ClassPathResource resource = new ClassPathResource(file);
        InputStream in = resource.getInputStream();
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        List<String> result = new ArrayList<>();
        while ((str = br.readLine()) != null) {
            result.add(str);
        }
        return result;
    }

    /**
     * 获取namespace列表
     * @return
     */
    public List<Map<String, Object>> getNamespaceList() {
        List<Map<String, Object>> resultList = new ArrayList<>();
        List<Namespace> namespaceList = kubernetesClient.namespaces().list().getItems();
        for (Namespace namespace: namespaceList) {
            log.info(namespace.toString());
            if (namespace.getMetadata().getAnnotations() != null) {
                Map<String, Object> map = new HashMap<>();
                map.put("name", namespace.getMetadata().getName());
                map.put("description", namespace.getMetadata().getAnnotations().get("description"));
                map.put("owner", namespace.getMetadata().getAnnotations().get("owner"));
                map.put("status", namespace.getStatus().getPhase());
                map.put("createdAt",namespace.getMetadata().getCreationTimestamp());
                resultList.add(map);
            }
        }
        return  resultList;
    }

    public List<Map<String, Object>> getServiceList() {
        List<Map<String, Object>> serviceList = new ArrayList<>();
        kubernetesClient.services().inNamespace("fe").list();
        return null;
    }

    public List<Map<String, Object>> getPodList(String namespace) {
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).list().getItems();
        List<Map<String, Object>> result = new ArrayList<>();
        for (Pod pod: podList) {
            if (pod.getStatus().getPhase().equals("Running")) {
                Map<String, Object> map = new HashMap<>();
                // 端口映射
                List<Map<String, Object>> portMappingList = new ArrayList<>();
                ObjectMeta podMetadata= pod.getMetadata();
                String serviceName = podMetadata.getLabels().get("qcloud-app");
                Service service = kubernetesClient.services().inNamespace(namespace).withName(serviceName).get();
                List<ServicePort> servicePortList = service.getSpec().getPorts();
                if (servicePortList.size() > 0) {
                    for (ServicePort servicePort: servicePortList) {
                        if (servicePort.getNodePort() != null) {
                            map.put("port_" + servicePort.getName(), servicePort.getNodePort());
                        }
                        Map<String, Object> portMap = new HashMap<>();
                        portMap.put("name", servicePort.getName());
                        portMap.put("nodePort", servicePort.getNodePort());
                        portMap.put("port", servicePort.getPort());
                        portMap.put("protocol", servicePort.getProtocol());
                        portMap.put("targetPort", servicePort.getTargetPort());
                        portMappingList.add(portMap);
                    }
                }
                if (portMappingList.size() > 0) {
                    map.put("portMappings", portMappingList);
                }
                map.put("createdAt", podMetadata.getCreationTimestamp());
                map.put("serviceName", serviceName);
                map.put("podName", podMetadata.getName());
                map.put("labels", podMetadata.getLabels());
                List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
                map.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
                map.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
                map.put("lanIp", pod.getStatus().getHostIP());
                map.put("podIp", pod.getStatus().getPodIP());
                map.put("startTime", pod.getStatus().getStartTime());
                map.put("status", containerStatuses.get(0).getReady());
                result.add(map);
            }
        }
        return result;
    }

    public static void main(String[] args) {
        K8sService k8sService = new K8sService();
//        List<Service> serviceList = k8sService.kubernetesClient.services().inNamespace("fe").list().getItems();
//        List<Pod> podList = k8sService.kubernetesClient.pods().inNamespace("fe").list().getItems();
//        Pod pod =  podList.get(22);

//        System.out.println("Metadata: " + pod.getMetadata());
//        System.out.println("Spec: " + pod.getSpec());
//        System.out.println("Status: " + pod.getStatus());
//        System.out.println("Api: " + pod.getApiVersion());
//        System.out.println(pod.getStatus().getContainerStatuses().get(0).getImage());
//        System.out.println(pod.getStatus().getContainerStatuses().get(0).getImageID());
//        System.out.println(pod.getStatus().getPhase());
//        System.out.println(pod.getSpec().getContainers().get(0).getName());

//        System.out.println(serviceList.get(19).getSpec().getPorts());
        Service service = k8sService.kubernetesClient.services().inNamespace("fe").withName("holmes").get();
        System.out.println(service);

    }

}
