package cn.qg.holmes.service.gitlab.impl;

import cn.qg.holmes.entity.gitlab.Pipeline;
import cn.qg.holmes.mapper.gitlab.PipelineMapper;
import cn.qg.holmes.service.gitlab.PipelineService;
import cn.qg.holmes.service.jira.DingRobotService;
import cn.qg.holmes.utils.DingdingUtils;
import cn.qg.holmes.utils.JenkinsService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;

@Slf4j
@Service
public class PipelineServiceImpl extends ServiceImpl<PipelineMapper, Pipeline> implements PipelineService {

    @Autowired
    PipelineMapper pipelineMapper;

    @Autowired
    JenkinsService jenkinsService;

    @Autowired
    DingRobotService dingRobotService;

    /**
     * 构建pipeline
     * @param gitlabData gitlab发送过来的推送事件
     */
    @Override
    public void buildPipelineJob(String gitlabData) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String projectName = JsonPath.read(gitlabData, "$.project.name");
        String branch = JsonPath.read(gitlabData, "$.ref").toString().split("/")[2];
        Integer totalCommitCount = JsonPath.read(gitlabData, "$.total_commits_count");
        if (totalCommitCount.equals(0)) {
            return;
        }
        String message = JsonPath.read(gitlabData, "$.commits[-1].message");
        String author = JsonPath.read(gitlabData, "$.user_username");
        // push event给过来的日期为带T的，需要进行转换
        String commitDate = simpleDateFormat.format(DateTime.parse(JsonPath.read(gitlabData, "$.commits[-1].timestamp")).toDate());
        QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
        pipelineQueryWrapper.eq("service_name", projectName);
        pipelineQueryWrapper.eq("service_branch", branch);
        Pipeline pipeline = pipelineMapper.selectOne(pipelineQueryWrapper);
        if (pipeline != null) {
            if (pipeline.getEnable() == 1) {
                log.info("pipeline中找到对应项目，且项目是启动状态，开始构建：{}", pipeline.toString());
                Integer buildNum = jenkinsService.buildJenkinsJob(pipeline.getServiceType(), projectName, branch, pipeline.getNamespace());
                String markdown = DingdingUtils.buildPipelineMarkdownMsg(projectName, branch, pipeline.getServiceType(), author, message, commitDate, pipeline.getNamespace(), buildNum.toString());
                DingdingUtils.sendToDingding(markdown, dingRobotService.getById(pipeline.getDingRobotId()).getDingUrl());
            } else {
                log.info("pipeline中找到项目，但是项目状态为不启用，因此不构建！");
            }
        } else {
            log.info("pipeline中未找到该项目相关信息，不触发自动构建！");
        }
    }
}
