package cn.qg.holmes.service.k8s;

import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import io.fabric8.kubernetes.api.model.*;
import io.fabric8.kubernetes.api.model.apps.DeploymentStrategy;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.apps.DeploymentSpec;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.DefaultKubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Component
public class TkeService {

    private KubernetesClient kubernetesClient;

    public TkeService() {
        try {
            String configYAML = String.join("\n", readConfigFile("kube-config.yml"));
            Config config = Config.fromKubeconfig(configYAML);
            kubernetesClient = new DefaultKubernetesClient(config);
            String configCrt = String.join("\n", readConfigFile("tke-cluster-ca.crt"));
            config.setCaCertData(configCrt);
        } catch (Exception e) {
            log.info("k8s service 初始化失败！");
            e.printStackTrace();
        }

    }

    /**
     * 读取k8s配置文件
     * @param file 配置文件名
     * @return
     * @throws IOException
     */
    private static List<String> readConfigFile(String file) throws IOException {
        String str = "";
        ClassPathResource resource = new ClassPathResource(file);
        InputStream in = resource.getInputStream();
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        List<String> result = new ArrayList<>();
        while ((str = br.readLine()) != null) {
            result.add(str);
        }
        return result;
    }

    /**
     * 获取namespace列表
     * @return
     */
    public List<Map<String, Object>> getNamespaceList() {
        List<Map<String, Object>> resultList = new ArrayList<>();
        List<Namespace> namespaceList = kubernetesClient.namespaces().list().getItems();
        for (Namespace namespace: namespaceList) {
            log.info(namespace.toString());
            if (namespace.getMetadata().getAnnotations() != null) {
                Map<String, Object> map = new HashMap<>();
                map.put("name", namespace.getMetadata().getName());
                map.put("description", namespace.getMetadata().getAnnotations().get("description"));
                map.put("owner", namespace.getMetadata().getAnnotations().get("owner"));
                map.put("status", namespace.getStatus().getPhase());
                map.put("createdAt",namespace.getMetadata().getCreationTimestamp());
                resultList.add(map);
            }
        }
        return  resultList;
    }

    /**
     * 获取运行中的pod列表
     * @param namespace 环境
     * @return
     */
    public List<Map<String, Object>> getPodList(String namespace) {
        List<Pod> podList = kubernetesClient.pods().inNamespace(namespace).list().getItems();
        List<Map<String, Object>> result = new ArrayList<>();
        for (Pod pod: podList) {
            if (pod.getStatus().getPhase().equals("Running")) {
                Map<String, Object> map = new HashMap<>();
                // 端口映射
                List<Map<String, Object>> portMappingList = new ArrayList<>();
                ObjectMeta podMetadata= pod.getMetadata();
                String serviceName = podMetadata.getLabels().get("qcloud-app");
                Service service = kubernetesClient.services().inNamespace(namespace).withName(serviceName).get();
                List<ServicePort> servicePortList = service.getSpec().getPorts();
                if (servicePortList.size() > 0) {
                    for (ServicePort servicePort: servicePortList) {
                        if (servicePort.getNodePort() != null) {
                            map.put("port_" + servicePort.getName(), servicePort.getNodePort());
                        }
                        Map<String, Object> portMap = new HashMap<>();
                        portMap.put("name", servicePort.getName());
                        portMap.put("nodePort", servicePort.getNodePort());
                        portMap.put("port", servicePort.getPort());
                        portMap.put("protocol", servicePort.getProtocol());
                        portMap.put("targetPort", servicePort.getTargetPort());
                        portMappingList.add(portMap);
                    }
                }
                if (portMappingList.size() > 0) {
                    map.put("portMappings", portMappingList);
                }
                map.put("createdAt", podMetadata.getCreationTimestamp());
                map.put("serviceName", serviceName);
                map.put("podName", podMetadata.getName());
                map.put("labels", podMetadata.getLabels());
                List<ContainerStatus> containerStatuses = pod.getStatus().getContainerStatuses();
                map.put("image", pod.getStatus().getContainerStatuses().get(0).getImage());
                map.put("imageId", pod.getStatus().getContainerStatuses().get(0).getImageID());
                map.put("lanIp", pod.getStatus().getHostIP());
                map.put("podIp", pod.getStatus().getPodIP());
                map.put("startTime", pod.getStatus().getStartTime());
                map.put("status", containerStatuses.get(0).getReady());
                result.add(map);
            }
        }
        return result;
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName podName
     * @return
     */
    public boolean resetPod(String namespace, String podName) {
        return kubernetesClient.pods().inNamespace(namespace).withName(podName).delete();
    }

    /**
     * 删除一个pod
     * @param namespace 环境
     * @param serviceName 服务名
     * @return
     */
    public boolean deleteService(String namespace, String serviceName) {
        // 删除service
        kubernetesClient.services().inNamespace(namespace).withName(serviceName).delete();
        Map<String, String > map = new HashMap<>();
        map.put("qcloud-app", serviceName);
        LabelSelector labelSelector = new LabelSelector();
        labelSelector.setMatchLabels(map);
        // 删除deployment
        boolean deploymentResult = kubernetesClient.apps().deployments().inNamespace(namespace).withName(serviceName).delete();
        // 删除replicationControllers
        boolean rpcResult = kubernetesClient.replicationControllers().inNamespace(namespace).withLabelSelector(labelSelector).delete();
        // 删除pvc
        boolean pvcResult = kubernetesClient.persistentVolumeClaims().inNamespace(namespace).withName(serviceName).delete();
        // 删除service
        boolean serviceResult = kubernetesClient.services().inNamespace(namespace).withLabelSelector(labelSelector).delete();
        // 删除ingress
        boolean ingressResult = kubernetesClient.extensions().ingresses().inNamespace(namespace).withName(serviceName).delete();
        return deploymentResult && rpcResult && pvcResult && serviceResult && ingressResult;
    }

    /**
     * 部署Java服务
     * @return
     */
    public Service createJavaService(String namespace, String serviceName, String serviceType, String label) {
        Service service = new Service();
        ObjectMeta objectMeta = new ObjectMeta();
        ServiceSpec serviceSpec = new ServiceSpec();
        // 设置meta
        Map<String, String> labels = new HashMap<>();
        labels.put("type", label);
        labels.put("qcloud-app", serviceName);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labels);
        // 设置spec
        List<ServicePort> portList = new ArrayList<>();
        ServicePort servicePort1 = new ServicePort();
        servicePort1.setName("80");
        servicePort1.setPort(80);
        ServicePort servicePort2 = new ServicePort();
        servicePort1.setName("5005");
        servicePort1.setPort(5005);
        ServicePort servicePort3 = new ServicePort();
        servicePort1.setName("20880");
        servicePort1.setPort(20880);
        portList.add(servicePort1);
        portList.add(servicePort2);
        portList.add(servicePort3);
        Map<String, String> selector = new HashMap<>();
        selector.put("qcloud-app", serviceName);

        // 设置serviceSpec
        serviceSpec.setType(serviceType);
        serviceSpec.setPorts(portList);
        serviceSpec.setSelector(selector);

        // 设置service
        service.setApiVersion("v1");
        service.setKind("Service");
        service.setMetadata(objectMeta);
        service.setSpec(serviceSpec);
        return kubernetesClient.services().inNamespace(namespace).create(service);
    }

    /**
     * 部署Java Deployment
     * @param serviceCreateVo
     * @return
     */
    public Deployment createJavaDeployment(ServiceCreateVo serviceCreateVo, DockerProject dockerProject) {
        String namespace = serviceCreateVo.getNamespace();
        String serviceName = serviceCreateVo.getServiceName();
        String mock = serviceCreateVo.getMock().toString();
        Deployment deployment = new Deployment();
        ObjectMeta objectMeta = new ObjectMeta();
        DeploymentSpec deploymentSpec = new DeploymentSpec();

        // 设置metadata
        Map<String, String> labels = new HashMap<>();
        labels.put("type", serviceCreateVo.getLabel());
        labels.put("qcloud-app", serviceName);
        objectMeta.setName(serviceName);
        objectMeta.setNamespace(namespace);
        objectMeta.setLabels(labels);

        // 设置labelSelector
        LabelSelector labelSelector = new LabelSelector();
        Map<String, String> matchLabels = new HashMap<>();
        matchLabels.put("qcloud-app", serviceName);
        labelSelector.setMatchLabels(matchLabels);

        // 设置strategy
        DeploymentStrategy deploymentStrategy = new DeploymentStrategy();
        deploymentStrategy.setType("Recreate");

        // 设置pod Template
        PodTemplateSpec podTemplateSpec = new PodTemplateSpec();
        ObjectMeta templateObjectMeta = new ObjectMeta();
        Map<String, String> templateLabels = new HashMap<>();
        templateLabels.put("qcloud-app", serviceName);
        templateLabels.put("type", serviceCreateVo.getLabel());
        templateLabels.put("mock", mock);
        templateObjectMeta.setLabels(templateLabels);

        PodSpec podSpec = new PodSpec();
        List<Container> containerList = new ArrayList<>();
        Container container = new Container();
        container.setName(serviceName);
        container.setImage("ccr.ccs.tencentyun.com/" + serviceCreateVo.getImage());
        container.setImagePullPolicy("IfNotPresent");

        // volumeMounts
        List<VolumeMount> volumeMountList = new ArrayList<>();
        VolumeMount volumeMount = new VolumeMount();
        volumeMount.setName("filelog");
        volumeMount.setMountPath("/home/logs");
        volumeMountList.add(volumeMount);
        container.setVolumeMounts(volumeMountList);

        // env
        List<EnvVar> envVarList = new ArrayList<>();
        EnvVar envVar1 = new EnvVar();
        envVar1.setName("NAMESPACE");
        envVar1.setValue(namespace);
        EnvVar envVar2 = new EnvVar();
        envVar2.setName("NAMESPACECLUSTER");
        envVar2.setValue(namespace + "@" + serviceCreateVo.getCluster());
        EnvVar envVar3 = new EnvVar();
        envVar3.setName("CLUSTER");
        envVar3.setValue(serviceCreateVo.getCluster());
        EnvVar envVar4 = new EnvVar();
        envVar4.setName("CLUSTER");
        envVar4.setValue(serviceName);
        EnvVar envVar5 = new EnvVar();
        envVar5.setName("HOSTS");
        envVar5.setValue("192.168.4.50=>eos.quantgroups.com&172.17.1.7=>apollo-dev.quantgroups.com&172.20.3.11=>sentry.quantgroups.com&");
        EnvVar envVar6 = new EnvVar();
        envVar6.setName("DEBUG");
        envVar6.setValue(serviceCreateVo.getDebug().toString());
        EnvVar envVar7 = new EnvVar();
        envVar7.setName("MOCK");
        envVar7.setValue(mock);
        EnvVar envVar8 = new EnvVar();
        EnvVarSource envVarSource = new EnvVarSource();
        ObjectFieldSelector objectFieldSelector = new ObjectFieldSelector();
        objectFieldSelector.setFieldPath("status.hostIP");
        envVarSource.setFieldRef(objectFieldSelector);
        envVar8.setName("DUBBO_IP_TO_REGISTRY");
        envVar8.setValueFrom(envVarSource);
        envVarList.add(envVar1);
        envVarList.add(envVar2);
        envVarList.add(envVar3);
        envVarList.add(envVar4);
        envVarList.add(envVar5);
        envVarList.add(envVar6);
        envVarList.add(envVar7);
        envVarList.add(envVar8);
        container.setEnv(envVarList);

        // resources
        ResourceRequirements resourceRequirements = new ResourceRequirements();
        Map<String, Quantity> requests = new HashMap<>();
        Map<String, Quantity> limits = new HashMap<>();
        Quantity cpuQuantity = new Quantity();
        Quantity memoryQuantity = new Quantity();
        Quantity cpuLimit = new Quantity();
        Quantity memoryLimit = new Quantity();
        cpuQuantity.setAmount(dockerProject.getCpuRequest());
        cpuQuantity.setFormat("m");
        memoryQuantity.setAmount(dockerProject.getMenRequest());
        memoryQuantity.setFormat("Mi");
        cpuLimit.setAmount(dockerProject.getCpuLimit());
        cpuLimit.setFormat("m");
        memoryLimit.setAmount(dockerProject.getMemLimit());
        memoryLimit.setFormat("Mi");
        requests.put("cpu", cpuQuantity);
        requests.put("memory", memoryQuantity);
        resourceRequirements.setRequests(requests);
        resourceRequirements.setLimits(limits);
        container.setResources(resourceRequirements);

        // livenessProbe
        Probe livenessProbe = new Probe();
        ExecAction execAction = new ExecAction();
        List<String> commandList = new ArrayList<>();
        commandList.add("/home/quant_group/readyCheck.sh");
        execAction.setCommand(commandList);
        livenessProbe.setExec(execAction);
        livenessProbe.setInitialDelaySeconds(200);
        livenessProbe.setSuccessThreshold(1);
        livenessProbe.setFailureThreshold(5);
        container.setLivenessProbe(livenessProbe);

        //readinessProbe
        Probe readinessProbe = new Probe();
        readinessProbe.setExec(execAction);
        readinessProbe.setInitialDelaySeconds(30);
        readinessProbe.setTimeoutSeconds(2);
        readinessProbe.setPeriodSeconds(5);
        container.setReadinessProbe(readinessProbe);

        containerList.add(container);

        // volumes
        List<Volume> volumeList = new ArrayList<>();
        Volume volume = new Volume();
        HostPathVolumeSource hostPathVolumeSource = new HostPathVolumeSource();
        hostPathVolumeSource.setPath("/var/log/containers2/" + namespace + "/" + serviceName);
        volume.setName("filelog");
        volume.setHostPath(hostPathVolumeSource);
        volumeList.add(volume);

        // imagePullSecrets
        List<LocalObjectReference> referenceList = new ArrayList<>();
        LocalObjectReference reference1 = new LocalObjectReference();
        LocalObjectReference reference2 = new LocalObjectReference();
        reference1.setName("qcloudregistrykey");
        reference2.setName("tencenthubkey");
        referenceList.add(reference1);
        referenceList.add(reference2);

        // 设置podSpec
        podSpec.setTerminationGracePeriodSeconds(0L);
        podSpec.setContainers(containerList);
        podSpec.setVolumes(volumeList);
        podSpec.setRestartPolicy("Always");
        podSpec.setDnsPolicy("ClusterFirst");
        podSpec.setImagePullSecrets(referenceList);

        // 设置PodTemplateSpec
        podTemplateSpec.setMetadata(templateObjectMeta);
        podTemplateSpec.setSpec(podSpec);

        // 设置Deployment Spec
        deploymentSpec.setReplicas(1);
        deploymentSpec.setRevisionHistoryLimit(1);
        deploymentSpec.setSelector(labelSelector);
        deploymentSpec.setStrategy(deploymentStrategy);
        deploymentSpec.setTemplate(podTemplateSpec);

        // deployment设置
        deployment.setApiVersion("apps/v1beta1");
        deployment.setKind("Deployment");
        deployment.setMetadata(objectMeta);
        deployment.setSpec(deploymentSpec);

        return kubernetesClient.apps().deployments().inNamespace(namespace).create(deployment);
    }

    public static void main(String[] args) {
        TkeService tkeService = new TkeService();
//        List<Service> serviceList = k8sService.kubernetesClient.services().inNamespace("fe").list().getItems();
//        List<Pod> podList = k8sService.kubernetesClient.pods().inNamespace("fe").list().getItems();
//        Pod pod =  podList.get(22);

//        System.out.println("Metadata: " + pod.getMetadata());
//        System.out.println("Spec: " + pod.getSpec());
//        System.out.println("Status: " + pod.getStatus());
//        System.out.println("Api: " + pod.getApiVersion());
//        System.out.println(pod.getStatus().getContainerStatuses().get(0).getImage());
//        System.out.println(pod.getStatus().getContainerStatuses().get(0).getImageID());
//        System.out.println(pod.getStatus().getPhase());
//        System.out.println(pod.getSpec().getContainers().get(0).getName());

//        System.out.println(serviceList.get(19).getSpec().getPorts());
//        Service service = k8sService.kubernetesClient.services().inNamespace("fe").withName("holmes").get();
//        System.out.println(service);

//        System.out.println(k8sService.resetPod("fe", "acm-ui-58864499d9-h47rl"));
        tkeService.deleteService("fe", "kdsp");
    }

}
