package cn.qg.holmes.controller.gitlab;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.config.annotation.RequiresPermissions;
import cn.qg.holmes.entity.auth.UserInfoVo;
import cn.qg.holmes.entity.gitlab.DingRobotPipelineVo;
import cn.qg.holmes.entity.gitlab.Pipeline;
import cn.qg.holmes.entity.quality.DingRobot;
import cn.qg.holmes.entity.quality.SubmitTestInstruction;
import cn.qg.holmes.service.auth.TokenService;
import cn.qg.holmes.service.gitlab.PipelineService;
import cn.qg.holmes.service.quality.DingRobotService;
import cn.qg.holmes.service.quality.JiraService;
import cn.qg.holmes.service.quality.SubmitTestInstructionService;
import cn.qg.holmes.utils.DingdingUtils;
import cn.qg.holmes.utils.HttpClientUtils;
import cn.qg.holmes.utils.JenkinsService;
import com.alibaba.fastjson.JSONArray;
import com.atlassian.jira.rest.client.api.domain.BasicProject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@CrossOrigin
@RequestMapping("/pipeline")
public class PipelineController {

    @Autowired
    PipelineService pipelineService;

    @Autowired
    DingRobotService dingRobotService;

    @Autowired
    HttpServletRequest servletRequest;

    @Autowired
    TokenService tokenService;

    @Autowired
    SubmitTestInstructionService submitTestInstructionService;

    @Autowired
    JiraService jiraService;

    @Autowired
    JenkinsService jenkinsService;

    /**
     * gitlab webhook
     * @param request
     */
    @PostMapping("/webhook")
    public void piplineWebhook(HttpServletRequest request) {
        StringBuilder piplineData = new StringBuilder();
        try {
            BufferedReader br = request.getReader();
            String str = "";
            while((str = br.readLine()) != null){
                piplineData.append(str);
            }
            br.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        log.info("收到gitlab推送来的数据：{}", piplineData);
        pipelineService.buildPipelineJob(String.valueOf(piplineData));
    }

    @GetMapping("/list")
    public JsonResult getDingRobotAndPipelineList(String jiraProjectKey, String projectName, String namespace, Integer status,
                                                  @RequestParam(defaultValue = "1") Integer pageNum,
                                                  @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<DingRobot> page = new Page<>(pageNum, pageSize);
        QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper();
        if (!StringUtils.isEmpty(jiraProjectKey)) {
            dingRobotQueryWrapper.eq("jira_project_key", jiraProjectKey);
        }
        if (!StringUtils.isEmpty(projectName)) {
            dingRobotQueryWrapper.like("project_name", projectName);
        }
        if (!StringUtils.isEmpty(namespace)) {
            dingRobotQueryWrapper.eq("namespace", namespace);
        }
        if (status != null) {
            dingRobotQueryWrapper.eq("status", status);
        }
        dingRobotQueryWrapper.orderByDesc("id");
        IPage<DingRobot> projectRobotIPage = dingRobotService.page(page, dingRobotQueryWrapper);
        List<DingRobot> dingRobotList = projectRobotIPage.getRecords();
        List<DingRobotPipelineVo> dingRobotPipelineVoList = new ArrayList<>();
        // 获取jira项目Map
        Map<String, String> projectMap = new HashMap<>();
        try {
            Iterable<BasicProject> projects = jiraService.getJiraProjectList();
            for (BasicProject project: projects) {
                projectMap.put(project.getKey(), project.getName());
            }
        } catch (Exception e) {
            log.info("获取jira项目Map失败.");
            e.printStackTrace();
        }
        for (DingRobot dingRobot: dingRobotList) {
            // 设置jiraProjectName
            dingRobot.setJiraProjectName(projectMap.get(dingRobot.getJiraProjectKey()));
            DingRobotPipelineVo dingRobotPipelineVo = new DingRobotPipelineVo();
            dingRobotPipelineVo.setDingRobot(dingRobot);
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobot.getId());
            List<Pipeline> pipelineList = pipelineService.list(pipelineQueryWrapper);
            dingRobotPipelineVo.setPipelineList(pipelineList);
            dingRobotPipelineVoList.add(dingRobotPipelineVo);
        }
        Map<String, Object> map = new HashMap<>();
        map.put("total", projectRobotIPage.getTotal());
        map.put("list", dingRobotPipelineVoList);
        return JsonResult.buildSuccessResult(map);
    }

    /**
     * 新增ding robot 以及对应的pipeline
     * @return
     */
    @RequiresPermissions("qa:process:add")
    @PostMapping("/add")
    public JsonResult addRobotAndPipeline(@RequestBody DingRobotPipelineVo dingRobotPipelineVo) {
        try {
            DingRobot dingRobot = dingRobotPipelineVo.getDingRobot();
            // 设置创建人
            String token = servletRequest.getHeader("token");
            UserInfoVo userInfoVo = tokenService.getUserInfoFromCache(token);
            dingRobot.setCreator(userInfoVo.getChineseName());
            String projectName = dingRobot.getProjectName();
            String jiraProjectKey = dingRobot.getJiraProjectKey();
            QueryWrapper<DingRobot> dingRobotQueryWrapper = new QueryWrapper<>();
            dingRobotQueryWrapper.eq("jira_project_key", jiraProjectKey);
            dingRobotQueryWrapper.eq("project_name", projectName);
            // 根据jira project key和项目名判断是否已存在
            if (dingRobotService.getOne(dingRobotQueryWrapper) != null) {
                return JsonResult.buildErrorStateResult("项目已存在", false);
            }
            boolean dingRobotResult = dingRobotService.save(dingRobot);
            if (!dingRobotResult) {
                return JsonResult.buildErrorStateResult("服务器异常，保存失败！", false);
            }
            // 获取该robot id，并赋值给pipelineList
            DingRobot dingRobotNew = dingRobotService.getOne(dingRobotQueryWrapper);
            Integer dingRobotId = dingRobotNew.getId();
            List<Pipeline> pipelineList = dingRobotPipelineVo.getPipelineList();
            // 判断pipelineList是否为空
            if (pipelineList.size() > 0) {
                for (Pipeline pipeline: pipelineList) {
                    pipeline.setDingRobotId(dingRobotId);
                }
                pipelineService.saveBatch(pipelineList);
            }
            String dingUrl = dingRobot.getDingUrl();
            Integer submitTestInstructionId = dingRobot.getSubmitTestInstructionId();
            SubmitTestInstruction submitTestInstruction = submitTestInstructionService.getById(submitTestInstructionId);
            // 提测说明不为空时
            if (submitTestInstruction != null) {
                // 开始循环执行SQL
                String database = submitTestInstruction.getDatabase();
                List<Map> databaseMapList = JSONArray.parseArray(database, Map.class);
                String sqlExecUrl = "https://qa-platform.liangkebang.com/sql/execute";
                Map<String, String> sqlExecParams = new HashMap<>();
                sqlExecParams.put("namespace", dingRobot.getNamespace());
                for (Map map: databaseMapList) {
                    String db = map.get("db").toString();
                    String sql = map.get("sql").toString();
                    sqlExecParams.put("database", db);
                    sqlExecParams.put("sql", sql);
                    sqlExecParams.put("type", sql.split(" ")[0]);
                    String response = HttpClientUtils.doGet(sqlExecUrl, sqlExecParams);
                    log.info("开始在数据库：{} 执行SQL: {}, 返回结果：{}", db, sql, response);
                }
                // 开始循环部署项目
                for (Pipeline pipeline: pipelineList) {
                    log.info("首次创建提测说明，部署项目：{}, 分支：{}, 项目类型：{}, Namespace：{}", pipeline.getServiceName(),
                            pipeline.getServiceBranch(), pipeline.getServiceType(), pipeline.getNamespace());
                    jenkinsService.buildTkeJob(pipeline.getServiceType(), pipeline.getServiceName(), pipeline.getServiceBranch(), pipeline.getNamespace());
                }

                String submitTestUrl = "http://flash-fe.liangkebang.net/qa/detailTestDescription?projectName=" + projectName + "&jiraProjectKey=" + jiraProjectKey;
                String msg = DingdingUtils.buildTestSubmitInstructionMsg(projectName, submitTestUrl);
                boolean sendToDingResult = DingdingUtils.sendToDingding(msg, dingUrl);
                log.info("创建测试流程时，发送给数据组的通知结果：" + sendToDingResult);
            }
            log.info("新增dingRobot以及对应pipeline成功: {}", dingRobotPipelineVo);
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("新增失败", false);
        }
    }

    /**
     * 修改pipeline
     * @param dingRobotPipelineVo
     * @return
     */
    @RequiresPermissions("qa:process:edit")
    @PostMapping("/modify")
    public JsonResult modifyPipeline(@RequestBody DingRobotPipelineVo dingRobotPipelineVo) {
        try {
            DingRobot dingRobot = dingRobotPipelineVo.getDingRobot();
            boolean dingRobotResult = dingRobotService.saveOrUpdate(dingRobot);
            if (!dingRobotResult) {
                return JsonResult.buildErrorStateResult("修改失败！", false);
            }
            Integer dingRobotId = dingRobot.getId();
            List<Pipeline> pipelineList = dingRobotPipelineVo.getPipelineList();
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobotId);
            List<Pipeline> pipelineListInDb = pipelineService.list(pipelineQueryWrapper);
            // 判断是否有删除的部分
            if (pipelineListInDb.size() > 0) {
                for (Pipeline pipelineDb: pipelineListInDb) {
                    boolean flag = false;
                    if (pipelineList.size() > 0) {
                        for (Pipeline pipeline: pipelineList) {
                            if (pipelineDb.getId().equals(pipeline.getId())) {
                                flag = true;
                                break;
                            }
                        }
                    }
                    if (!flag) {
                        pipelineService.removeById(pipelineDb.getId());
                    }
                }
            }

            // 判断是否有新增的
            for (Pipeline pipeline: pipelineList) {
                if (pipeline.getId() == null) {
                    pipeline.setDingRobotId(dingRobotId);
                }
            }
            // 剩下的直接更新
            pipelineService.saveOrUpdateBatch(pipelineList);
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("修改失败！", false);
        }
    }

    /**
     * 删除pipeline
     * @param dingRobotId
     * @return
     */
    @RequiresPermissions("qa:process:del")
    @GetMapping("/delete")
    public JsonResult deletePipeline(@RequestParam Integer dingRobotId) {
        try {
            if (dingRobotService.getById(dingRobotId) != null) {
                dingRobotService.removeById(dingRobotId);
            }
            QueryWrapper<Pipeline> pipelineQueryWrapper = new QueryWrapper<>();
            pipelineQueryWrapper.eq("ding_robot_id", dingRobotId);
            if (pipelineService.list(pipelineQueryWrapper) != null) {
                pipelineService.remove(pipelineQueryWrapper);
            }
            return JsonResult.buildSuccessResult(true);
        } catch (Exception e) {
            return JsonResult.buildErrorStateResult("删除失败", false);
        }
    }

    /**
     * 更新状态
     * @param dingRobotId 机器人id
     * @param status 状态，1-进行中，2-已完成
     * @return
     */
    @RequiresPermissions("qa:process:finish")
    @PostMapping("/finish")
    public JsonResult finishPipeline(@RequestParam Integer dingRobotId, @RequestParam Integer status) {
        try {
            DingRobot dingRobot = dingRobotService.getById(dingRobotId);
            if (dingRobot == null) {
                return JsonResult.buildErrorStateResult("该测试流程不存在！", false);
            }
            // 设置状态为已完成
            dingRobot.setStatus(status);
            if (!dingRobotService.updateById(dingRobot)) {
                return JsonResult.buildErrorStateResult("更新状态失败！", false);
            }
            return JsonResult.buildSuccessResult("更新状态成功！", true);
        } catch (Exception e) {
            return JsonResult.buildErrorStateResult("更新状态失败！", false);
        }
    }

}
