package cn.qg.holmes.utils;

import com.alibaba.fastjson.JSON;
import com.atlassian.jira.rest.client.api.domain.Issue;
import lombok.extern.slf4j.Slf4j;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 钉钉发送消息类
 * @author libo
 */
@Slf4j
public class DingdingUtils {

    /**
     * 发送钉钉消息基本方法
     * @param jsonString 发送的json字符串
     * @param webhook 钉钉webhook
     * @return 是否发送成功，true/false
     */
    public static boolean sendToDingding(String jsonString, String webhook) {
        Map<String, String> headers = new HashMap<>();
        headers.put("Content-Type", "application/json; charset=utf-8");
        String result = HttpClientUtils.doPostJson(webhook, headers, jsonString);
        log.info("发送给钉钉: {}, 内容：{}, 钉钉返回结果：{}", webhook, jsonString, result);
        Map<String, String> resultMap = JSON.parseObject(result, Map.class);
        return resultMap.get("errmsg").equals("ok");
    }

    /**
     * 构建markdown消息基础类
     * @param title 标题
     * @param content 内容
     * @param atAll 是否at所有人
     * @return markdown消息的json字符串
     */
    public static String buildMarkdownMsg(String title, String content, boolean atAll) {
        Map<String, Object> markdown = new HashMap<>();
        Map<String, String> contentMap = new HashMap<>();
        Map<String, Object> atMap = new HashMap<>();
        atMap.put("isAtAll", atAll);
        markdown.put("at", atMap);
        markdown.put("msgtype", "markdown");
        contentMap.put("title", title);
        contentMap.put("text", content);
        markdown.put("markdown", contentMap);
        return JSON.toJSONString(markdown);
    }

    /**
     * 构建BUG刚提交时，发送给钉钉的markdown消息
     * @param key jira issue key
     * @param summary 标题
     * @param creator 创建人
     * @param assignee 经办人
     * @param priority 优先级
     * @param module 模块
     * @return json字符串类型的markdown消息
     */
    public static String buildBugCommitMarkdownMsg(String key, String summary, String creator, String assignee, String priority, String module) {
        String title = "测试同学" + creator + "提交了BUG：" + key;
        String issueUrl = "http://jira2.quantgroup.cn/browse/" + key;
        String content = "#### 概要：[" + summary + "](" + issueUrl + ")"+ "\n"
                + "- 优先级：" + priority + "\n"
                + "- 模块：" + module + "\n"
                + "- 经办人：" + assignee;
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * for 数据工单
     * 根据issueList构建未解决issueList的markdown格式
     * @param issues Issue的迭代器
     * @return
     */
    public static String buildMarkdownMsgForUnsolvedIssueList(Iterable<Issue> issues) {
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        String title =  "今日剩余未解决BUG统计";
        String content = "#### 今日剩余未解决BUG数：" + issueList.size() + "\n";
        int count = 0;
        for (Issue issue: issueList) {
            count = count + 1;
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
            content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
            content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
            content = content + "- 状态：" + issue.getStatus().getName() + "\n";
            content = content + "- 经办人：" + issue.getAssignee().getDisplayName() + "\n\n";
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建测试进度的markdown消息
     * @param issues issue迭代器
     * @param progress 测试进度
     * @param risk 风险点
     * @return 测试进度markdown消息的json字符串
     */
    public static String buildTestScheduleMarkdownMsg(Iterable<Issue> issues, String progress, String risk) {
        List<Issue> issueList = new ArrayList<>();
        for (Issue issue: issues) {
            issueList.add(issue);
        }
        Date date=new Date();
        DateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        String title =  format.format(date) + "今日测试进度汇报";
        String content = format.format(date) + "\n\n";
        content +=  "今日测试进度：" + progress + " \n\n";
        content = content + "剩余未解决BUG列表：" + "\n\n";
        if (issueList.size() == 0) {
            content += "暂无未解决BUG！";
        }
        int count = 0;
        for (Issue issue: issueList) {
            count += 1;
            String issueUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
            content = content + count + "、[" + issue.getSummary().replace("\"", "") + "](" + issueUrl + ")\n";
            content = content + "- 优先级：" + issue.getPriority().getName() + "\n";
            content = content + "- 状态：" + issue.getStatus().getName() + "\n";
            content = content + "- 经办人：" + issue.getAssignee().getDisplayName() + "\n\n";
        }
        if (risk != null && !risk.isEmpty()) {
            content =  content + "**风险：" + risk + "**\n";
        }
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建BUG fix 提醒消息
     * @param issue Jira Issue
     * @return
     */
    public static String buildBugFixRemindMsg(Issue issue, Integer elapsed) {
        String duration = "";
        if (elapsed >= 4 && elapsed <8) {
            duration = elapsed.toString() + "小时";
        }
        if (elapsed >= 8) {
            duration = (elapsed / 8) + "天";
        }
        String title = "BUG修复提醒";
        String content = issue.getAssignee().getDisplayName() + "同学，你有一个BUG已经超过" + duration + "未解决，测试同学等得花儿都谢啦，麻烦赶紧修复哦！\n\n";
        String bugUrl = "http://jira2.quantgroup.cn/browse/" + issue.getKey();
        content += "BUG具体信息：\n\n";
        content += "- 标题：[" + issue.getSummary().replace("\"", "") + "](" + bugUrl + ")\n";
        content += "- 优先级：" + issue.getPriority().getName() + "\n";
        content += "- 经办人：" + issue.getAssignee().getDisplayName() + "\n\n";
        return buildMarkdownMsg(title, content, true);
    }

    /**
     * 构建pipeline钉钉消息
     * @param serviceName 服务名
     * @param serviceBranch 分支
     * @param creator 创建人
     * @param commitMsg 提交信息
     * @param date 日期
     * @param namespace 环境
     * @return
     */
    public static String buildPipelineMarkdownMsg(String serviceName, String serviceBranch, String creator, String commitMsg, String date, String namespace) {
        String title = "测试环境自动构建提醒";
        String content = "> 项目名称：" + serviceName + "\n\n"
                + "> 分支名称：" + serviceBranch + "\n\n"
                + "> 部署空间：" + namespace + "\n\n"
                + "> 触发同学：" + creator + "\n\n"
                + "> 触发信息：" + commitMsg + "\n\n"
                + "> 触发时间：" + date + "\n\n"
                + "[查看详情](" + "" + ")";
        return buildMarkdownMsg(title, content, false);
    }

    public static void main(String[] args) {
//        String markdown = buildMarkdownMsg("YXM-1499", "【羊小咩v7.6.00】【VCC首次交易率提升专题】巴拉巴拉", "黎博", "于巧玲", "p1", "kddsp");
        String markdown = buildPipelineMarkdownMsg("holmes", "master", "bo.li", "测试", "2021-06-03 14:59:45", "fe");
        sendToDingding(markdown, "https://oapi.dingtalk.com/robot/send?access_token=835663338d638e40daaf3ab358af741ef0680a826a962c91bedc53b149d85ee1");
    }
}
