package cn.qg.holmes.controller.effect;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.service.effect.DatabaseSyncService;
import cn.qg.holmes.utils.JenkinsService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/dbsync")
public class DbSyncController {

    @Autowired
    DatabaseSyncService databaseSyncService;

    @Autowired
    JenkinsService jenkinsService;

    @Value("${dbsync.mysql.ip}")
    private String ip;

    @Value("${dbsync.mysql.port}")
    private String port;

    @Value("${dbsync.mysql.username}")
    private String username;

    @Value("${dbsync.mysql.password}")
    private String password;

    @GetMapping("/one")
    public JsonResult syncSingleTable(@RequestParam String namespace, @RequestParam String dbName, @RequestParam String tableName) {
        try {
            long startTime = System.currentTimeMillis();
            Map<String, String> map = databaseSyncService.getMysqlInfoByNamespace(namespace);
            String destIp = map.get("ip");
            String destPort = map.get("port");
            log.info("获取到{}环境的Mysql地址为：{}", namespace, destIp + ":" + destPort);
            if (tableName.equalsIgnoreCase("all") || tableName.equals("")) {
                log.info("开始同步{}库下所有的表", dbName);
                databaseSyncService.getDbInfoFromSource(ip, port, username, password, dbName);
                databaseSyncService.syncDbToDest(destIp, destPort, "qa", "qatest", dbName, namespace);
            } else {
                log.info("开始同步{}库下{}表", dbName, tableName);
                databaseSyncService.getSingleTableFromSource(ip, port, username, password, dbName, tableName);
                databaseSyncService.syncSingleTableToDest(destIp, destPort, "qa", "qatest", dbName, tableName);
            }
            long endTime = System.currentTimeMillis();
            long elapsedTime = (endTime - startTime) / 1000;
            return JsonResult.buildSuccessResult(elapsedTime + "s");
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("同步失败！", false);
        }
    }

    /**
     * 获取数据库列表
     * @return
     */
    @GetMapping("/dbs")
    public JsonResult getDatabaseList() {
        return JsonResult.buildSuccessResult(databaseSyncService.getDatabaseList(ip, port, username, password));
    }

    /**
     * 获取某个库下的表列表
     * @param dbName 数据库名
     * @return
     */
    @GetMapping("/tables")
    public JsonResult getTableList(@RequestParam String dbName) {
        List<Object> tableList = databaseSyncService.getTableListByDb(ip, port, username, password, dbName);
        tableList.add(0, "all");
        return JsonResult.buildSuccessResult(tableList);
    }

    /**
     * 老数据库同步，调用Jenkins job sync_database_schema
     * @param dbName 数据库名
     * @param host 数据库
     * @param port 端口号
     * @param replaceDomain 是否替换域名
     * @param notDeleteBusinessData 是否删除业务数据
     * @param useCache 是否使用两个小时之前的备份
     * @return
     */
    @PostMapping("/old")
    public JsonResult dbSyncOld(String dbName, String host, String port, boolean replaceDomain, boolean notDeleteBusinessData, boolean useCache) {
        int buildNum = jenkinsService.buildDbSyncJob(dbName, host, port, replaceDomain, notDeleteBusinessData,  useCache);
        if (buildNum == 0)  {
            return JsonResult.buildErrorStateResult("数据库同步失败！", false);
        } else {
            return JsonResult.buildSuccessResult("开始同步数据库！", true);
        }
    }

    @GetMapping("/query")
    public JsonResult queryDbSyncResult(String jobName, int buildNum) {
        try {
            Map<String, String> result = jenkinsService.getBuildInfo(jobName, buildNum);
            return JsonResult.buildSuccessResult(result);
        }catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("查询失败", false);
        }
    }
}
