package cn.qg.holmes.task.jira;

import cn.qg.holmes.entity.jira.JiraBugPool;
import cn.qg.holmes.service.jira.JiraBugPoolService;
import cn.qg.holmes.service.jira.JiraIssueService;
import cn.qg.holmes.utils.DingdingUtils;
import com.atlassian.jira.rest.client.api.domain.Issue;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * Jira BUG通知到钉钉定时任务类
 */
@Component
@Slf4j
public class JiraToDingding {

    @Autowired
    JiraIssueService jiraIssueService;

    @Autowired
    JiraBugPoolService jiraBugPoolService;

    @Value("${sjgd.ding.url}")
    private String sjgdDingUrl;

    @Value("${task.debug}")
    private String taskDebug;

    /**
     * 数据工单bug统计
     * @throws Exception
     */
    @Scheduled(cron = "0 0 19 * * ?")
    public void SJGDJiraIssueStatisticsTimedJob() throws Exception {
        if (taskDebug.equals("true")) {
            log.info("开始执行数据工单BUG统计定时任务！");
            Iterable<Issue> issues = jiraIssueService.getUnsolvedIssueListByProject("SJGD");
            String markdownMsg = DingdingUtils.buildMarkdownMsgForUnsolvedIssueList(issues);
            DingdingUtils.sendToDingding(markdownMsg, sjgdDingUrl);
        } else {
            log.info("当前部署的是mock，不执行定时任务！");
        }
    }

    @Scheduled(cron = "0 */1 * * * ?")
    public void cycleJiraBugPool() throws Exception {
        log.info("开始执行jira bug pool定时任务！");
        QueryWrapper<JiraBugPool> jiraBugPoolQueryWrapper = new QueryWrapper<>();
        jiraBugPoolQueryWrapper.eq("enable", 1);
        List<JiraBugPool> jiraBugPoolList = jiraBugPoolService.list(jiraBugPoolQueryWrapper);
        for (JiraBugPool jiraBugPool: jiraBugPoolList) {
            Issue issue = jiraIssueService.getJiraIssueByKey(jiraBugPool.getKey());
            String resolveResult = issue.getFieldByName("解决结果").getName();
            // 如果已解决或已关闭，仅修改状态
            if (resolveResult.equals("已解决") || resolveResult.equals("已关闭")) {
                jiraBugPool.setEnable(0);
                // 更新状态
                jiraBugPoolService.saveOrUpdate(jiraBugPool);
            } else {
                DateTime startDate = issue.getCreationDate();
                DateTime endDate = issue.getUpdateDate();
                int duration = jiraIssueService.calculateBugFixTime(startDate, endDate);
                // 如果已超过4个小时，则发送钉钉通知
                if (duration >= 4) {
                    DingdingUtils.sendToDingding(jiraBugPool.getSummary(), jiraBugPool.getDingUrl());
                }
            }
        }
    }
}
