package cn.qg.holmes.config;

import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigService;
import com.ctrip.framework.apollo.build.ApolloInjector;
import com.ctrip.framework.apollo.core.ConfigConsts;
import com.ctrip.framework.apollo.core.enums.Env;
import com.ctrip.framework.apollo.internals.ConfigServiceLocator;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigRegistrar;
import com.ctrip.framework.apollo.spring.annotation.EnableApolloConfig;
import com.ctrip.framework.apollo.util.http.HttpRequest;
import com.ctrip.framework.apollo.util.http.HttpResponse;
import com.ctrip.framework.apollo.util.http.HttpUtil;
import com.ctrip.framework.foundation.Foundation;
import com.ctrip.framework.foundation.internals.provider.DefaultApplicationProvider;
import com.ctrip.framework.foundation.internals.provider.DefaultServerProvider;
import com.google.common.collect.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.boot.context.properties.source.ConfigurationPropertySource;
import org.springframework.boot.context.properties.source.ConfigurationPropertySources;
import org.springframework.boot.logging.LogFile;
import org.springframework.boot.logging.LoggingInitializationContext;
import org.springframework.boot.logging.LoggingSystem;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.env.CompositePropertySource;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.util.ResourceUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@Configuration
@EnableApolloConfig
@Import(ApolloConfigRegistrar.class)
public class ApolloPropertySourceInitializer implements ApplicationContextInitializer<ConfigurableApplicationContext>, Ordered {


    private int order = Ordered.HIGHEST_PRECEDENCE + 5;

    private final Multimap<Integer, String> NAMESPACE_NAMES = HashMultimap.create();

    private static final Logger logger = LoggerFactory.getLogger(ConfigServiceLocator.class);

    private static final int DEFAULT_ORDER = Ordered.LOWEST_PRECEDENCE;
    private static final String DEFAULT_NAMESPACE = ConfigConsts.NAMESPACE_APPLICATION;

    //todo 需要补充服务地址. 暂时测试

    public ApolloPropertySourceInitializer() {
        this(Integer.parseInt(Foundation.app().getProperty("order", String.valueOf(DEFAULT_ORDER))),
                StringUtils.commaDelimitedListToStringArray(Foundation.app().getProperty("namespace", DEFAULT_NAMESPACE)));
    }

    public ApolloPropertySourceInitializer(String... namespace) {
        this(DEFAULT_ORDER, namespace);
    }

    public ApolloPropertySourceInitializer(int order, String... namespace) {
        ArrayList<String> namespaceArr = Lists.newArrayList(namespace);
        if (!namespaceArr.contains(DEFAULT_NAMESPACE)) {
            NAMESPACE_NAMES.putAll(DEFAULT_ORDER, Lists.newArrayList(DEFAULT_NAMESPACE));
        }
        NAMESPACE_NAMES.putAll(order, namespaceArr);
    }

    @Override
    public int getOrder() {
        return this.order;
    }

    @Override
    public void initialize(ConfigurableApplicationContext applicationContext) {
        ConfigurableEnvironment environment = applicationContext.getEnvironment();
        if (environment.getPropertySources().contains("ApolloPropertySources")) {
            return;
        }

        initIpAndNamespace(environment);
        initKubernetesEnv(environment);

        logger.info("Fetching remote config by apollo and try to reinitialize logging system if necessary.");

        CompositePropertySource composite = new CompositePropertySource("ApolloPropertySources");
        ImmutableSortedSet<Integer> orders = ImmutableSortedSet.copyOf(NAMESPACE_NAMES.keySet());
        UnmodifiableIterator iterator = orders.iterator();

        while (iterator.hasNext()) {
            int order = ((Integer) iterator.next()).intValue();
            Iterator it = NAMESPACE_NAMES.get(order).iterator();

            while (it.hasNext()) {
                String namespace = (String) it.next();
                Config config = ConfigService.getConfig(namespace);
                composite.addPropertySource(new ConfigPropertySource(namespace, config));
            }
        }
        environment.getPropertySources().addFirst(composite);

        //reinitialize logging system.
        reinitializeLoggingSystem(environment);

        DefaultApplicationProvider defaultApplicationProvider = new DefaultApplicationProvider();
        defaultApplicationProvider.initialize();
        DefaultServerProvider defaultServerProvider = new DefaultServerProvider();
        defaultServerProvider.initialize();
    }

    /**
     * 初始化 tech.localhost 参数
     * 初始化 NAMESPACE 参数
     *
     * @param environment
     */
    private void initIpAndNamespace(ConfigurableEnvironment environment) {
        //获取本机 IP
        String localhost = environment.getProperty("tech.localhost", Foundation.app().getProperty("tech.localhost", null));
        if (StringUtils.isEmpty(localhost)) {
            localhost = Foundation.net().getHostAddress();
        } else {
            logger.warn("获取 tech.localhost 信息结果 : {}, 测试环境应为本机IP", localhost);
        }
        environment.getSystemProperties().put("tech.localhost", localhost);
        //如果不存在命名空间, 本机 IP 的后缀当做命名空间.
        String[] ipSplit = localhost.split("\\.");
        String localNamespace = environment.getProperty("NAMESPACE", ipSplit[ipSplit.length - 1]);
        environment.getSystemProperties().putIfAbsent("NAMESPACE", localNamespace);
    }

    private void initKubernetesEnv(ConfigurableEnvironment environment) {
        //如果系统中存在namespace, 那我们什么都不做好了. 这时候应该是 kubernetes 环境
        if (System.getenv("NAMESPACE") != null) {
            //do nothing
            logger.info("运行在 kubernets 环境.");
            return;
        }
        boolean isPro = Env.PRO.name().equalsIgnoreCase(Foundation.server().getEnvType());
        //如果是生产环境, 我也啥都不干. 太吓人了.
        if (isPro) {
            logger.info("哇, 生产环境. 配置中心静悄悄. 什么都不敢做.");
            return;
        }
        String namespace = environment.getProperty("NAMESPACE", Foundation.app().getProperty("NAMESPACE", null));
        if (namespace == null) {
            logger.info("你好像没有配置 NAMESPACE 哦?你不打算连接到 kubernetes 内部么?");
            return;
        }

        HttpUtil httpUtil = ApolloInjector.getInstance(HttpUtil.class);
        String kubernetesServer = Foundation.app().getProperty("eos_server_host", "http://eos.quantgroups.com/");

        HttpRequest httpRequest = new HttpRequest(kubernetesServer + "api/apollo/env_vars?namespace=" + namespace);

        HttpResponse<KubeEnvInfo> mapHttpResponse = httpUtil.doGet(httpRequest, KubeEnvInfo.class);
        KubeEnvInfo body = mapHttpResponse.getBody();
        if (body != null && body.success) {
            logger.info("客官请放心, kubernets 的环境变量已经注入, 您可以放心的在 kubernetes 之外启动你的服务了");
            environment.getSystemProperties().putAll(body.details);
            return;
        }
        logger.error("额... 看起来 kubernetes server 有点问题, 返回了false, serverInfo:{} ,body:{}", kubernetesServer, body);
    }

    private static class KubeEnvInfo {
        boolean success;
        Map<String, String> details;
    }

    private void reinitializeLoggingSystem(ConfigurableEnvironment environment) {

        Iterable<ConfigurationPropertySource> sources = ConfigurationPropertySources.get(environment);
        Binder binder = new Binder(sources);
        BindResult<Properties> bindResult = binder.bind( "",Properties.class);
        Properties properties = bindResult.get();
        Set<Object> keys = new HashSet<>();
        properties.forEach((i,j)->{
            keys.add(i);
        });
        if (!keys.stream().filter(i->StringUtils.startsWithIgnoreCase((String) i,"logging.")).collect(Collectors.toSet()).isEmpty()) {
            String logConfig = environment.resolvePlaceholders("${logging.config:}");
            LogFile logFile = LogFile.get(environment);
            LoggingSystem system = LoggingSystem
                    .get(LoggingSystem.class.getClassLoader());
            try {
                ResourceUtils.getURL(logConfig).openStream().close();
                // Three step initialization that accounts for the clean up of the logging
                // context before initialization. Spring Boot doesn't initialize a logging
                // system that hasn't had this sequence applied (since 1.4.1).
                system.cleanUp();
                system.beforeInitialize();
                system.initialize(new LoggingInitializationContext(environment),
                        logConfig, logFile);
            } catch (Exception ex) {
                ApolloPropertySourceInitializer.logger
                        .warn("Logging config file location '" + logConfig
                                + "' cannot be opened and will be ignored");
            }
        }
    }


}
