package cn.qg.holmes.controller.k8s;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.auth.UserInfoVo;
import cn.qg.holmes.entity.k8s.DockerProject;
import cn.qg.holmes.entity.k8s.ServiceCreateVo;
import cn.qg.holmes.service.auth.TokenService;
import cn.qg.holmes.service.k8s.*;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.tencentcloudapi.tcr.v20190924.models.RepoInfo;
import com.tencentcloudapi.tcr.v20190924.models.TagInfo;
import com.tencentcloudapi.tcr.v20190924.models.TagInfoResp;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * k8及容器操作相关接口
 * @author libo
 */
@Slf4j
@CrossOrigin
@RestController
@RequestMapping("/k8s")
public class K8sController {

    @Autowired
    K8sService k8sService;

    @Autowired
    ImageService imageService;

    @Autowired
    DockerProjectService dockerProjectService;

    @Autowired
    MqService mqService;

    @Autowired
    RedisService redisService;

    @Autowired
    TokenService tokenService;

    /**
     * 获取namespace列表
     * @return
     */
    @GetMapping("/namespace/list")
    public JsonResult getNamespaceList(String env) {
        return JsonResult.buildSuccessResult(k8sService.getNamespaceList(env));
    }

    /**
     * 获取pod列表
     * @param namespace 环境
     * @return
     */
    @GetMapping("/service/list")
    public JsonResult getServiceList(@RequestParam String namespace) {
        List<Map<String, Object>> podList = k8sService.getPodListNew(namespace);
        return JsonResult.buildSuccessResult(podList);
    }

    @GetMapping("/service/detail")
    public JsonResult getServiceDetail(@RequestParam String namespace,
                                       @RequestParam String serviceType,
                                       @RequestParam String serviceName) {
        return JsonResult.buildSuccessResult(k8sService.getPodAndServiceInfo(namespace, serviceType, serviceName));
    }

    /**
     * 根据服务名称获取镜像列表
     * @param repoName 服务名称
     * @return
     */
    @GetMapping("/image/list")
    public JsonResult getImageListByServiceName(@RequestParam String repoName) {
        return JsonResult.buildSuccessResult(JSON.toJSONString(imageService.getImageListByService(repoName)));
    }

    /**
     * 新增deployment, 如果service和pvc没有，也会同时创建
     * @param serviceCreateVo 创建实体类
     * @return
     */
    @PostMapping("/service/create")
    public JsonResult createPod(@RequestBody ServiceCreateVo serviceCreateVo) {
        try {
            String serviceName = serviceCreateVo.getServiceName();
            String type = serviceCreateVo.getType();
            String label = serviceCreateVo.getLabel();
            String namespace = serviceCreateVo.getNamespace();
            String image = serviceCreateVo.getImage();
            // 如果Deployment已存在，则不重新添加
            if (k8sService.queryIfDeploymentExistByName(namespace, serviceName)) {
                return JsonResult.buildErrorStateResult("服务已存在，无法重复添加！", false);
            }
            QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("project_name", serviceName);
            if (label.equals("base")) {
                // 如果有Service，则首先删除Service
                if (k8sService.queryIfServiceExistByName(namespace, serviceName)) {
                    k8sService.deleteService(namespace, serviceName);
                }
                // 重新创建Service
                switch (serviceName) {
                    case "mysql":
                        k8sService.createMysqlService(namespace);
                        break;
                    case "redis":
                        k8sService.createRedisService(namespace);
                        break;
                    case "mongodb":
                        k8sService.createMongodbService(namespace);
                        break;
                    case "zookeeper":
                        k8sService.createZookeeperService(namespace);
                        break;
                    case "rabbitmq":
                        k8sService.createRabbitmqService(namespace);
                        break;
                    default:
                        break;
                }
                // Pvc不存在则创建，已存在无需重新创建
                if (!k8sService.queryIfPvcExistByName(namespace, serviceName)) {
                    switch (serviceName) {
                        case "mysql":
                            k8sService.createMysqlPvc(namespace);
                            break;
                        case "mongodb":
                            k8sService.createMongodbPvc(namespace);
                            break;
                        case "zookeeper":
                            k8sService.createZookeeperPvc(namespace);
                            break;
                        case "rabbitmq":
                            k8sService.createRabbitmqPvc(namespace);
                            break;
                        default:
                            break;
                    }
                }
                // 创建Deployment
                switch (serviceName) {
                    case "mysql":
                        k8sService.createMysqlDeployment(namespace, image);
                        break;
                    case "redis":
                        k8sService.createRedisDeployment(namespace, image);
                        break;
                    case "mongodb":
                        k8sService.createMongoDbDeployment(namespace, image);
                        break;
                    case "zookeeper":
                        k8sService.createZookeeperDeployment(namespace, image);
                        break;
                    case "rabbitmq":
                        k8sService.createRabbitmqDeployment(namespace, image);
                        break;
                    default:
                        break;
                }
            } else if (label.equals("java")) {
                DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
                // 直接创建Java Deployment
                k8sService.createJavaDeployment(serviceCreateVo, dockerProject);

                // 如果存在Service，则首先删除，再创建新的
                if (k8sService.queryIfServiceExistByName(namespace, serviceName)) {
                    k8sService.deleteService(namespace, serviceName);
                }
                k8sService.createJavaService(namespace, serviceName, serviceCreateVo.getDebug());

                // 如果存在Ingress，则首先删除Ingress，再创建新的
                if (k8sService.queryIfIngressExistByName(namespace, serviceName)) {
                    k8sService.deleteIngress(namespace, serviceName);
                }
                k8sService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());

            } else if (label.equals("ui") || label.equals("node")) {
                DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
                // 如果存在Service，则首先删除，再创建新的
                if (k8sService.queryIfServiceExistByName(namespace, serviceName)) {
                    k8sService.deleteService(namespace, serviceName);
                }
                k8sService.createUIAndNodeService(namespace, serviceName, type, label);

                // 如果存在Ingress，则首先删除Ingress，再创建新的
                if (k8sService.queryIfIngressExistByName(namespace, serviceName)) {
                    k8sService.deleteIngress(namespace, serviceName);
                }
                k8sService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());

                k8sService.createUIAndNodeDeployment(serviceCreateVo, dockerProject);

            } else {
                return JsonResult.buildErrorStateResult("服务类型暂不支持！", false);
            }
            return JsonResult.buildSuccessResult("pod创建成功！");
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("pod创建失败！", false);
        }
    }

    /**
     * 重置pod
     * @param namespace 环境
     * @param podName podName
     * @return
     */
    @PostMapping("/service/redeploy")
    public JsonResult resetPodByName(String namespace, String podName) {
        return JsonResult.buildSuccessResult(k8sService.resetPod(namespace, podName));
    }

    /**
     * jenkins打包部署镜像接口，专门提供给global-jenkinsfile使用
     * @param serviceCreateVo
     * @return
     */
    @PostMapping("/service/modify")
    public JsonResult modifyPod(@RequestBody ServiceCreateVo serviceCreateVo) {
        log.info("接口/k8s/service/modify收到请求，请求参数为：{}", JSON.toJSONString(serviceCreateVo));
        String serviceName = serviceCreateVo.getServiceName();
        String type = serviceCreateVo.getType();
        String label = serviceCreateVo.getLabel();
        String namespace = serviceCreateVo.getNamespace();
        //  默认mock为0
        if (serviceCreateVo.getMock() ==  null) {
            serviceCreateVo.setMock(0);
        }
        // 默认设置debug为0
        if (serviceCreateVo.getDebug() == null) {
            serviceCreateVo.setDebug(0);
        }
        QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_name", serviceName);
        if (label.equals("java")) {
            // JAVA
            DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
            // 创建Deployment
            if (k8sService.queryIfDeploymentExistByName(namespace, serviceName)) {
                log.info("{}环境{}服务的Deployment已存在，首先删除该Deployment.", namespace, serviceName);
                k8sService.deleteDeployment(namespace, serviceName);
            }
            k8sService.createJavaDeployment(serviceCreateVo, dockerProject);

            // 如果不存在Service，则重新创建Service
            if (!k8sService.queryIfServiceExistByName(namespace, serviceName)) {
                k8sService.createJavaService(namespace, serviceName, serviceCreateVo.getDebug());
            }

            //  如果不存在ingress则创建ingress
            if (!k8sService.queryIfIngressExistByName(namespace,  serviceName)) {
                k8sService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());
            }
        } else if (label.equals("ui") || label.equals("node")) {
            // UI
            DockerProject dockerProject = dockerProjectService.getOne(queryWrapper);
            // 创建Deployment
            if (k8sService.queryIfDeploymentExistByName(namespace, serviceName)) {
                log.info("{}环境{}服务的Deployment已存在，首先删除该Deployment.", namespace, serviceName);
                k8sService.deleteDeployment(namespace, serviceName);
            }
            k8sService.createUIAndNodeDeployment(serviceCreateVo, dockerProject);

            // 如果不存在Service，则重新创建Service
            if (!k8sService.queryIfServiceExistByName(namespace, serviceName)) {
                k8sService.createUIAndNodeService(namespace, serviceName, type, label);
            }

            //  如果不存在ingress则创建ingress
            if (!k8sService.queryIfIngressExistByName(namespace,  serviceName)) {
                k8sService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());
            }
        } else {
            return JsonResult.buildErrorStateResult("暂不支持！", false);
        }
        return JsonResult.buildSuccessResult("镜像部署成功");
    }

    /**
     * 更新pod
     * @param serviceCreateVo
     * @return
     */
    @PostMapping("/service/update")
    public JsonResult updatePod(@RequestBody ServiceCreateVo serviceCreateVo) {
        String namespace = serviceCreateVo.getNamespace();
        String serviceName = serviceCreateVo.getServiceName();
        String label = serviceCreateVo.getLabel();
        String type = serviceCreateVo.getType();
        // 获取dockerProject
        QueryWrapper<DockerProject> dockerProjectQueryWrapper = new QueryWrapper<>();
        dockerProjectQueryWrapper
                .eq("project_name", serviceName)
                .eq("project_type", type);
        DockerProject dockerProject = dockerProjectService.getOne(dockerProjectQueryWrapper);
        // 先删除原有pod
        if (!k8sService.deleteDeployment(namespace, serviceName)) {
            return JsonResult.buildErrorStateResult("更新服务失败！", false);
        }
        // 再创建新pod
        Deployment deployment = null;
        // 删除原来的Service
        if (k8sService.queryIfServiceExistByName(namespace, serviceName)) {
            k8sService.deleteService(namespace, serviceName);
        }
        // 类型为Java
        if (label.equals("java")) {
            deployment = k8sService.createJavaDeployment(serviceCreateVo, dockerProject);
            k8sService.createJavaService(namespace, serviceName, serviceCreateVo.getDebug());
        }
        // 类型为UI或Node
        if (label.equals("ui") || label.equals("node")) {
            deployment = k8sService.createUIAndNodeDeployment(serviceCreateVo, dockerProject);
            k8sService.createUIAndNodeService(namespace, serviceName, type, label);
        }
        // 删除原有Ingress
        if (k8sService.queryIfIngressExistByName(namespace, serviceName)) {
            k8sService.deleteIngress(namespace, serviceName);
        }
        // 创建新的Ingress
        k8sService.createIngress(namespace, serviceName, serviceCreateVo.getDomain());
        if (deployment != null) {
            return JsonResult.buildSuccessResult("更新成功！", true);
        } else {
            return JsonResult.buildErrorStateResult("更新失败", false);
        }
    }

    /**
     * 删除pod
     * @param namespace 环境
     * @param serviceName 服务名称
     * @return
     */
    @PostMapping("/service/delete")
    public JsonResult deletePodByName(String namespace, String serviceName) {
        return JsonResult.buildSuccessResult(k8sService.deleteDeployment(namespace, serviceName));
    }

    /**
     * 根据类型获取所有的镜像名
     * @param namespace qa-base，qa-ui，qa-java等
     * @return
     */
    @GetMapping("/getSingleTypeApp")
    public JsonResult getSingleTypeApp(@RequestParam String namespace, @RequestParam String type) {
        List<RepoInfo> repoInfoList = imageService.getRepoInfoList(namespace);
        Map<String, Object> resultMap =  new HashMap<>();
        if (type.equals("base")) {
            resultMap.put("RepoInfo", repoInfoList);
            resultMap.put("TotalCount", repoInfoList.size());
            return JsonResult.buildSuccessResult(resultMap);
        }
        if (repoInfoList.size() > 0) {
            List<RepoInfo> newRepoInfoList = new ArrayList<>();
            QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
            queryWrapper
                    .eq("project_type", type)
                    .eq("is_active", 1)
                    .eq("deploy_to_docker", 1);
            List<DockerProject> dockerProjectList = dockerProjectService.list(queryWrapper);
            for (RepoInfo repoInfo: repoInfoList) {
                String projectName = repoInfo.getRepoName().split("/")[1];
                for (DockerProject dockerProject: dockerProjectList) {
                    if (dockerProject.getProjectName().equals(projectName)) {
                        newRepoInfoList.add(repoInfo);
                    }
                }
            }
            resultMap.put("RepoInfo", newRepoInfoList);
            resultMap.put("TotalCount", newRepoInfoList.size());
            return JsonResult.buildSuccessResult(resultMap);
        } else {
            return JsonResult.buildSuccessResult(null);
        }
    }

    /**
     * 同步线上mq数据到测试
     * @param host 测试环境mq地址
     * @return
     */
    @PostMapping("/sync/mq")
    public JsonResult syncOnlineMqToTest(@RequestParam String host) {
        return JsonResult.buildSuccessResult(mqService.setDefinitions(host));
    }

    /**
     * 情况测试环境redis
     * @param namespace 环境
     * @return
     */
    @PostMapping("/redis/flush")
    public JsonResult flushRedis(@RequestParam String namespace) {
        try {
            Map<String, Object> redisInfo = k8sService.getPodAndServiceInfo(namespace, "base", "redis");
            String lanIp = redisInfo.get("lanIp").toString();
            List<Map<String, Object>> portMappings = (List<Map<String, Object>>) redisInfo.get("portMappings");
            for (Map<String, Object> portMap: portMappings) {
                redisService.flushRedis(lanIp, (Integer) portMap.get("nodePort"));
            }
            return JsonResult.buildSuccessResult("redid清除成功！", true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("redis清除失败！", false);
        }
    }

    /**
     * 创建新的namespace
     */
    @PostMapping("/namespace/create")
    public JsonResult createNewNamespace(@RequestParam String name,
                                         @RequestParam String desc,
                                         @RequestParam String owner) {
        return JsonResult.buildSuccessResult(k8sService.createNewNamespace(name, desc, owner));
    }

    /**
     * 根据roleId获取有权限展示按钮的namespace
     */
    @GetMapping("/role/namespace")
    public JsonResult getNamespaceListByRoleId(@RequestHeader String token) {
        UserInfoVo userInfoVo = tokenService.getUserInfoFromCache(token);
        Integer roleId = userInfoVo.getRoleIds().get(0);
        List<String> nsStrList = new ArrayList<>();
        List<Map> nsMapList = new ArrayList<>();
        if (roleId == 1 || roleId == 2) {
            nsMapList = k8sService.getNamespaceList("all");
        }
        if (roleId == 3) {
            nsMapList = k8sService.getNamespaceList("dev");
        }
        for (Map map: nsMapList) {
            nsStrList.add(map.get("name").toString());
        }
        return JsonResult.buildSuccessResult(String.join(",", nsStrList));
    }

    /**
     * 给jenkins构建提供的namespace列表
     * @param env all、test、dev
     * @return
     */
    @GetMapping("/get_namespace_for_jenkins")
    public String getNamespaceForJenkins(@RequestParam String env) {
        List<Map> mapList = k8sService.getNamespaceList(env);
        List<String> resultList = new ArrayList<>();
        for (Map map: mapList) {
            resultList.add(map.get("name").toString());
        }
        resultList.sort(String::compareTo);
        return String.join("\n", resultList);
    }

    /**
     * 按照类型获取所有的服务
     * @return
     */
    @GetMapping("/repo/getAll")
    public JsonResult getAllRepos() {
        List<Map<String, Object>> resultMapList = new ArrayList<>();
        List<RepoInfo> repoInfoList = imageService.getRepoInfoList("qa-base");
        Map<String, Object> base = new HashMap<>();
        List<Map<String, String>> baseList = new ArrayList<>();
        for (RepoInfo repoInfo: repoInfoList) {
            String serviceName = repoInfo.getRepoName().split("/")[1];
            TagInfo[] tagInfos = imageService.getImageListByService(repoInfo.getRepoName()).getTagInfo();
            for (TagInfo tagInfo: tagInfos) {
                Map<String, String> map = new HashMap<>();
                map.put("appname", serviceName + ":" + tagInfo.getTagName());
                map.put("type", serviceName);
                map.put("label", "base");
                map.put("tag", "qa-base/" + serviceName + ":" + tagInfo.getTagName());
                map.put("domain", serviceName + tagInfo.getTagName());
                baseList.add(map);
            }
        }
        base.put("type", "base");
        base.put("repos", baseList);
        resultMapList.add(base);

        QueryWrapper<DockerProject> queryWrapper = new QueryWrapper<>();
        queryWrapper.groupBy("project_type").select("project_type");
        List<DockerProject> projectList = dockerProjectService.list(queryWrapper);
        queryWrapper.clear();

        for (DockerProject dp: projectList) {
            String type = dp.getProjectType();
            Map<String, Object> serviceMap = new HashMap<>();
            List<Map<String, String>> serviceMapList = new ArrayList<>();
            queryWrapper
                    .eq("project_type", type)
                    .eq("is_active", 1);
            List<DockerProject> dockerProjectList = dockerProjectService.list(queryWrapper);
            for (DockerProject dockerProject: dockerProjectList) {
                Map<String, String> map = new HashMap<>();
                map.put("type", type);
                map.put("label", type);
                map.put("appname", dockerProject.getProjectName());
                map.put("domain", dockerProject.getHostName().split("\\.")[0]);
                map.put("tag", "qa-test/" + dockerProject.getProjectName() + ":latest");
                serviceMapList.add(map);
            }
            serviceMap.put("type", type);
            serviceMap.put("repos", serviceMapList);
            resultMapList.add(serviceMap);
            queryWrapper.clear();
        }

        return JsonResult.buildSuccessResult(resultMapList);
    }
}
