package cn.qg.holmes.controller.monitor;

import cn.qg.holmes.common.JsonResult;
import cn.qg.holmes.entity.monitor.MonitorDataSource;
import cn.qg.holmes.entity.monitor.MonitorTask;
import cn.qg.holmes.service.monitor.MonitorDataSourceService;
import cn.qg.holmes.service.monitor.MonitorTaskService;
import cn.qg.holmes.utils.DingdingUtils;
import cn.qg.holmes.utils.JdbcUtils;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 定时任务Controller
 */
@CrossOrigin
@RestController
@RequestMapping("/monitor/task")
public class MonitorTaskController {

    @Autowired
    MonitorTaskService monitorTaskService;

    @Autowired
    MonitorDataSourceService monitorDataSourceService;

    @Autowired
    JdbcTemplate jdbcTemplate;

    /**
     * 获取监控列表，带搜索和分页
     * @param pageNum 第几页
     * @param pageSize 每页多少个
     * @param serviceName 服务名
     * @param taskName 监控名称
     * @return
     */
    @GetMapping("/list")
    public JsonResult getMonitorTaskList(@RequestParam(defaultValue = "1") Integer pageNum,
                                         @RequestParam(defaultValue = "10") Integer pageSize,
                                         @RequestParam String serviceName,
                                         @RequestParam String taskName) {
        QueryWrapper<MonitorTask> queryWrapper = new QueryWrapper<>();
        if (!StringUtils.isEmpty(serviceName)) {
            queryWrapper.eq("service_name", serviceName);
        }
        if (!StringUtils.isEmpty(taskName)) {
            queryWrapper.eq("task_name", taskName);
        }
        queryWrapper.orderByDesc("id");
        IPage<MonitorTask> pageParam = new Page<>(pageNum, pageSize);
        IPage<MonitorTask> monitorTaskIPage = monitorTaskService.page(pageParam, queryWrapper);
        List<MonitorTask> monitorTaskList = monitorTaskIPage.getRecords();
        for (MonitorTask monitorTask: monitorTaskList) {
            monitorTask.setDsName(monitorDataSourceService.getById(monitorTask.getDsId()).getName());
        }
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("list", monitorTaskList);
        resultMap.put("total", monitorTaskIPage.getTotal());
        return JsonResult.buildSuccessResult(resultMap);
    }

    /**
     * 新增监控
     *
     * @param monitorTask
     * @return
     */
    @PostMapping("/add")
    public JsonResult addMonitorTask(@RequestBody MonitorTask monitorTask) {
        return JsonResult.buildSuccessResult(monitorTaskService.save(monitorTask));
    }

    /**
     * 修改监控
     *
     * @param monitorTask
     * @return
     */
    @PostMapping("/edit")
    public JsonResult editMonitorTask(@RequestBody MonitorTask monitorTask) {
        return JsonResult.buildSuccessResult(monitorTaskService.saveOrUpdate(monitorTask));
    }

    /**
     * 删除监控
     *
     * @param monitorTask
     * @return
     */
    @PostMapping("/delete")
    public JsonResult deleteMonitorTask(@RequestBody MonitorTask monitorTask) {
        Integer taskId = monitorTask.getId();
        if (monitorTaskService.getById(taskId) == null) {
            return JsonResult.buildErrorStateResult("定时任务不存在!", false);
        }
        boolean delResult = monitorTaskService.removeById(taskId);
        if (delResult) {
            return JsonResult.buildSuccessResult("删除成功!");
        } else {
            return JsonResult.buildErrorStateResult("删除失败!", false);
        }
    }

    /**
     * 立即执行SQL
     *
     * @param dsId 数据源id
     * @param sql  SQL
     * @return
     */
    @PostMapping("/execute")
    public JsonResult executeSql(@RequestParam Integer dsId, @RequestParam String sql) {
        MonitorDataSource monitorDataSource = monitorDataSourceService.getById(dsId);
        List<Map<String, Object>> resultList = JdbcUtils.queryForList(monitorDataSource.getHost(), monitorDataSource.getPort(), monitorDataSource.getUsername(),
                monitorDataSource.getPassword(), null, sql);
        return JsonResult.buildSuccessResult(resultList);
    }

    /**
     * 发送测试，立即测试下定时任务发送消息是否准确
     *
     * @param monitorTask 监控任务
     * @return
     */
    @PostMapping("/test")
    public JsonResult testMonitorTask(@RequestBody MonitorTask monitorTask) {
        try {
            Integer dsId = monitorTask.getDsId();
            MonitorDataSource monitorDataSource = monitorDataSourceService.getById(dsId);
            List<Map<String, Object>> resultList = JdbcUtils.queryForList(monitorDataSource.getHost(), monitorDataSource.getPort(),
                    monitorDataSource.getUsername(), monitorDataSource.getPassword(), null, monitorTask.getSql());
            String dingUrl = monitorTask.getDingUrl();
            String dingText = monitorTask.getDingText();
            dingText = dingText.replaceAll("\\{count}", String.valueOf(resultList.size()));
            String markdownMsg = DingdingUtils.buildMarkdownMsg("监控测试", dingText + "\n" + JSON.toJSONString(resultList), false);
            boolean result = DingdingUtils.sendToDingding(markdownMsg, dingUrl);
            return JsonResult.buildSuccessResult(result ? "发送成功！" : "发送失败！", true);
        } catch (Exception e) {
            e.printStackTrace();
            return JsonResult.buildErrorStateResult("发送异常！", false);
        }
    }
}
