package com.js.sync.utils;

import cn.hutool.core.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.session.SqlSessionFactory;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

/**
 * 数据库工具类需配合通用 mapper 使用。
 */
@Slf4j
public class DBUtil {

    /**
     * 批量插入
     *
     * 性能不及通用 mapper 中的 insertList,但无数据大小限制。
     * @param list  保存对象的列表，需要相应的类型
     * @param mapperClazz 将要使用的 mapper  class
     * @param sqlSessionFactory mybatis session 工场，可以在所需类中直接使用 autowired 注入该对象。
     *
     */
    public static <T> void insertBatch(List<T> list, Class mapperClazz, SqlSessionFactory sqlSessionFactory) {
        if (ObjectUtil.isEmpty(list) || ObjectUtil.isEmpty(mapperClazz) || ObjectUtil.isEmpty(sqlSessionFactory)) {
            return;
        }
        // 使用 mabatis 的批量插入功能。
        SqlSession sqlSession = sqlSessionFactory.openSession(ExecutorType.BATCH,false);
        if (!ObjectUtil.isEmpty(list)) {
            try {
                Object sqlSessionMapper = sqlSession.getMapper(mapperClazz);
                Method insertSelective = mapperClazz.getMethod("insertSelective", Object.class);
                for (T item : list) {
                    insertSelective.invoke(sqlSessionMapper, item);
                }
                sqlSession.commit();
                log.info("批量插入成功");
            } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
                sqlSession.rollback();
                log.error("批量插入功能调用失败。", e);
            } finally {
                sqlSession.close();
            }
        }
    }

    /**
     * 批量更新
     *
     * @param list  更新对象的列表，需要相应的类型
     * @param mapperClazz 将要使用的 mapper  class
     * @param sqlSessionFactory mybatis session 工场，可以在所需类中直接使用 autowired 注入该对象。
     *
     */
    public static <T> void updateBatch(List<T> list, Class mapperClazz, SqlSessionFactory sqlSessionFactory) {
        if (ObjectUtil.isEmpty(list) || ObjectUtil.isEmpty(mapperClazz) || ObjectUtil.isEmpty(sqlSessionFactory)) {
            return;
        }
        // 使用 mabatis 的批量更新功能。
        SqlSession sqlSession = sqlSessionFactory.openSession(ExecutorType.BATCH,false);
        if (!ObjectUtil.isEmpty(list)) {
            try {
                Object sqlSessionMapper = sqlSession.getMapper(mapperClazz);
                Method updateByPrimaryKeySelective = mapperClazz.getMethod("updateByPrimaryKeySelective", Object.class);
                for (int i = 0; i < list.size(); i++) {
                    updateByPrimaryKeySelective.invoke(sqlSessionMapper, list.get(i));
                    if (i != 0 && i % 1000 == 0) {
                        sqlSession.commit();
                    } else if (i == list.size() - 1) {
                        sqlSession.commit();
                    } else {
                        // nothing to do
                    }
                }
                sqlSession.commit();
                log.info("批量更新成功");
            } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
                sqlSession.rollback();
                log.error("批量更新功能调用失败。", e);
            } finally {
                sqlSession.close();
            }
        }
    }
}
