package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazonservices.mws.client.MwsUtl;
import com.amazonservices.mws.orders._2013_09_01.*;
import com.amazonservices.mws.orders._2013_09_01.model.*;
import com.js.common.constant.Constant;
import com.js.common.enums.AmazonEndpointEnum;
import com.js.dal.dao.mapper.JsSyncAmazonOrderAddressMapper;
import com.js.dal.dao.mapper.JsSyncAmazonOrderMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsSyncAmazonOrder;
import com.js.dal.dao.model.JsSyncAmazonOrderAddress;
import com.js.dal.dao.model.KycStore;
import com.js.api.sync.service.AmazonOrderService;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.XMLGregorianCalendar;
import java.time.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonOrderServiceImpl implements AmazonOrderService {

    @Autowired
    SqlSessionFactory sqlSessionFactory;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonOrderAddressMapper jsSyncAmazonOrderAddressMapper;
    @Autowired
    KycStoreMapper kycStoreMapper;

    @Override
    public CompletableFuture<Void> startSyncOrderWhole(String storeId) {
        // 抓取条件设为当前时间的两年之前
        Date syncDate = Date.from(ZonedDateTime.of(LocalDateTime.now(), ZoneId.of("UTC")).minusYears(2).toInstant());
        log.info("准备全量抓取，抓取最近两年的订单。使用时间为：{}", syncDate);
        try {
            startSyncOrderBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncOrderBySyncDate 异常", e);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> startSyncOrderIncrement(String storeId) {
        Date previousSyncDate = AmaMWSCommonUtil.getSyncDate(storeId, JsSyncAmazonOrder.class, jsSyncAmazonOrderMapper);
        if (previousSyncDate.getTime() < 0) {
            log.info("亚马逊店铺订单尚未获取到上次更新时间，改为使用当前时间。店铺 id: {}", storeId);
            previousSyncDate = new Date();
        }
        log.info("准备增量抓取，抓取最近一个月的订单。使用时间为：{}", previousSyncDate);
        Date syncDate = Date.from(ZonedDateTime.ofInstant(previousSyncDate.toInstant(), ZoneId.of("UTC")).minusMonths(1).toInstant());
        try {
            startSyncOrderBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncOrderBySyncDate 异常", e);
        }
        return CompletableFuture.completedFuture(null);
    }

    /**
     * 根据指定的时间开始同步数据。
     */
    private void startSyncOrderBySyncDate(String storeId, Date syncDate) {
        if (ObjectUtil.isEmpty(storeId) || ObjectUtil.isEmpty(syncDate)) {
            log.error("亚马逊订单同步失败 storeId, syncDate 不可以为空");
            return;
        }
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            log.error("订单同步失败，无此店铺 id：{}", storeId);
            return;
        }
        // 获取亚马逊客户端
        MarketplaceWebServiceOrdersClient client = getClient(kycStore);
        // 获取请求
        ListOrdersRequest request = getListOrdersRequest(kycStore, syncDate);
        // 获取库中已存订单
        List<JsSyncAmazonOrder> jsSyncAmazonOldOrders = getJsSyncAmazonOldOrders(kycStore.getId(), syncDate);
        // 获取库中对应订单地址信息
        List<JsSyncAmazonOrderAddress> jsOldOrderAddresses = getOrderAddress(jsSyncAmazonOldOrders);
        // 获取查询结果
        ListOrdersResponse listOrdersResponse = client.listOrders(request);
        ListOrdersResult listOrdersResult = listOrdersResponse.getListOrdersResult();
        List<Order> orders = listOrdersResult.getOrders();

        // 保存订单
        saveAmazonOrder(kycStore, orders, jsSyncAmazonOldOrders, jsOldOrderAddresses);
        // 如果没传完隔一分钟后继续
        if (listOrdersResult.isSetNextToken()) {
            getOrdersByNextToken(client, kycStore, listOrdersResult.getNextToken(), jsSyncAmazonOldOrders, jsOldOrderAddresses);
        } else {
            log.info("订单拉取完毕。");
        }
    }

    /**
     * 设置用户授权信息
     */
    private ListOrdersRequest getListOrdersRequest(KycStore kycStore, Date syncDate) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        ListOrdersRequest request = new ListOrdersRequest();
        request.setSellerId(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());
        // 设置市场信息
        request.withMarketplaceId(amazonEndpointEnum.getMarketPlaceId());
        // 设置订单搜索条件
        XMLGregorianCalendar createdAfter = MwsUtl.getDTF().newXMLGregorianCalendar();

        ZonedDateTime utcSearch = ZonedDateTime.ofInstant(syncDate.toInstant(), ZoneId.of("UTC"));

        createdAfter.setYear(utcSearch.getYear());
        createdAfter.setMonth(utcSearch.getMonthValue());
        createdAfter.setDay(utcSearch.getDayOfMonth());
        request.setCreatedAfter(createdAfter);
        log.info("准备同步订单数据，同步使用的时间为：{}", utcSearch);
        return request;
    }

    private List<JsSyncAmazonOrder> getJsSyncAmazonOldOrders(String storeId, Date syncDate) {
        Example example = new Example(JsSyncAmazonOrder.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        // 这里加一点时间，防止老数据覆盖不到新数据
        Date selectTime = Date.from(ZonedDateTime.ofInstant(syncDate.toInstant(), ZoneId.systemDefault()).minusMonths(1).toInstant());
        criteria.andGreaterThanOrEqualTo("purchaseDate", selectTime);
        return jsSyncAmazonOrderMapper.selectByExample(example);
    }

    /**
     * 获取与亚马逊订单客户端
     */
    private MarketplaceWebServiceOrdersClient getClient(KycStore kycStore) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        MarketplaceWebServiceOrdersConfig config = new MarketplaceWebServiceOrdersConfig();
        config.setServiceURL(amazonEndpointEnum.getEndpoint());
        String applicationName = "dmjishiyupay";
        String applicationVersion = "1.0";
        return new MarketplaceWebServiceOrdersAsyncClient(amazonEndpointEnum.getAccessKey(), amazonEndpointEnum.getSecrectKey(),
                applicationName, applicationVersion, config, (ExecutorService)null);
    }

    /**
     * 批量更新订单数据
     */
    private void saveAmazonOrder(KycStore kycStore, List<Order> orders, List<JsSyncAmazonOrder> jsSyncAmazonOldOrders, List<JsSyncAmazonOrderAddress> jsOldOrderAddresses) {
        Date syncDate = Date.from(Instant.now());
        List<JsSyncAmazonOrder> insertList = new ArrayList<>();
        List<JsSyncAmazonOrder> updateList = new ArrayList<>();
        List<JsSyncAmazonOrderAddress> insertAddressList = new ArrayList<>();
        List<JsSyncAmazonOrderAddress> updateAddressList = new ArrayList<>();
        for (Order order : orders) {
            boolean isExist = false;
            for (JsSyncAmazonOrder jsSyncAmazonOldOrder : jsSyncAmazonOldOrders) {
                if (order.getAmazonOrderId().equals(jsSyncAmazonOldOrder.getAmazonOrderId())) {
                    isExist = true;
                    if (AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(order.getLastUpdateDate()).getTime() > jsSyncAmazonOldOrder.getLastUpdateDate().getTime()) {
                        // 如果现在不是最新的则更新数据
                        jsSyncAmazonOldOrder.setSyncDate(syncDate);
                        jsSyncAmazonOldOrder.setUpdateDts(syncDate);
                        jsSyncAmazonOldOrder.setUpdateId("sys");
                        jsSyncAmazonOldOrder.setUpdateName("sys");
                        setJsAmazonOrderProperites(jsSyncAmazonOldOrder, order);
                        updateList.add(jsSyncAmazonOldOrder);
                        JsSyncAmazonOrderAddress jsSyncAmazonOrderAddress = getJsSyncAmazonOrderAddress(jsSyncAmazonOldOrder, jsOldOrderAddresses);
                        prepareSaveAddress(kycStore, insertAddressList, updateAddressList, order, jsSyncAmazonOrderAddress);
                        // 关联地址 id
                        jsSyncAmazonOldOrder.setShippingAddressId(jsSyncAmazonOrderAddress.getId());
                    }
                }
            }
            if (!isExist) {
                JsSyncAmazonOrder jsSyncAmazonOrder = new JsSyncAmazonOrder();
                AmaMWSCommonUtil.initBaseData(jsSyncAmazonOrder);
                jsSyncAmazonOrder.setSyncDate(syncDate);
                jsSyncAmazonOrder.setStoreId(kycStore.getId());
                jsSyncAmazonOrder.setKycNaturalId(kycStore.getKycNaturalId());
                setJsAmazonOrderProperites(jsSyncAmazonOrder, order);
                insertList.add(jsSyncAmazonOrder);
                JsSyncAmazonOrderAddress jsSyncAmazonOrderAddress = getJsSyncAmazonOrderAddress(jsSyncAmazonOrder, jsOldOrderAddresses);
                prepareSaveAddress(kycStore, insertAddressList, updateAddressList, order, jsSyncAmazonOrderAddress);
                jsSyncAmazonOrder.setShippingAddressId(jsSyncAmazonOrderAddress.getId());
            }
        }
        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonOrderMapper.insertListWithKey(insertList);
        }
        if (ObjectUtil.isNotEmpty(insertAddressList)) {
            jsSyncAmazonOrderAddressMapper.insertListWithKey(insertAddressList);
        }
        if (ObjectUtil.isNotEmpty(updateList)) {
            DBUtil.updateBatch(updateList, JsSyncAmazonOrderMapper.class, sqlSessionFactory);
        }
        if (ObjectUtil.isNotEmpty(updateAddressList)) {
            DBUtil.updateBatch(updateAddressList, JsSyncAmazonOrderAddressMapper.class, sqlSessionFactory);
        }
    }

    /**
     * 生成需要保存的地址对象，放到相应的列表中去。
     */
    private void prepareSaveAddress(KycStore kycStore, List<JsSyncAmazonOrderAddress> insertAddressList, List<JsSyncAmazonOrderAddress> updateAddressList, Order order, JsSyncAmazonOrderAddress jsSyncAmazonOrderAddress) {
        if (jsSyncAmazonOrderAddress == null){
            return;
        } else if (jsSyncAmazonOrderAddress.getId() != null) {
            jsSyncAmazonOrderAddress.setUpdateId("sys");
            jsSyncAmazonOrderAddress.setUpdateName("sys");
            jsSyncAmazonOrderAddress.setUpdateDts(Date.from(Instant.now()));
            setJsSyncAddressProperties(jsSyncAmazonOrderAddress, order.getShippingAddress());
            updateAddressList.add(jsSyncAmazonOrderAddress);
        } else {
            AmaMWSCommonUtil.initBaseData(jsSyncAmazonOrderAddress);
            jsSyncAmazonOrderAddress.setKycNaturalId(kycStore.getKycNaturalId());
            setJsSyncAddressProperties(jsSyncAmazonOrderAddress, order.getShippingAddress());
            insertAddressList.add(jsSyncAmazonOrderAddress);
        }
    }

    private JsSyncAmazonOrderAddress getJsSyncAmazonOrderAddress(JsSyncAmazonOrder jsSyncAmazonOldOrder,
                                                                 List<JsSyncAmazonOrderAddress> jsOldOrderAddresses) {
        List<JsSyncAmazonOrderAddress> jsSyncAmazonOrderAddresses = jsOldOrderAddresses.stream()
                .filter(e -> e.getId().equals(jsSyncAmazonOldOrder.getShippingAddressId()))
                .collect(Collectors.toList());
        if (jsSyncAmazonOrderAddresses.size() > 1) {
            log.error("订单地址数据异常，符合条件地址不为一。");
            return null;
        } else if (jsSyncAmazonOrderAddresses.size() == 1) {
            return jsSyncAmazonOrderAddresses.get(0);
        } else if (jsSyncAmazonOrderAddresses.size() == 0) {
            return new JsSyncAmazonOrderAddress();
        } else {
            log.error("订单地址数据异常");
            return null;
        }
    }

    private void setJsSyncAddressProperties(JsSyncAmazonOrderAddress jsSyncAmazonOrderAddress, Address shippingAddress) {
        if (ObjectUtil.isEmpty(jsSyncAmazonOrderAddress) || ObjectUtil.isEmpty(shippingAddress)) {
            // 有取消的订单，就会有空值
            return;
        }
        jsSyncAmazonOrderAddress.setName(shippingAddress.getName());
        jsSyncAmazonOrderAddress.setAddressLine1(shippingAddress.getAddressLine1());
        jsSyncAmazonOrderAddress.setAddressLine2(shippingAddress.getAddressLine2());
        jsSyncAmazonOrderAddress.setAddressLine3(shippingAddress.getAddressLine3());
        jsSyncAmazonOrderAddress.setCity(shippingAddress.getCity());
        jsSyncAmazonOrderAddress.setCounty(shippingAddress.getCounty());
        jsSyncAmazonOrderAddress.setDistrict(shippingAddress.getDistrict());
        jsSyncAmazonOrderAddress.setStateOrRegion(shippingAddress.getStateOrRegion());
        jsSyncAmazonOrderAddress.setMunicipality(shippingAddress.getMunicipality());
        jsSyncAmazonOrderAddress.setPostalCode(shippingAddress.getPostalCode());
        jsSyncAmazonOrderAddress.setCountryCode(shippingAddress.getCountryCode());
        jsSyncAmazonOrderAddress.setPhone(shippingAddress.getPhone());
        jsSyncAmazonOrderAddress.setAddressType(shippingAddress.getAddressType());
    }

    /**
     * 获取已有订单地址 id 列表
     */
    private List<JsSyncAmazonOrderAddress> getOrderAddress(List<JsSyncAmazonOrder> jsSyncAmazonOldOrders) {
        List<String> addressIds = jsSyncAmazonOldOrders.stream()
                .map(JsSyncAmazonOrder::getShippingAddressId)
                .collect(Collectors.toList());
        if (addressIds.size() > 0) {
            Example example = new Example(JsSyncAmazonOrderAddress.class);
            Example.Criteria criteria = example.createCriteria();
            criteria.andIn("id", addressIds);
            return jsSyncAmazonOrderAddressMapper.selectByExample(example);
        } else {
            return new ArrayList<>();
        }
    }

    /**
     * 获取其余订单
     */
    private void getOrdersByNextToken(MarketplaceWebServiceOrdersClient client, KycStore kycStore,
                                      String nextToken, List<JsSyncAmazonOrder> jsSyncAmazonOldOrders, List<JsSyncAmazonOrderAddress> jsOldOrderAddresses) {
        // 一分钟请求一次
        ListOrdersByNextTokenResult listOrdersByNextTokenResult;
        try {
            log.info("订单列表有 nextToken 暂停一分钟");
            Thread.sleep(60000);

            ListOrdersByNextTokenRequest request = new ListOrdersByNextTokenRequest();
            request.setSellerId(kycStore.getSellId());
            request.setMWSAuthToken(kycStore.getMwsAuthToken());
            request.setNextToken(nextToken);
            ListOrdersByNextTokenResponse listOrdersByNextTokenResponse = client.listOrdersByNextToken(request);
            listOrdersByNextTokenResult = listOrdersByNextTokenResponse.getListOrdersByNextTokenResult();
            List<Order> orders = listOrdersByNextTokenResult.getOrders();
            // 批量保存订单
            saveAmazonOrder(kycStore, orders, jsSyncAmazonOldOrders, jsOldOrderAddresses);
        } catch (Exception e) {
            log.error("拉取亚e马逊订单数据时出错：{}, 当前正拉取的 nextToken：{}", e.getMessage(), nextToken);
            log.error("拉取亚马逊订单数据时出错", e);
            return;
        }
        // 如果没有传完，递归。
        if (listOrdersByNextTokenResult.isSetNextToken()) {
            getOrdersByNextToken(client, kycStore, listOrdersByNextTokenResult.getNextToken(), jsSyncAmazonOldOrders, jsOldOrderAddresses);
        } else {
            log.info("订单拉取完毕。");
        }
    }

    private void setJsAmazonOrderProperites(JsSyncAmazonOrder jsSyncAmazonOrder, Order order) {
        jsSyncAmazonOrder.setAmazonOrderId(order.getAmazonOrderId());
        jsSyncAmazonOrder.setSellerOrderId(order.getSellerOrderId());
        jsSyncAmazonOrder.setPurchaseDate(AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(order.getPurchaseDate()));
        jsSyncAmazonOrder.setLastUpdateDate(AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(order.getLastUpdateDate()));
        jsSyncAmazonOrder.setOrderStatus(order.getOrderStatus());
        if (ObjectUtil.isNotEmpty(order.getOrderTotal())) {
            jsSyncAmazonOrder.setCurrencyCode(order.getOrderTotal().getCurrencyCode());
            jsSyncAmazonOrder.setOrderTotal(AmaMWSCommonUtil.convertBigDecimal(order.getOrderTotal().getAmount()));
        }
        jsSyncAmazonOrder.setNumberOfItemsShipped(AmaMWSCommonUtil.convertBigDecimal(order.getNumberOfItemsShipped().toString()));
        jsSyncAmazonOrder.setNumberOfItemsUnshipped(AmaMWSCommonUtil.convertBigDecimal(order.getNumberOfItemsUnshipped().toString()));
        jsSyncAmazonOrder.setMarketplaceId(order.getMarketplaceId());
        jsSyncAmazonOrder.setBuyerEmail(order.getBuyerEmail());
        jsSyncAmazonOrder.setBuyerName(order.getBuyerName());
        jsSyncAmazonOrder.setOrderType(order.getOrderType());
        jsSyncAmazonOrder.setReplacedOrderId(order.getReplacedOrderId());
        jsSyncAmazonOrder.setIsReplacementOrder(order.getIsReplacementOrder());
        jsSyncAmazonOrder.setPromiseResponseDueDate(AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(order.getPromiseResponseDueDate()));
        jsSyncAmazonOrder.setIsEstimatedShipDateSet(order.getIsEstimatedShipDateSet());
    }

}
