package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceAsyncClient;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceClient;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceConfig;
import com.amazon.mws.finances._2015_05_01.model.*;
import com.js.common.constant.Constant;
import com.js.common.enums.AmazonEndpointEnum;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.api.sync.service.AmazonFinanceEventService;
import com.js.sync.utils.AmaMWSCommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.time.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonFinanceEventServiceImpl implements AmazonFinanceEventService {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    JsSyncAmazonShipmentEventItemMapper jsSyncAmazonShipmentEventItemMapper;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;


    @Override
    public CompletableFuture<Void> startSyncFinanceEventWhole(String storeId) {
        Date syncDate = Date.from(LocalDateTime.of(1, 1, 1, 0, 0, 0).toInstant(ZoneOffset.UTC));
        try {
            startSyncFinanceEventBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncFinanceEventWhole 出错", e);
        }
        log.info("付款事件拉取完毕");
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> startSyncFinanceEventIncrement(String storeId) {
        Date previousSyncDate = AmaMWSCommonUtil.getSyncDate(storeId, JsSyncAmazonFinancialEventGroup.class, jsSyncAmazonFinancialEventGroupMapper);
        if (previousSyncDate.getTime() < 0) {
            log.info("亚马逊店铺事件明细尚未获取到上次更新时间，改为使用当前时间。店铺 id: {}", storeId);
            previousSyncDate = new Date();
        }
        log.info("准备增量抓取，抓取最近一个月的事件明细。使用时间为：{}", previousSyncDate);
        Date syncDate = Date.from(ZonedDateTime.ofInstant(previousSyncDate.toInstant(), ZoneId.of("UTC")).minusMonths(1).toInstant());
        try {
            startSyncFinanceEventBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncFinanceEventIncrement 出错", e);
        }
        log.info("亚马逊店铺事件明细增量抓取完毕。店铺 id: {}", storeId);
        return CompletableFuture.completedFuture(null);
    }

    public void startSyncFinanceEventBySyncDate(String storeId, Date syncDate) {
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        MWSFinancesServiceClient client = getClient(kycStore);
        // 因为 eventGroupId 是经过过滤的。所以这里不需要再次过滤了。
        List<JsSyncAmazonFinancialEventGroup> financeEventGroups = getFinanceEventGroupsByStoreAndSyncDate(storeId, syncDate);

        for (JsSyncAmazonFinancialEventGroup financeEventGroup : financeEventGroups) {
            log.info("开始拉取付款事件，事件组 id: {}", financeEventGroup.getFinancialEventGroupId());
            try {
                log.info("暂停5秒钟");
                Thread.sleep(5000);
                ListFinancialEventsRequest request = getRequest(kycStore, financeEventGroup.getFinancialEventGroupId());
                ListFinancialEventsResponse listFinancialEventsResponse = client.listFinancialEvents(request);
                ListFinancialEventsResult listFinancialEventsResult = listFinancialEventsResponse.getListFinancialEventsResult();
                FinancialEvents financialEvents = listFinancialEventsResult.getFinancialEvents();
                transferEventToJsSyncShipmentEvent(kycStore, financeEventGroup.getFinancialEventGroupId(), financialEvents);
                if (listFinancialEventsResult.isSetNextToken()) {
                    getFinancialEventByNextToken(client, kycStore, listFinancialEventsResult.getNextToken(),
                            financeEventGroup.getFinancialEventGroupId());
                }
            } catch (Exception e) {
                log.error("拉取付款事件时出错。事件组 id：{}", financeEventGroup.getFinancialEventGroupId());
                log.error("拉取付款事件时出错堆栈", e);
            }
        }

    }

    /**
     * 将不同的事件转为目标对象
     *
     * @param kycStore 店铺 id
     * @param financialEventGroupId 事件组 id
     * @param financialEvents 源对象列表
     *
     */
    private void transferEventToJsSyncShipmentEvent(KycStore kycStore, String financialEventGroupId,
                                                    FinancialEvents financialEvents) {
        List<JsSyncAmazonShipmentEvent> insertList = new ArrayList<>();
        List<ShipmentEvent> shipmentEventList = financialEvents.getShipmentEventList();
        saveJsSyncShipmentEvent(kycStore, insertList, AmazonEventTypeEnum.SHIPMENT.ordinal(),
                financialEventGroupId, shipmentEventList);
        List<ShipmentEvent> refundEventList = financialEvents.getRefundEventList();
        saveJsSyncShipmentEvent(kycStore, insertList, AmazonEventTypeEnum.REFOUN.ordinal(),
                financialEventGroupId, refundEventList);
        List<ShipmentEvent> guaranteeClaimEventList = financialEvents.getGuaranteeClaimEventList();
        saveJsSyncShipmentEvent(kycStore, insertList, AmazonEventTypeEnum.GUARANTEE_CLAIM.ordinal(),
                financialEventGroupId, guaranteeClaimEventList);
        List<ShipmentEvent> chargebackEventList = financialEvents.getChargebackEventList();
        saveJsSyncShipmentEvent(kycStore, insertList, AmazonEventTypeEnum.CHARGEBACK.ordinal(),
                financialEventGroupId, chargebackEventList);
        List<JsSyncAmazonShipmentEvent> jsSyncShipmentOldEvents = getJsSyncShipmentOldEvents(financialEventGroupId);

        List<JsSyncAmazonShipmentEvent> insertList2 = new ArrayList<>();
        for (JsSyncAmazonShipmentEvent jsSyncShipmentEvent : insertList) {
            boolean isExist = false;
            for (JsSyncAmazonShipmentEvent jsSyncShipmentOldEvent : jsSyncShipmentOldEvents) {
                if (jsSyncShipmentOldEvent.getFinancialEventGroupId().equals(jsSyncShipmentEvent.getFinancialEventGroupId())
                        && jsSyncShipmentOldEvent.getAmazonOrderId().equals(jsSyncShipmentEvent.getAmazonOrderId())
                        && jsSyncShipmentOldEvent.getEventType().equals(jsSyncShipmentEvent.getEventType())) {
                    isExist = true;
                    // 这些事件应该都不会变，如果数据库中已经存在了那么直接跳过
                }
            }
            if (!isExist) {
                insertList2.add(jsSyncShipmentEvent);
            }
        }
        if (ObjectUtil.isNotEmpty(insertList2)) {
            jsSyncAmazonShipmentEventMapper.insertListWithKey(insertList2);
        } else {
            log.info("没有需要保存的付款事件信息。");
        }

    }

    private void getFinancialEventByNextToken(MWSFinancesServiceClient client, KycStore kycStore, String nextToken,
                                              String financialEventGroupId) {
        ListFinancialEventsByNextTokenRequest request = new ListFinancialEventsByNextTokenRequest();
        request.setSellerId(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());
        request.setNextToken(nextToken);
        try {
            log.info("有 nextToken 暂停5秒钟");
            Thread.sleep(5000);
            ListFinancialEventsByNextTokenResponse response = client.listFinancialEventsByNextToken(request);
            ListFinancialEventsByNextTokenResult listFinancialEventsByNextTokenResult = response.getListFinancialEventsByNextTokenResult();
            FinancialEvents financialEvents = listFinancialEventsByNextTokenResult.getFinancialEvents();
            transferEventToJsSyncShipmentEvent(kycStore, financialEventGroupId, financialEvents);
            if (listFinancialEventsByNextTokenResult.isSetNextToken()) {
                getFinancialEventByNextToken(client, kycStore, listFinancialEventsByNextTokenResult.getNextToken(),
                        financialEventGroupId);
            }
        } catch (Exception e) {
            log.error("拉取付款事件时出错 nextToken: {}", nextToken);
            log.error("拉取付款事件时出错堆栈", e);
        }

    }

    /**
     * 取出库中老数据
     */
    private List<JsSyncAmazonShipmentEvent> getJsSyncShipmentOldEvents(String financeEventGroupId) {
        Example example = new Example(JsSyncAmazonShipmentEvent.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("financialEventGroupId", financeEventGroupId);
        return jsSyncAmazonShipmentEventMapper.selectByExample(example);
    }

    /**
     * 因为级联的对象太多了， 不再进行批量保存。转好一个存一个。
     *
     * @param kycStore
     * @param inserList
     * @param eventType
     * @param financialEventGroupId
     * @param shipmentEventList
     */
    private void saveJsSyncShipmentEvent(KycStore kycStore, List<JsSyncAmazonShipmentEvent> inserList, int eventType,
                                         String financialEventGroupId, List<ShipmentEvent> shipmentEventList) {
        for (ShipmentEvent shipmentEvent : shipmentEventList) {
            JsSyncAmazonShipmentEvent jsSyncAmazonShipmentEvent = new JsSyncAmazonShipmentEvent();
            AmaMWSCommonUtil.initBaseData(jsSyncAmazonShipmentEvent);
            jsSyncAmazonShipmentEvent.setEventType(eventType);
            jsSyncAmazonShipmentEvent.setKycNaturalId(kycStore.getKycNaturalId());
            jsSyncAmazonShipmentEvent.setStoreId(kycStore.getId());
            jsSyncAmazonShipmentEvent.setFinancialEventGroupId(financialEventGroupId);
            jsSyncAmazonShipmentEvent.setAmazonOrderId(shipmentEvent.getAmazonOrderId());
            jsSyncAmazonShipmentEvent.setSellerOrderId(shipmentEvent.getSellerOrderId());
            jsSyncAmazonShipmentEvent.setMarketplaceName(shipmentEvent.getMarketplaceName());
            inserList.add(jsSyncAmazonShipmentEvent);
            saveShipmentEventItem(kycStore, shipmentEvent);
            log.info("事件保存成功");
        }
    }

    private void saveShipmentEventItem(KycStore kycStore, ShipmentEvent shipmentEvent) {
        List<JsSyncAmazonShipmentEventItem> jsSyncAmazonShipmentEventItems = getAmazonShipmentEventItems(kycStore, shipmentEvent);
        List<JsSyncAmazonShipmentEventItem> insertList = new ArrayList<>();
        for (ShipmentItem shipmentItem : shipmentEvent.getShipmentItemList()) {
            boolean isExist = false;
            for (JsSyncAmazonShipmentEventItem jsSyncAmazonShipmentEventItem : jsSyncAmazonShipmentEventItems) {
                // 实际测试有可以出现空的情况。如果库里面已经存了一个空的了。则不再继续存新的了
                if (shipmentItem.getOrderItemId() == null && jsSyncAmazonShipmentEventItem.getOrderItemId() == null) {
                    isExist = true;
                    break;
                }
                if (shipmentItem.getOrderItemId().equals(jsSyncAmazonShipmentEventItem.getOrderItemId())) {
                    isExist = true;
                    break;
                }
            }
            if (!isExist) {
                JsSyncAmazonShipmentEventItem jsSyncAmazonShipmentEventItem = new JsSyncAmazonShipmentEventItem();
                AmaMWSCommonUtil.initBaseData(jsSyncAmazonShipmentEventItem);
                jsSyncAmazonShipmentEventItem.setKycNaturalId(kycStore.getKycNaturalId());
                jsSyncAmazonShipmentEventItem.setStoreId(kycStore.getId());
                jsSyncAmazonShipmentEventItem.setSellersku(shipmentItem.getSellerSKU());
                jsSyncAmazonShipmentEventItem.setOrderId(shipmentEvent.getAmazonOrderId());
                jsSyncAmazonShipmentEventItem.setOrderItemId(shipmentItem.getOrderItemId());
                jsSyncAmazonShipmentEventItem.setOrderAdjustmentItemId(shipmentItem.getOrderAdjustmentItemId());
                jsSyncAmazonShipmentEventItem.setQuantityShipped(new BigDecimal(shipmentItem.getQuantityShipped()));
                insertList.add(jsSyncAmazonShipmentEventItem);
                saveShipmentEventCharge(kycStore, shipmentEvent, shipmentItem);
            }
        }
        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonShipmentEventItemMapper.insertListWithKey(insertList);
            log.info("事件明细表存成功");
        } else {
            //log.info("没有需要保存的事件明细。");
        }
    }

    private List<JsSyncAmazonShipmentEventItem> getAmazonShipmentEventItems(KycStore kycStore, ShipmentEvent shipmentEvent) {
        Example example = new Example(JsSyncAmazonShipmentEventItem.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycStore.getKycNaturalId());
        criteria.andEqualTo("storeId", kycStore.getId());
        criteria.andEqualTo("orderId", shipmentEvent.getAmazonOrderId());
        return jsSyncAmazonShipmentEventItemMapper.selectByExample(example);
    }

    private void saveShipmentEventCharge(KycStore kycStore, ShipmentEvent shipmentEvent, ShipmentItem shipmentItem) {
        List<JsSyncAmazonShipmentEventCharge> insertList = new ArrayList<>();
        List<JsSyncAmazonShipmentEventCharge> amazonShipmentEventCharge = getAmazonShipmentEventCharge(kycStore, shipmentEvent);
        for (ChargeComponent chargeComponent : shipmentItem.getItemChargeList()) {
            boolean isExist = false;
            for (JsSyncAmazonShipmentEventCharge syncAmazonShipmentEventCharge : amazonShipmentEventCharge) {
                if (syncAmazonShipmentEventCharge.getOrderItemId().equals(shipmentItem.getOrderItemId())
                        && syncAmazonShipmentEventCharge.getChargeType().equals(chargeComponent.getChargeType())) {
                    isExist = true;
                    break;
                }
            }
            if (!isExist) {
                JsSyncAmazonShipmentEventCharge jsSyncAmazonShipmentEventCharge = new JsSyncAmazonShipmentEventCharge();
                AmaMWSCommonUtil.initBaseData(jsSyncAmazonShipmentEventCharge);
                jsSyncAmazonShipmentEventCharge.setKycNaturalId(kycStore.getKycNaturalId());
                jsSyncAmazonShipmentEventCharge.setStoreId(kycStore.getId());
                jsSyncAmazonShipmentEventCharge.setSellersku(shipmentItem.getSellerSKU());
                jsSyncAmazonShipmentEventCharge.setOrderId(shipmentEvent.getAmazonOrderId());
                jsSyncAmazonShipmentEventCharge.setOrderItemId(shipmentItem.getOrderItemId());
                if (ObjectUtil.isNotEmpty(chargeComponent.getChargeAmount())) {
                    jsSyncAmazonShipmentEventCharge.setChargeAmountCode(chargeComponent.getChargeAmount().getCurrencyCode());
                    jsSyncAmazonShipmentEventCharge.setChargeAmount(chargeComponent.getChargeAmount().getCurrencyAmount());
                }
                jsSyncAmazonShipmentEventCharge.setChargeType(chargeComponent.getChargeType());
                insertList.add(jsSyncAmazonShipmentEventCharge);
            }
        }
        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonShipmentEventChargeMapper.insertListWithKey(insertList);
            log.info("商品费用保存成功");
        } else {
            log.info("没有需要保存的商品费用。");
        }
    }

    private List<JsSyncAmazonShipmentEventCharge> getAmazonShipmentEventCharge(KycStore kycStore, ShipmentEvent shipmentEvent) {
        Example example = new Example(JsSyncAmazonShipmentEventCharge.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycStore.getKycNaturalId());
        criteria.andEqualTo("storeId", kycStore.getId());
        criteria.andEqualTo("orderId", shipmentEvent.getAmazonOrderId());
        return jsSyncAmazonShipmentEventChargeMapper.selectByExample(example);
    }

    /**
     * 生成请求参数
     * @return
     */
    private ListFinancialEventsRequest getRequest(KycStore kycStore, String financialEventGroupId) {
        ListFinancialEventsRequest request = new ListFinancialEventsRequest();
        request.setSellerId(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());
//        Integer maxResultsPerPage = 1;
//        request.setMaxResultsPerPage(maxResultsPerPage);
//        String amazonOrderId = "example";
//        request.setAmazonOrderId(amazonOrderId);
        request.setFinancialEventGroupId(financialEventGroupId);
//        XMLGregorianCalendar postedAfter = MwsUtl.getDTF().newXMLGregorianCalendar();
//        request.setPostedAfter(postedAfter);
//        XMLGregorianCalendar postedBefore = MwsUtl.getDTF().newXMLGregorianCalendar();
//        request.setPostedBefore(postedBefore);
        return request;
    }

    /**
     * 获取同步时间内的付款账单事件组 id 列表
     */
    private List<JsSyncAmazonFinancialEventGroup> getFinanceEventGroupsByStoreAndSyncDate(String storeId, Date syncDate) {
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andGreaterThanOrEqualTo("syncDate", syncDate);
        List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventGroups =
                jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);
        return jsSyncAmazonFinancialEventGroups;
    }

    private MWSFinancesServiceClient getClient(KycStore kycStore) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        MWSFinancesServiceConfig config = new MWSFinancesServiceConfig();
        config.setServiceURL(amazonEndpointEnum.getEndpoint());
        MWSFinancesServiceClient client = new MWSFinancesServiceAsyncClient(amazonEndpointEnum.getAccessKey(),
                amazonEndpointEnum.getSecrectKey(), "dmjishiyupay", "1.0", config, (ExecutorService)null);
        return client;
    }

}
